﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AudioEchoback.cpp,PageSampleAudioAudioEchoback}
 *
 * @brief
 * オーディオ入出力機能の同時利用のサンプルプログラム
 */

/**
 * @page PageSampleAudioAudioEchoback オーディオ入出力機能の同時利用
 * @tableofcontents
 *
 * @brief
 * オーディオ入出力機能の同時利用のサンプルプログラムの解説です。
 *
 * @section PageSampleAudioAudioEchoback_SectionBrief 概要
 * オーディオ入力とオーディオ出力を同時に利用して、エコーバックを実現するサンプルです。
 *
 * @section PageSampleAudioAudioEchoback_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/AudioEchoback
 * Samples/Sources/Applications/AudioEchoback @endlink 以下にあります。
 *
 * @section PageSampleAudioAudioEchoback_SectionNecessaryEnvironment 必要な環境
 * オーディオ入力（マイクなど）、オーディオ出力（スピーカなど）が利用可能である必要があります。
 *
 * @section PageSampleAudioAudioEchoback_SectionHowToOperate 操作方法
 * サンプルプログラムを実行すると、エコーバックします。
 * キーボードや DebugPad による入力を用いてプログラムを操作することが出来ます。
 *
 * <p>
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> [Start/Space]   </td><td> プログラムの終了 </td></tr>
 * </table>
 * </p>
 *
 * @section PageSampleAudioAudioEchoback_SectionPrecaution 注意事項
 * デフォルトのオーディオ入力およびオーディオ出力の、以下の値が一致している必要があります。
 * - チャンネル数
 * - サンプルレート
 * - サンプルフォーマット
 *
 * @section PageSampleAudioAudioEchoback_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAudioAudioEchoback_SectionDetail 解説
 * このサンプルプログラムは、オーディオ入力により録音した波形を、そのままオーディオ出力に出力するものです。
 *
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - デフォルトのオーディオ入力およびオーディオ出力をオープン
 * - オーディオ入力、オーディオ出力の情報を取得、確認
 * - 録音および再生用バッファを準備、登録
 * - 録音、再生を開始
 * - エコーバックを実行
 * - 録音、再生を終了
 * - オーディオ入力、オーディオ出力をクローズ
 * - 録音および再生用バッファを破棄
 *
 * 本サンプルプログラムでは、簡単のため、オーディオ入力とオーディオ出力の以下の情報が一致していることを仮定しています。
 *
 * - チャンネル数
 * - サンプルレート
 * - サンプルフォーマット
 *
 * これらが異なる場合は、適切なデータ変換（ダウンミックス、サンプルレート変換、など）が必要となります。
 *
 * 録音、再生のためのバッファは 50 ミリ秒の長さのものを 3 つ用意します。
 * バッファの長さや数については利用者側で自由に決めることができますが、短かすぎたり、
 * 数が少なすぎると、音が途切れるなどの問題が発生することがあります。
 * 用意したバッファは nn::audio::AppendAudioInBuffer() および nn::audio::AppendAudioOutBuffer() により登録します。
 * 登録は nn::audio::StartAudioIn() および nn::audio::StartAudioOut() の呼び出し前でも呼び出し後でも可能です。
 * 実際の録音および再生は nn::audio::StartAudioIn() および nn::audio::StartAudioOut() が呼ばれるまでは行われません。
 *
 * 録音、再生開始後は nn::audio::GetReleasedAudioInBuffer(), nn::audio::GetReleasedAudioOutBuffer() により
 * 録音および再生が完了したバッファ情報を取得し、録音データを再生バッファにコピーし、それぞれを再度登録します。
 * ダウンミックスやサンプルレート変換などが必要な場合は、このタイミングで行います。
 * この処理を繰り返すことでエコーバックを実現します。
 *
 * プログラムの終了が要求されると、上記の繰り返し処理を抜けて、録音、再生を停止し、
 * オーディオ入力およびオーディオ出力をクローズし、メモリの破棄を行います。
 * nn::audio::StopAudioIn(), nn::audio::StopAudioOut() を呼ぶと、録音、再生は停止し、
 * 登録済み（かつ未録音、未再生）バッファへのアクセスはなくなりますが、再度 nn::audio::StartAudioIn(),
 * nn::audio::StartAudioOut() を行うと、アクセスが発生することには注意が必要です。
 * nn::audio::CloseAudioIn(), nn::audio::CloseAudioOut() 後はアクセスが発生することはありません。
 */

#include <limits>
#include <string>  // std::memset, std::memcpy
#include <mutex>
#include <algorithm>

#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nns/nns_Log.h>
#include <nn/audio.h>
#include <nn/mem.h>
#include <nn/os.h>
#include <nn/nn_TimeSpan.h>

#include <nns/audio/audio_HidUtilities.h>
#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Npad.h>

#include <nn/settings/settings_DebugPad.h>

#include "AudioEchobackUtil.h"

namespace {

const int MaxAudioIns = 2;

char g_HeapBuffer[128 * 1024 * 16];
const size_t ThreadStackSize = 8192 * 2;
NN_ALIGNAS(4096) char g_AudioOutThreadStack[ThreadStackSize];
nn::os::ThreadType    g_AudioOutThread;
NN_ALIGNAS(4096) char g_AudioInThreadStack[ThreadStackSize];
nn::os::ThreadType    g_AudioInThread;


const char Title[] = "AudioEchoback";


void InitializeHidDevices()
{
    nn::hid::InitializeDebugPad();
    nn::hid::InitializeNpad();
    const nn::hid::NpadIdType npadIds[2] = { nn::hid::NpadId::No1, nn::hid::NpadId::Handheld };
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    //キーボードのキーを DebugPad のボタンに割り当てます。
    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonA     = nn::hid::KeyboardKey::A::Index;
    map.buttonB     = nn::hid::KeyboardKey::B::Index;
    map.buttonX     = nn::hid::KeyboardKey::X::Index;
    map.buttonY     = nn::hid::KeyboardKey::Y::Index;
    map.buttonL     = nn::hid::KeyboardKey::L::Index;
    map.buttonR     = nn::hid::KeyboardKey::R::Index;
    map.buttonZL    = nn::hid::KeyboardKey::U::Index;
    map.buttonZR    = nn::hid::KeyboardKey::V::Index;
    map.buttonLeft  = nn::hid::KeyboardKey::LeftArrow::Index;
    map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
    map.buttonUp    = nn::hid::KeyboardKey::UpArrow::Index;
    map.buttonDown  = nn::hid::KeyboardKey::DownArrow::Index;
    map.buttonStart = nn::hid::KeyboardKey::Space::Index;
    nn::settings::SetDebugPadKeyboardMap(map);
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("Command Line Argument: Set buffer length in milliseconds.\n");
    NNS_LOG("    Minimum: 5 ms, Max: 50 ms\n");
    NNS_LOG("[Start/Space]   Shut down sample program\n");
    NNS_LOG("-------------------------------------------------------\n");
}

//
// サンプルが動作するための前提条件をチェックします
//
// このサンプルでは簡単のため、オーディオ入力およびオーディオ出力の以下の情報が一致していることを前提とします。
// - チャンネル数
// - サンプルレート
// - サンプルフォーマット
//
bool CheckAudioInputOutputProperties(nn::audio::AudioIn* pAudioIn, nn::audio::AudioOut* pAudioOut)
{
    if(nn::audio::GetAudioInChannelCount(pAudioIn) != nn::audio::GetAudioOutChannelCount(pAudioOut))
    {
        NNS_LOG("AudioIn and AudioOut are supposed to have same channel count in this sample.\n");
        return false;
    }
    if(nn::audio::GetAudioInSampleRate(pAudioIn) != nn::audio::GetAudioOutSampleRate(pAudioOut))
    {
        NNS_LOG("AudioIn and AudioOut are supposed to have same sample rate in this sample.\n");
        return false;
    }
    if(nn::audio::GetAudioInSampleFormat(pAudioIn) != nn::audio::GetAudioOutSampleFormat(pAudioOut))
    {
        NNS_LOG("AudioIn and AudioOut are supposed to have same sample format in this sample.\n");
        return false;
    }
    return true;
}

struct MicData
{
    MicData()
    : mutex(false)
    , frameSize(0)
    , readIndex(0)
    , writeIndex(0)
    , readyFrames(0)
    , maxFrames(0)
    {
    }

    void SetFrameSize(size_t _frameSize)
    {
        frameSize = _frameSize;
        maxFrames = static_cast<int>(sizeof(buffer) / frameSize);
    }

    void PutFrame(void* data, size_t size)
    {
        NN_ASSERT(size == frameSize);
        std::lock_guard<nn::os::Mutex> lock(mutex);

        memcpy(&buffer[writeIndex * frameSize], data, frameSize);
        writeIndex = (writeIndex + 1) % maxFrames;
        ++readyFrames;
        readyFrames = std::min(readyFrames, maxFrames);
        cond.Signal();
    }

    void GetFrameAdd(void* data, size_t size, int32_t volume)
    {
        NN_ASSERT(size == frameSize);
        std::lock_guard<nn::os::Mutex> lock(mutex);

        while(readyFrames <= 0 && isOpen)
        {
            cond.Wait(mutex);
        }
        if(!isOpen)
        {
            return;
        }

        int16_t* in = reinterpret_cast<int16_t*>(&buffer[readIndex * frameSize]);
        int16_t* out = reinterpret_cast<int16_t*>(data);
        const auto Q = 15;
        for(size_t i = 0; i < size / 2; ++i)
        {
            out[i] += static_cast<int16_t>((in[i] * volume) >> Q);
        }
        memset(&buffer[readIndex * frameSize], 0, frameSize);
        readIndex = (readIndex + 1) % maxFrames;
        --readyFrames;
        NN_ASSERT(readyFrames >= 0);
    }

    void Open()
    {
        isOpen = true;
    }

    void Close()
    {
        isOpen = false;
        cond.Signal();
    }

    bool IsOpen()
    {
        return isOpen.load();
    }

    nn::os::ConditionVariable cond;
    nn::os::Mutex mutex;
    size_t frameSize;
    int readIndex;
    int writeIndex;
    int readyFrames;
    int maxFrames;
    std::atomic<bool> isOpen;
    char buffer [4096 * 8];
};
MicData g_MicData[MaxAudioIns];

struct AudioOutArgs
{
    nn::os::SystemEvent* event;
    nn::audio::AudioOut* audioOut;
    std::atomic<bool> isRunning;
};
AudioOutArgs g_OutArgs;

void AudioOutThread(void* arg)
{
    AudioOutArgs* args = reinterpret_cast<AudioOutArgs*>(arg);
    while(args->isRunning)
    {
        // 録音が完了したバッファを取得します。
        args->event->Wait();

        if(!args->isRunning)
        {
            break;
        }

        nn::audio::AudioOutBuffer* pAudioOutBuffer = nullptr;

        pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(args->audioOut);

        int bufferCount = 0;
        while(pAudioOutBuffer && args->isRunning)
        {
            ++bufferCount;
            // データをコピーし、再度登録します。
            void* pOutBuffer = nn::audio::GetAudioOutBufferDataPointer(pAudioOutBuffer);
            size_t outSize = nn::audio::GetAudioOutBufferDataSize(pAudioOutBuffer);

            memset(pOutBuffer, 0, outSize);
            for(int i = 0; i < MaxAudioIns; ++i)
            {
                if (g_MicData[i].IsOpen())
                {
                    g_MicData[i].GetFrameAdd(pOutBuffer, outSize,  std::numeric_limits<int16_t>::max() / MaxAudioIns);
                }
            }

            nn::audio::AppendAudioOutBuffer(args->audioOut, pAudioOutBuffer);
            pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(args->audioOut);
        }
    }
}

struct AudioInArgs
{
    nn::mem::StandardAllocator* allocator;
    nn::os::Event* shutdownEvent;
    std::atomic<bool> isRunning;
    size_t dataSize;
};
AudioInArgs g_InArgs;

struct AudioInConnection
{
    static const int BufferCount = 4;
    nn::audio::AudioIn audioIn;
    nn::os::SystemEvent bufferEvent;
    nn::os::MultiWaitHolderType bufferHolder;
    char name[nn::audio::AudioIn::NameLength];
    nn::audio::AudioInBuffer buffers[BufferCount];
    void*                    rawBuffers[BufferCount];
    bool isConnected;
};

void InitializeConnections(AudioInConnection* connections, AudioInArgs* args)
{
    size_t bufferSize = nn::util::align_up(args->dataSize, nn::audio::AudioOutBuffer::SizeGranularity);
    for (int i = 0; i < MaxAudioIns; ++i)
    {
        auto& connection = connections[i];
        connection.isConnected = false;
        memset(connection.name, 0, sizeof(connection.name));
        for (int j = 0; j < AudioInConnection::BufferCount; ++j)
        {
            connection.rawBuffers[j] = args->allocator->Allocate(bufferSize, nn::audio::AudioOutBuffer::AddressAlignment);
            NN_ASSERT(connection.rawBuffers[j]);
            nn::audio::SetAudioInBufferInfo(&connection.buffers[j], connection.rawBuffers[j], bufferSize, args->dataSize);
        }
    }
}

void FinalizeConnections(AudioInConnection* connections, AudioInArgs* args)
{
    for (int i = 0; i < MaxAudioIns; ++i)
    {
        auto& connection = connections[i];
        connection.isConnected = false;
        for (int j = 0; j < AudioInConnection::BufferCount; ++j)
        {
            args->allocator->Free(connection.rawBuffers[j]);
        }
    }
}

bool OpenConnection(AudioInConnection& connection, const nn::audio::AudioInInfo& info)
{
    nn::audio::AudioInParameter param;
    nn::audio::InitializeAudioInParameter(&param);

    if(!nn::audio::OpenAudioIn(&connection.audioIn, &connection.bufferEvent, info.name, param).IsSuccess())
    {
        return false;
    }


    for (int i = 0; i < AudioInConnection::BufferCount; ++i)
    {
        if (!nn::audio::AppendAudioInBuffer(&connection.audioIn, &connection.buffers[i]))
        {
            return false;
        }
    }

    if (!nn::audio::StartAudioIn(&connection.audioIn).IsSuccess())
    {
        nn::audio::CloseAudioIn(&connection.audioIn);
        nn::os::DestroySystemEvent(connection.bufferEvent.GetBase());
        return false;
    }

    if(!CheckAudioInputOutputProperties(&connection.audioIn, g_OutArgs.audioOut))
    {
        nn::audio::StopAudioIn(&connection.audioIn);
        nn::audio::CloseAudioIn(&connection.audioIn);
        nn::os::DestroySystemEvent(connection.bufferEvent.GetBase());
        return false;
    }
    connection.isConnected = true;
    strcpy(connection.name, info.name);
    NNS_LOG("Start audio echoback: %s\n", connection.name);
    return true;
}

void CloseConnection(AudioInConnection& connection)
{
    NN_ABORT_UNLESS(connection.isConnected);
    nn::audio::StopAudioIn(&connection.audioIn);
    nn::audio::CloseAudioIn(&connection.audioIn);

    nn::os::DestroySystemEvent(connection.bufferEvent.GetBase());

    NNS_LOG("Stop %s\n", connection.name);
    connection.isConnected = false;
    memset(connection.name, 0, sizeof(connection.name));
}

int FindAvailableConnection(AudioInConnection* connections, nn::audio::AudioInInfo* infos, int infoCount, int offset)
{
    for (int i = offset; i < infoCount; ++i)
    {
        auto& info = infos[i];
        bool infoConnected = false;
        for (int j = 0; j < MaxAudioIns; ++j)
        {
            auto& connection = connections[j];
            if (connection.isConnected && strcmp(info.name, connection.name) == 0)
            {
                infoConnected = true;
            }
        }
        if (!infoConnected)
        {
            return i;
        }
    }
    return -1;
}

void HandleConnect(AudioInConnection* connections, nn::os::MultiWaitType& multiWait)
{
    nn::audio::AudioInInfo audioInInfos[8];
    int count = nn::audio::ListAudioIns(&audioInInfos[0], 8);

    int offset = 0;
    for (int i = 0; i < MaxAudioIns; ++i)
    {
        auto& connection = connections[i];
        if (!connection.isConnected)
        {
            auto index = FindAvailableConnection(connections, audioInInfos, count, offset);
            if (index != -1)
            {
                if (OpenConnection(connection, audioInInfos[index]))
                {
                    nn::os::InitializeMultiWaitHolder(&connection.bufferHolder, connection.bufferEvent.GetBase());
                    nn::os::SetMultiWaitHolderUserData(&connection.bufferHolder, static_cast<uintptr_t>(i));
                    nn::os::LinkMultiWaitHolder(&multiWait, &connection.bufferHolder);
                    g_MicData[i].Open();
                }
                else
                {
                    ++offset;
                    --i;
                }
            }
        }
    }
}

void HandleDisconnect(AudioInConnection* connections, bool forceDisconnect)
{
    nn::audio::AudioInInfo audioInInfos[8];
    int count = nn::audio::ListAudioIns(&audioInInfos[0], 8);

    for (int i = 0; i < MaxAudioIns; ++i)
    {
        auto& connection = connections[i];
        bool shouldClose = true;

        if (!connection.isConnected)
        {
            continue;
        }
        for (int j = 0; j < count; ++j)
        {
            auto& info = audioInInfos[j];
            if (strcmp(info.name, connection.name) == 0)
            {
                shouldClose = false;
                break;
            }
        }

        if (shouldClose || forceDisconnect)
        {
            CloseConnection(connection);
            nn::os::UnlinkMultiWaitHolder(&connection.bufferHolder);
            nn::os::FinalizeMultiWaitHolder(&connection.bufferHolder);
            g_MicData[i].Close();
        }
    }
}

void HandleUpdate(AudioInConnection& connection, int index)
{
    nn::audio::AudioInBuffer* pAudioInBuffer = nullptr;

    pAudioInBuffer = nn::audio::GetReleasedAudioInBuffer(&connection.audioIn);
    while (pAudioInBuffer)
    {
        // データをコピーし、再度登録します。
        void* pInBuffer = nn::audio::GetAudioInBufferDataPointer(pAudioInBuffer);
        size_t inSize = nn::audio::GetAudioInBufferDataSize(pAudioInBuffer);

        g_MicData[index].PutFrame(pInBuffer, inSize);
        nn::audio::AppendAudioInBuffer(&connection.audioIn, pAudioInBuffer);
        pAudioInBuffer = nn::audio::GetReleasedAudioInBuffer(&connection.audioIn);
    }
}

void AudioInThread(void* arg)
{
    AudioInArgs* args = reinterpret_cast<AudioInArgs*>(arg);
    AudioInConnection connections[MaxAudioIns];
    nn::os::MultiWaitType           multiWait;
    nn::os::MultiWaitHolderType     hotplugHolder;
    nn::os::MultiWaitHolderType     shutdownHolder;

    nn::os::InitializeMultiWait( &multiWait );

    nn::os::InitializeMultiWaitHolder(&shutdownHolder, args->shutdownEvent->GetBase());
    nn::os::LinkMultiWaitHolder(&multiWait, &shutdownHolder);

    const uintptr_t HotplugData = 0xFFFFFFFF;

    nn::os::SystemEvent hotplugEvent;
    auto attachEventResult = AcquireAudioInDeviceNotification(&hotplugEvent);

    if(attachEventResult.IsSuccess())
    {
        nn::os::InitializeMultiWaitHolder(&hotplugHolder, hotplugEvent.GetBase());
        nn::os::SetMultiWaitHolderUserData(&hotplugHolder, HotplugData);
        nn::os::LinkMultiWaitHolder(&multiWait, &hotplugHolder);
    }

    InitializeConnections(connections, args);
    HandleConnect(connections, multiWait);

    while(args->isRunning)
    {
        auto holder = nn::os::WaitAny(&multiWait);

        if(!args->isRunning)
        {
            break;
        }

        if (holder == nullptr)
        {
            continue;
        }

        auto data = nn::os::GetMultiWaitHolderUserData(holder);

        if (data != HotplugData)
        {
            auto index = data;
            auto& connection = connections[index];
            connection.bufferEvent.Clear();
            HandleUpdate(connection, static_cast<int>(index));
        }
        else
        {
            HandleDisconnect(connections, false);
            HandleConnect(connections, multiWait);
            hotplugEvent.Clear();
        }
    }
    HandleDisconnect(connections, true);
    FinalizeConnections(connections, args);

    if(attachEventResult.IsSuccess())
    {
        nn::os::UnlinkMultiWaitHolder(&hotplugHolder);
        nn::os::FinalizeMultiWaitHolder(&hotplugHolder);
        nn::os::DestroySystemEvent(hotplugEvent.GetBase());
    }

    nn::os::UnlinkMultiWaitHolder(&shutdownHolder);
    nn::os::FinalizeMultiWaitHolder(&shutdownHolder);

    nn::os::FinalizeMultiWait(&multiWait);
}

} // namespace

//
// メイン関数です。
//
extern "C" void nnMain()
{
    nn::mem::StandardAllocator allocator(g_HeapBuffer, sizeof(g_HeapBuffer));
    InitializeHidDevices();
    PrintUsage();
    // オーディオ機能を有効にするためのユーティリティ関数を呼び出します。
    // EnableAudioIn() は本ソースコードと同じディレクトリにある AudioEchobackUtil-spec.xxx.cpp で実装されています。
    EnableAudioIn();

    nn::audio::AudioOutParameter audioOutParameter;
    nn::audio::AudioOut audioOut;

    auto argc = nn::os::GetHostArgc();
    auto argv = nn::os::GetHostArgv();
    int bufferLengthInMilliseconds = 50;
    if(argc >= 2)
    {
        bufferLengthInMilliseconds = atoi(argv[1]);
    }
    if(bufferLengthInMilliseconds > 50 || bufferLengthInMilliseconds < 0)
    {
        NNS_LOG("Clamping bufferLengthInMilliseconds from %d to 50\n", bufferLengthInMilliseconds);
        bufferLengthInMilliseconds = 50;
    }
    nn::os::SystemEvent audioOutSystemEvent;
    // オーディオ入力、オーディオ出力をオープンします。

    nn::audio::InitializeAudioOutParameter(&audioOutParameter);
    NN_ABORT_UNLESS(
        nn::audio::OpenDefaultAudioOut(&audioOut, &audioOutSystemEvent, audioOutParameter).IsSuccess(),
        "Failed to open AudioOut."
    );

    int channelCount = nn::audio::GetAudioOutChannelCount(&audioOut);
    int sampleRate = nn::audio::GetAudioOutSampleRate(&audioOut);
    nn::audio::SampleFormat sampleFormat = nn::audio::GetAudioOutSampleFormat(&audioOut);
    // バッファに関するパラメータを準備します。
    NNS_LOG("bufferLengthInMilliseconds = %d\n", bufferLengthInMilliseconds);
    int millisecondsPerSecond = 1000;
    int frameRate = millisecondsPerSecond / bufferLengthInMilliseconds;
    int frameSampleCount = sampleRate / frameRate;
    size_t dataSize = frameSampleCount * channelCount * nn::audio::GetSampleByteSize(sampleFormat);
    size_t bufferSize = nn::util::align_up(dataSize, nn::audio::AudioOutBuffer::SizeGranularity);

    const int outBufferCount = 3;

    for(int i = 0; i < MaxAudioIns; ++i)
    {
        g_MicData[i].SetFrameSize(dataSize);
    }
    // バッファを確保、登録します。
    nn::audio::AudioOutBuffer audioOutBuffer[outBufferCount];
    void* outBuffer[outBufferCount];

    for (int i = 0; i < outBufferCount; ++i)
        {
            outBuffer[i] = allocator.Allocate(bufferSize, nn::audio::AudioOutBuffer::AddressAlignment);
            NN_ASSERT(outBuffer[i]);
            std::memset(outBuffer[i], 0, bufferSize);
            nn::audio::SetAudioOutBufferInfo(&audioOutBuffer[i], outBuffer[i], bufferSize, dataSize);
            nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer[i]);
        }
    // 録音、再生を開始します。
    NN_ABORT_UNLESS(
        nn::audio::StartAudioOut(&audioOut).IsSuccess(),
        "Failed to start AudioOut."
    );

    // 1 フレーム待ちます。
    const nn::TimeSpan interval(nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / frameRate));
    nn::os::SleepThread(interval);

    // 15 秒間、エコーバックを行います。

    g_OutArgs.event = &audioOutSystemEvent;
    g_OutArgs.audioOut = &audioOut;
    g_OutArgs.isRunning = true;

    nn::os::Event audioInShutdownEvent (nn::os::EventClearMode_ManualClear);
    g_InArgs.isRunning = true;
    g_InArgs.allocator = &allocator;
    g_InArgs.dataSize = dataSize;
    g_InArgs.shutdownEvent = &audioInShutdownEvent;

    nn::os::CreateThread(&g_AudioInThread, AudioInThread, &g_InArgs, g_AudioInThreadStack,
                ThreadStackSize, nn::os::HighestThreadPriority);

    nn::os::CreateThread(&g_AudioOutThread, AudioOutThread, &g_OutArgs, g_AudioOutThreadStack,
                    ThreadStackSize, nn::os::HighestThreadPriority);

    nn::os::StartThread(&g_AudioInThread);
    nn::os::StartThread(&g_AudioOutThread);

    for(;;)
    {
        nn::hid::NpadButtonSet npadButtonCurrent = {};
        nn::hid::NpadButtonSet npadButtonDown = {};

        // Npad の入力を取得します。
        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::No1).Test<nn::hid::NpadStyleFullKey>())
        {
            static nn::hid::NpadFullKeyState npadFullKeyState = {};
            nn::hid::NpadFullKeyState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::No1);
            npadButtonCurrent |= state.buttons;
            npadButtonDown |= state.buttons & ~npadFullKeyState.buttons;
            npadFullKeyState = state;
        }
        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld).Test<nn::hid::NpadStyleHandheld>())
        {
            static nn::hid::NpadHandheldState npadHandheldState = {};
            nn::hid::NpadHandheldState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::Handheld);
            npadButtonCurrent |= state.buttons;
            npadButtonDown |= state.buttons & ~npadHandheldState.buttons;
            npadHandheldState = state;
        }

        // DebugPad の入力を取得します。
        {
            static nn::hid::DebugPadState debugPadState = {};
            nn::hid::DebugPadButtonSet debugPadButtonCurrent = {};
            nn::hid::DebugPadButtonSet debugPadButtonDown = {};
            nn::hid::DebugPadState state;
            nn::hid::GetDebugPadState(&state);
            debugPadButtonCurrent |= state.buttons;
            debugPadButtonDown |= state.buttons & ~debugPadState.buttons;
            debugPadState = state;
            nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonCurrent, debugPadButtonCurrent);
            nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonDown, debugPadButtonDown);
        }

        if(npadButtonDown.Test< ::nn::hid::NpadButton::Plus >())
        {
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
    }

    g_OutArgs.isRunning = false;
    nn::os::WaitThread(&g_AudioOutThread);

    g_InArgs.isRunning = false;
    audioInShutdownEvent.Signal();
    nn::os::WaitThread(&g_AudioInThread);

    nn::os::DestroyThread(&g_AudioInThread);
    nn::os::DestroyThread(&g_AudioOutThread);

    // 録音、再生を停止します。
    nn::audio::StopAudioOut(&audioOut);

    // オーディオ入力、オーディオ出力をクローズします。
    nn::audio::CloseAudioOut(&audioOut);

    nn::os::DestroySystemEvent(audioOutSystemEvent.GetBase());

    // メモリを破棄します。

    for (int i = 0; i < outBufferCount; ++i)
    {
        allocator.Free(outBuffer[i]);
    }
    return;
} // NOLINT(readability/fn_size)
