﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AtkViewer.cpp,PageSampleAtkViewer}
 *
 * @brief
 *  nn::atk ライブラリのインゲーム編集機能のサンプルプログラム
 */

/**
 * @page PageSampleAtkViewer インゲーム編集機能
 * @tableofcontents
 *
 * @brief
 *  nn::atk ライブラリのインゲーム編集機能のサンプルプログラムの解説です。
 *
 * @section PageSampleAtkViewer_SectionBrief 概要
 *  SoundMaker とゲームアプリを接続し、ゲームアプリを起動したままサウンドアーカイブに含まれる
 *  パラメータの変更をプレビューすることができるインゲーム編集を利用するための初期化方法などを示したサンプルです。
 *
 * @section PageSampleAtkViewer_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/AtkViewer Samples/Sources/Applications/AtkViewer @endlink 以下にあります。
 *
 * @section PageSampleAtkViewer_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *  インゲーム編集機能を確認するためには、Tools/Audio/SoundMaker/SoundMaker.exe の起動が必要になります。
 *
 * @section PageSampleAtkViewer_SectionHowToOperate 操作方法
 *  コンソールに操作方法が表示されます。
 *  また、SoundMaker からこのサンプルに接続することで、サンプルでプレビューを行いながらサウンドアーカイブの内容の編集や調整が可能になります。
 *  詳しくは NintendoSDK ドキュメントの SoundMaker のインゲーム編集機能のページを参照してください。
 *
 * @section PageSampleAtkViewer_SectionPrecaution 注意事項
 *  実行中のログが、画面上とコンソールに表示され、
 *  起動直後のログとして操作方法が表示されます。
 *
 *  PC 版ではウィンドウは立ち上がらず、コンソール表示のみになる点にご注意ください。
 *
 * @section PageSampleAtkViewer_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAtkViewer_SectionDetail 解説
 *
 * @subsection PageSampleAtkViewer_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  AtkViewer.cpp
 *  @includelineno AtkViewer.cpp
 *
 * @subsection PageSampleAtkViewer_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの Atk に関連している処理の流れは以下の通りです（AtkSimple と違う部分には追加要素と書かれています）。
 *
 *  - nn::fs::MountHostRoot() を実行（追加要素）
 *  - Atk ライブラリの初期化 (InitializeAtk())
 *    - AtkSimple で実行している初期化処理
 *    - ラベル文字列の読み込み（追加要素）
 *    - インゲーム編集ライブラリの初期化（追加要素）
 *  - メインループでの処理
 *    - キー操作があれば再生・停止処理を実行
 *    - SoundArchivePlayer の更新
 *    - インゲーム編集ライブラリの更新（追加要素）
 *  - Atk ライブラリの終了処理 (FinalizeAtk())
 *  - nn::fs::UnmountHostRoot() を実行（追加要素）
 *
 * インゲーム編集ライブラリではアプリケーションから渡されたバッファの範囲内で可能な限りのデータ編集を行います。
 * データ編集に利用するバッファサイズは、SoundEdit::Option にて カスタマイズすることができます。
 *
 */

#include "Common.fsid"

#include <nn/atk.h>

#include <nn/htcs.h>

#include <nn/atk/viewer/atk_SoundEdit.h>

#include <nns/atk/atk_SampleCommon.h>

#include <nns/nns_Log.h>

namespace
{
    const char Title[] = "AtkViewer";
    const char ArchiveRelativePath[] = "Common.bfsar";

    const int SoundHeapSize = 4 * 1024 * 1024;

    nn::atk::SoundHeap          g_SoundHeap;
    nn::atk::FsSoundArchive     g_SoundArchive;
    nn::atk::SoundArchivePlayer g_SoundArchivePlayer;
    nn::atk::SoundDataManager   g_SoundDataManager;

    nn::audio::MemoryPoolType   g_MemoryPool;

    void* g_pMemoryForSoundSystem;
    void* g_pMemoryForSoundHeap;
    void* g_pMemoryForInfoBlock;
    void* g_pMemoryForSoundDataManager;
    void* g_pMemoryForSoundArchivePlayer;
    void* g_pMemoryForStreamBuffer;

    nn::atk::SoundHandle        g_SoundHandle;
    nn::atk::SoundHandle        g_SoundHandleHold;

#if defined(NN_ATK_CONFIG_ENABLE_DEV)
    nn::atk::viewer::SoundEdit g_SoundEdit;
    void* g_pMemoryForSoundEdit;
    void* g_pMemoryForLabelData;
    bool g_IsConnected;
#endif // NN_ATK_CONFIG_ENABLE_DEV
}

void Initialize()
{
    nns::atk::InitializeHeap();
    nns::atk::InitializeFileSystem();
    nns::atk::InitializeHidDevices();

    const nn::Result result = nn::fs::MountHostRoot();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::htcs::Initialize(nns::atk::Allocate, nns::atk::Free);
}

void Finalize()
{
    nn::htcs::Finalize();
    nn::fs::UnmountHostRoot();

    nns::atk::FinalizeHidDevices();
    nns::atk::FinalizeFileSystem();
    nns::atk::FinalizeHeap();
}

void InitializeAtk()
{
    bool isSuccess = true;

    // SoundSystem の初期化
    nn::atk::SoundSystem::SoundSystemParam param;
    std::size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize( param );
    g_pMemoryForSoundSystem = nns::atk::Allocate( memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize );
    isSuccess = nn::atk::SoundSystem::Initialize(
        param,
        reinterpret_cast<uintptr_t>( g_pMemoryForSoundSystem ),
        memSizeForSoundSystem );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundSystem" );

    // SoundHeap の初期化
    g_pMemoryForSoundHeap = nns::atk::Allocate( SoundHeapSize );
    isSuccess = g_SoundHeap.Create( g_pMemoryForSoundHeap, SoundHeapSize );
    NN_ABORT_UNLESS( isSuccess, "cannot create SoundHeap" );

    // SoundArchive の初期化
    const char* archiveAbsolutePath = nns::atk::GetAbsolutePath(ArchiveRelativePath);
    isSuccess = g_SoundArchive.Open(archiveAbsolutePath);
    NN_ABORT_UNLESS( isSuccess, "cannot open SoundArchive(%s)\n", archiveAbsolutePath );

    // SoundArchive のパラメータ情報をメモリにロード
    std::size_t infoBlockSize = g_SoundArchive.GetHeaderSize();
    g_pMemoryForInfoBlock = nns::atk::Allocate( infoBlockSize, nn::atk::FsSoundArchive::BufferAlignSize );
    isSuccess = g_SoundArchive.LoadHeader( g_pMemoryForInfoBlock, infoBlockSize );
    NN_ABORT_UNLESS( isSuccess, "cannot load InfoBlock" );

    // SoundDataManager の初期化
    std::size_t memSizeForSoundDataManager = g_SoundDataManager.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundDataManager = nns::atk::Allocate( memSizeForSoundDataManager, nn::atk::SoundDataManager::BufferAlignSize );
    isSuccess = g_SoundDataManager.Initialize(
        &g_SoundArchive,
        g_pMemoryForSoundDataManager,
        memSizeForSoundDataManager );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundDataManager" );

    // SoundArchivePlayer で用いるストリームバッファの初期化
    // ストリームバッファはメモリプール管理されているヒープから確保する必要があります。
    std::size_t memSizeForStreamBuffer = g_SoundArchivePlayer.GetRequiredStreamBufferSize( &g_SoundArchive );
    g_pMemoryForStreamBuffer = nns::atk::AllocateForMemoryPool(memSizeForStreamBuffer);

    // 専用のヒープをメモリプールにアタッチ
    nn::atk::SoundSystem::AttachMemoryPool(&g_MemoryPool, nns::atk::GetPoolHeapAddress(), nns::atk::GetPoolHeapSize());

    // SoundArchivePlayer の初期化
    std::size_t memSizeForSoundArchivePlayer = g_SoundArchivePlayer.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundArchivePlayer = nns::atk::Allocate( memSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize);
    isSuccess = g_SoundArchivePlayer.Initialize(
        &g_SoundArchive,
        &g_SoundDataManager,
        g_pMemoryForSoundArchivePlayer, memSizeForSoundArchivePlayer,
        g_pMemoryForStreamBuffer, memSizeForStreamBuffer );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundArchivePlayer" );

#if defined(NN_ATK_CONFIG_ENABLE_DEV)
    // SoundArchive のラベル文字列情報をメモリにロード
    std::size_t memSizeForLabelString = g_SoundArchive.GetLabelStringDataSize();
    g_pMemoryForLabelData = nns::atk::Allocate( memSizeForLabelString, nn::atk::FsSoundArchive::BufferAlignSize );
    g_SoundArchive.LoadLabelStringData( g_pMemoryForLabelData, memSizeForLabelString );

    // インゲーム編集機能の初期化
    nn::atk::viewer::SoundEdit::Option option;
    std::size_t memSizeForSoundEdit = g_SoundEdit.GetRequiredMemorySize( option );
    g_pMemoryForSoundEdit = nns::atk::Allocate( memSizeForSoundEdit );

    nn::atk::viewer::SoundEdit::InitializeArg args;
    args.buffer             = g_pMemoryForSoundEdit;
    args.bufferSize         = memSizeForSoundEdit;
    args.soundArchive       = &g_SoundArchive;
    args.soundArchivePlayer = &g_SoundArchivePlayer;
    g_SoundEdit.Initialize( args, option );

    // インゲーム編集を開始
    g_IsConnected = false;
    g_SoundEdit.Start();
#endif
}

void FinalizeAtk()
{
#if defined(NN_ATK_CONFIG_ENABLE_DEV)
    g_SoundEdit.Stop();
    g_SoundEdit.Finalize();
    nns::atk::Free( g_pMemoryForSoundEdit );
    nns::atk::Free( g_pMemoryForLabelData );
#endif // NN_ATK_CONFIG_ENABLE_DEV

    g_SoundArchivePlayer.Finalize();

    // 専用のヒープをメモリプールからデタッチ
    nn::atk::SoundSystem::DetachMemoryPool(&g_MemoryPool);

    g_SoundDataManager.Finalize();
    g_SoundArchive.Close();
    g_SoundHeap.Destroy();
    nn::atk::SoundSystem::Finalize();

    nns::atk::FreeForMemoryPool( g_pMemoryForStreamBuffer );
    nns::atk::Free( g_pMemoryForSoundArchivePlayer );
    nns::atk::Free( g_pMemoryForSoundDataManager );
    nns::atk::Free( g_pMemoryForInfoBlock );
    nns::atk::Free( g_pMemoryForSoundHeap );
    nns::atk::Free( g_pMemoryForSoundSystem );
}

void LoadData()
{
    bool isSuccess = true;

    isSuccess = g_SoundDataManager.LoadData( SEQ_MARIOKART, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SEQ_MARIOKART) failed." );

    isSuccess = g_SoundDataManager.LoadData( SE_YOSHI, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SE_YOSHI) failed." );
}

void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName)
{
    g_SoundHandle.Stop( 0 );

    bool isEdited = false;
#if defined(NN_ATK_CONFIG_ENABLE_DEV)
    isEdited = g_SoundEdit.GetSoundArchiveEditor().GetItemCacheState(
        g_SoundArchive.GetItemLabel(soundId)) == nn::atk::viewer::CacheState_Cached;
#endif

    bool result = g_SoundArchivePlayer.StartSound( &g_SoundHandle, soundId ).IsSuccess();
    NNS_LOG("StartSound(%s)%s ... (%d)\n", debugLabelName, isEdited ? " with edit" : "", result);
}

void PrintUsage()
{
    NNS_LOG("----------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("----------------------------------------\n");
    NNS_LOG("[A]              StartSound SEQ  (SEQ_MARIOKART)\n");
    NNS_LOG("[X]              StartSound WSD  (SE_YOSHI)\n");
    NNS_LOG("[Y]              StartSound STRM (STRM_MARIOKART)\n");
    NNS_LOG("[B]              Stop Sound\n");
    NNS_LOG("[L]              Print Usage\n");
    NNS_LOG("[+/Start][Space] Exit Application\n");
    NNS_LOG("---------------------------------------\n");
}

bool UpdateAtk()
{
    // StartSound / StopSound
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::A >() )
    {
        PlayWithStartSound(SEQ_MARIOKART, "SEQ_MARIOKART");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::X >() )
    {
        PlayWithStartSound(SE_YOSHI, "SE_YOSHI");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Y >() )
    {
        PlayWithStartSound(STRM_MARIOKART, "STRM_MARIOKART");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::B>() )
    {
        g_SoundHandle.Stop( 0 );
    }

    // Print Usage
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::L >() )
    {
        PrintUsage();
    }

    // Exit
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Start >() )
    {
        return false;
    }

    g_SoundArchivePlayer.Update();

#if defined(NN_ATK_CONFIG_ENABLE_DEV)
    g_SoundEdit.Update();

    if ( g_IsConnected != g_SoundEdit.IsConnected() )
    {
        if ( g_IsConnected )
        {
            g_IsConnected = false;
            NNS_LOG("disconnected.\n");
        }
        else
        {
            g_IsConnected = true;
            NNS_LOG("connected.\n");
        }
    }
#endif // NN_ATK_CONFIG_ENABLE_DEV

    return true;
}

extern "C" void nnMain()
{
    Initialize();
    InitializeAtk();

    LoadData();

    PrintUsage();

    for ( ;; )
    {
        nns::atk::UpdateHidDevices();

        if ( !UpdateAtk() )
        {
            break;
        }

        // Vsync の代わり
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    FinalizeAtk();
    Finalize();
}

