﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SoundActor.h"

#include "../FlagList.h"
#include "../GfxCode/DebugViewer.h"

namespace
{
    FlagList g_LocalFlagList(nullptr, 0);

    const char* GetFilterTypeString( int type )
    {
        switch( type )
        {
            case nn::atk::BiquadFilterType_Inherit:
                return "Inherit";
            case nn::atk::BiquadFilterType_None:
                return "None";
            case nn::atk::BiquadFilterType_LowPassFilter:
                return "LPF";
            case nn::atk::BiquadFilterType_HighPassFilter:
                return "HPF";
            case nn::atk::BiquadFilterType_BandPassFilter512:
                return "BPF512";
            case nn::atk::BiquadFilterType_BandPassFilter1024:
                return "BPF1024";
            case nn::atk::BiquadFilterType_BandPassFilter2048:
                return "BPF2048";
            case nn::atk::BiquadFilterType_LowPassFilterNw4fCompatible48k:
                return "LPF_NW4F";
            case nn::atk::BiquadFilterType_HighPassFilterNw4fCompatible48k:
                return "HPF_NW4F";
            case nn::atk::BiquadFilterType_BandPassFilter512Nw4fCompatible48k:
                return "BPF512_NW4F";
            case nn::atk::BiquadFilterType_BandPassFilter1024Nw4fCompatible48k:
                return "BPF1024_NW4F";
            case nn::atk::BiquadFilterType_BandPassFilter2048Nw4fCompatible48k:
                return "BPF2048_NW4F";
            default:
                return "INVALID";
        }
    }
}

void SoundActorCheckModule::CommonObjectForSoundActor::PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName) NN_NOEXCEPT
{
    //  m_SoundActor には同時に再生可能なサウンド数が設定されているため、
    //  m_SoundHandle.Stop する必要はありません

    bool result = m_pSoundActor->StartSound( &GetSoundHandle(), soundId ).IsSuccess();
    NN_LOG("StartSound(%s)%s ... (%d)\n", debugLabelName, CheckCacheState(soundId) ? " with edit" : "", result);
}

void SoundActorCheckModule::OnInitializeAtk() NN_NOEXCEPT
{
    m_CommonObject.Initialize();
    m_CommonObject.SetSoundActorPointer(&m_SoundActor);

    //  SoundActor の初期化
    m_SoundActor.Initialize( &m_CommonObject.GetSoundArchivePlayer(), &m_ActorPlayer );
    m_SoundActor.SetPlayableSoundCount( 0, 2 );
    m_SoundActorPitch = m_SoundActor.GetPitch();
    m_SoundActorVolume = m_SoundActor.GetVolume();
    m_SoundActorBiquadFilterType = nn::atk::BiquadFilterType_Inherit;
    m_SoundActorBiquadFilterValue = 0.5f;

    // ActorPlayer 共有のための実験
    m_SharePlayerSoundActor.Initialize( &m_CommonObject.GetSoundArchivePlayer(), &m_ActorPlayer );
}

void SoundActorCheckModule::OnFinalizeAtk() NN_NOEXCEPT
{
    m_SoundActor.Finalize();
    m_SharePlayerSoundActor.Finalize();

    m_CommonObject.SetSoundActorPointer(nullptr);
    m_CommonObject.Finalize();
}

void SoundActorCheckModule::OnLoadData() NN_NOEXCEPT
{
    m_CommonObject.LoadData();
}

void SoundActorCheckModule::OnPrintUsage() NN_NOEXCEPT
{
    m_CommonObject.PrintUsage();

    NN_LOG("[Left]         SoundActor Volume Down\n");
    NN_LOG("[Right]        SoundActor Volume Up\n");
    NN_LOG("[R + Left]     SoundActor Pitch  Down\n");
    NN_LOG("[R + Right]    SoundActor Pitch  Up\n");
    NN_LOG("[ZR(Z) + Right]SoundActor FilterValue Up\n");
    NN_LOG("[ZR(Z) + Left] SoundActor FilterValue Down\n");
    NN_LOG("[ZR(Z) + Up]   SoundActor FilterType Change\n");
    NN_LOG("[ZR(Z) + Down] SoundActor FilterType Change\n");
}

void SoundActorCheckModule::OnUpdateInput() NN_NOEXCEPT
{
    m_CommonObject.UpdateInput();

    if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::ZR >() )
    {
        bool isNeedToApply = false;
        const int BiquadFilterTypeMin = nn::atk::BiquadFilterType_Inherit;
        const int BiquadFilterTypeMax = nn::atk::BiquadFilterType_BandPassFilter2048Nw4fCompatible48k;
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Up >() )
        {
            m_SoundActorBiquadFilterType--;
            if( m_SoundActorBiquadFilterType < BiquadFilterTypeMin )
            {
                m_SoundActorBiquadFilterType = BiquadFilterTypeMax;
            }
            isNeedToApply= true;
        }
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Down >() )
        {
            m_SoundActorBiquadFilterType++;
            if( m_SoundActorBiquadFilterType > BiquadFilterTypeMax )
            {
                m_SoundActorBiquadFilterType = BiquadFilterTypeMin;
            }
            isNeedToApply= true;
        }

        const float FilterValueMin = 0.0f;
        const float FilterValueMax = 1.0f;
        const float FilterValueDiff = 0.125f;   //  入力 1 回あたりのフィルタ係数の変化量
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Left >() )
        {
            m_SoundActorBiquadFilterValue -= FilterValueDiff;
            if( m_SoundActorBiquadFilterValue < FilterValueMin )
            {
                m_SoundActorBiquadFilterValue = FilterValueMin;
            }
            isNeedToApply = true;
        }
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Right >() )
        {
            m_SoundActorBiquadFilterValue += FilterValueDiff;
            if( m_SoundActorBiquadFilterValue > FilterValueMax )
            {
                m_SoundActorBiquadFilterValue = FilterValueMax;
            }
            isNeedToApply = true;
        }

        if( isNeedToApply )
        {
            m_SoundActor.SetBiquadFilter( m_SoundActorBiquadFilterType, m_SoundActorBiquadFilterValue );
            NN_LOG( "[SoundActor] Filter: %s, %f\n", GetFilterTypeString( m_SoundActorBiquadFilterType ), m_SoundActorBiquadFilterValue );
        }
    }
    else if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::R >() )
    {
        const float PitchDiff = 0.125f;  //  入力 1 回あたりのピッチの変化量
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Left >() )
        {
            m_SoundActorPitch -= PitchDiff;
            m_SoundActor.SetPitch( m_SoundActorPitch );
            NN_LOG( "[SoundActor] Pitch: %f\n", m_SoundActorPitch );
        }
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Right >() )
        {
            m_SoundActorPitch += PitchDiff;
            m_SoundActor.SetPitch( m_SoundActorPitch );
            NN_LOG( "[SoundActor] Pitch: %f\n", m_SoundActorPitch );
        }
    }
    else
    {
        const float VolumeDiff = 0.25f;  //  入力 1 回あたりのボリュームの変化量
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Left >() )
        {
            m_SoundActorVolume -= VolumeDiff;
            m_SoundActor.SetVolume( m_SoundActorVolume );
            NN_LOG( "[SoundActor] Volume: %f\n", m_SoundActorVolume );
        }
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Right >() )
        {
            m_SoundActorVolume += VolumeDiff;
            m_SoundActor.SetVolume( m_SoundActorVolume );
            NN_LOG( "[SoundActor] Volume: %f\n", m_SoundActorVolume );
        }
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Up >() )
        {
            bool result = m_SoundActor.StartSound(&m_CommonObject.GetSoundHandle(), STRM_MARIOKART).IsSuccess();
            NN_LOG( "[SoundActor] StartSound: %s(%d)\n", "STRM_MARIOKART", result );
        }
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Down >() )
        {
            bool result = m_SharePlayerSoundActor.StartSound(&m_CommonObject.GetSoundHandle(), STRM_MARIOKART).IsSuccess();
            NN_LOG( "[SamePlayerSoundActor] StartSound: %s(%d)\n", "STRM_MARIOKART", result );
        }
    }
}

void SoundActorCheckModule::OnUpdateAtk() NN_NOEXCEPT
{
    m_CommonObject.Update();
}

#if defined( NN_ATK_ENABLE_GFX_VIEWING )
void SoundActorCheckModule::OnUpdateDraw() NN_NOEXCEPT
{
    m_CommonObject.UpdateDraw(GetModuleName());
}
#endif

FlagList& SoundActorCheckModule::GetLocalFlagList() NN_NOEXCEPT
{
    return g_LocalFlagList;
}
