﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ExternalFile.h"

#include <nn/nn_Assert.h>

#include "../FlagList.h"
#include "../GfxCode/DebugViewer.h"

namespace
{
    FlagList g_LocalFlagList(nullptr, 0);
}

void ExternalFileCheckModule::OnInitializeAtk() NN_NOEXCEPT
{
    // メモリに波形アーカイブをロード
    {
        nn::fs::FileHandle fileHandle;

        const char* ExternalBfwarPath = "content:/cache/WARC_WSDTEST.bfwar";

        nn::Result result = nn::fs::OpenFile( &fileHandle, ExternalBfwarPath, nn::fs::OpenMode_Read );
        NN_ASSERT(result.IsSuccess());

        result = nn::fs::GetFileSize( &m_SizeForExternalBfwarFile, fileHandle );
        m_pMemoryForExternalBfwarFile = nns::atk::AllocateForMemoryPool(static_cast<size_t>(m_SizeForExternalBfwarFile), nn::audio::BufferAlignSize );
        nn::fs::ReadFile( fileHandle, 0, m_pMemoryForExternalBfwarFile, static_cast<size_t>(m_SizeForExternalBfwarFile) );
        nn::fs::CloseFile( fileHandle );
    }

    // 外部波形ファイルをメモリにロード
    {
        nn::fs::FileHandle fileHandle;

        const char* ExternalBfwavPath = "content:/cache/wihaho.dspadpcm.bfwav";

        int64_t sizeForExternalBfwavFile;
        nn::Result result = nn::fs::OpenFile( &fileHandle, ExternalBfwavPath, nn::fs::OpenMode_Read );
        NN_ASSERT(result.IsSuccess());

        result = nn::fs::GetFileSize( &sizeForExternalBfwavFile, fileHandle );
        m_pMemoryForExternalBfwavFile = nns::atk::AllocateForMemoryPool(static_cast<size_t>(sizeForExternalBfwavFile), nn::audio::BufferAlignSize );
        nn::fs::ReadFile( fileHandle, 0, m_pMemoryForExternalBfwavFile, static_cast<size_t>(sizeForExternalBfwavFile) );
        nn::fs::CloseFile( fileHandle );
    }

    m_CommonObject.Initialize();
}

void ExternalFileCheckModule::OnFinalizeAtk() NN_NOEXCEPT
{
    // SoundDataManager の登録を解除し、波形アーカイブファイルをメモリから解放
    m_CommonObject.GetSoundDataManager().InvalidateSoundData( m_pMemoryForExternalBfwarFile, static_cast<size_t>(m_SizeForExternalBfwarFile) );
    nns::atk::FreeForMemoryPool( m_pMemoryForExternalBfwarFile );

    // 外部波形ファイルのメモリ開放
    nns::atk::FreeForMemoryPool( m_pMemoryForExternalBfwavFile );

    m_CommonObject.Finalize();
}

void ExternalFileCheckModule::OnLoadData() NN_NOEXCEPT
{
    m_CommonObject.LoadData();

    // SoundDataManager への登録
    uint32_t fileId = m_CommonObject.GetSoundArchive().GetItemFileId( WARC_WSDTEST );
    m_CommonObject.GetSoundDataManager().SetFileAddress( fileId, m_pMemoryForExternalBfwarFile );
}

void ExternalFileCheckModule::OnPrintUsage() NN_NOEXCEPT
{
    m_CommonObject.PrintUsage();

    NN_LOG("[Up]           StartSound Same Track and Channel Count STRM (STRM_PIANO16_PCM16)\n");
    NN_LOG("[Left]         StartSound Different Channel Count STRM (STRM_MARIOKART)\n");
    NN_LOG("[Right]        StartSound Different Track Count STRM (STRM_MULTITRACK)\n");
    NN_LOG("[Down]         StartSound External WSD (WSD_WIHAHO)\n");
}

void ExternalFileCheckModule::OnUpdateInput() NN_NOEXCEPT
{
    m_CommonObject.UpdateInput();
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Up >() )
    {
        // StartSound Same Track and Channel Count STRM
        nn::atk::SoundStartable::StartInfo info;
        info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundInfo;

        info.streamSoundInfo.externalPath = "content:/stream/PIANO16.pcm16.bfstm";

        CommonObject::StartParam startParam;
        startParam.pStartInfo = &info;

        m_CommonObject.PlayWithStartSound(STRM_PIANO16_PCM16, "STRM_PIANO16_PCM16 (External)", startParam);
    }
    else if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Left >() )
    {
        // StartSound Different Channel Count STRM
        nn::atk::SoundStartable::StartInfo info;
        info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundInfo |
                           nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundMetaInfo;
        info.streamSoundInfo.externalPath = "content:/stream/kart_title.32.dspadpcm.bfstm";

        const nn::atk::SoundArchive& player = m_CommonObject.GetSoundArchivePlayer().GetSoundArchive();
        // 既存のパラメータをメタ情報として流用
        player.ReadStreamSoundInfo(&info.streamSoundMetaInfo, STRM_PIANO16_PCM16);

        info.streamSoundMetaInfo.trackInfo[0].channelCount = 2;
        info.streamSoundMetaInfo.Setup();

        CommonObject::StartParam startParam;
        startParam.pStartInfo = &info;

        m_CommonObject.PlayWithStartSound(STRM_PIANO16_PCM16, "STRM_MARIOKART (External)", startParam);
    }
    else if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Right >() )
    {
        // StartSound Different Track Count STRM
        nn::atk::SoundStartable::StartInfo info;
        info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundInfo |
                           nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundMetaInfo;
        info.streamSoundInfo.externalPath = "content:/stream/STRM_MULTITRACK.bfstm";

        const nn::atk::SoundArchive& player = m_CommonObject.GetSoundArchivePlayer().GetSoundArchive();
        // 既存のパラメータをメタ情報として流用
        player.ReadStreamSoundInfo(&info.streamSoundMetaInfo, STRM_PIANO16_PCM16);

        info.streamSoundMetaInfo.allocateTrackFlags = 0x7;
        info.streamSoundMetaInfo.trackInfo[1] = info.streamSoundMetaInfo.trackInfo[0];
        info.streamSoundMetaInfo.trackInfo[2] = info.streamSoundMetaInfo.trackInfo[0];
        info.streamSoundMetaInfo.Setup();

        CommonObject::StartParam startParam;
        startParam.pStartInfo = &info;

        m_CommonObject.PlayWithStartSound(STRM_PIANO16_PCM16, "STRM_MULTITRACK (External)", startParam);
    }
    else if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Down >() )
    {
        // StartSound External WSD
        nn::atk::SoundStartable::StartInfo info;
        info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StartOffset;
        info.startOffset = 8000;
        info.startOffsetType = info.StartOffsetType_Sample;
        info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_WaveSoundInfo;

        info.waveSoundInfo.waveAddress = m_pMemoryForExternalBfwavFile;
        info.waveSoundInfo.enableParameterFlag |= nn::atk::SoundStartable::StartInfo::WaveSoundInfo::EnableParameterFlagBit_Release;
        info.waveSoundInfo.enableParameterFlag |= nn::atk::SoundStartable::StartInfo::WaveSoundInfo::EnableParameterFlagBit_ContextCalculationSkipMode;
        info.waveSoundInfo.release = 110;
        info.waveSoundInfo.isContextCalculationSkipMode = true;

        CommonObject::StartParam startParam;
        startParam.pStartInfo = &info;

        m_CommonObject.PlayWithStartSound(WSD_YOSHI_PCM16, "WSD_WIHAHO (External)", startParam);
    }
}

void ExternalFileCheckModule::OnUpdateAtk() NN_NOEXCEPT
{
    m_CommonObject.Update();
}

#if defined( NN_ATK_ENABLE_GFX_VIEWING )
void ExternalFileCheckModule::OnUpdateDraw() NN_NOEXCEPT
{
    m_CommonObject.UpdateDraw(GetModuleName());
}
#endif

FlagList& ExternalFileCheckModule::GetLocalFlagList() NN_NOEXCEPT
{
    return g_LocalFlagList;
}
