﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AtkGroup.cpp,PageSampleAtkGroup}
 *
 * @brief
 *  nn::atk ライブラリのグループを使ったサウンドデータの読み込みを示すサンプルプログラム
 */

/**
 * @page PageSampleAtkGroup グループの読み込みを用いた再生
 * @tableofcontents
 *
 * @brief
 *  nn::atk ライブラリのグループを使ったサウンドデータの読み込みに関する機能のサンプルプログラムの解説です。
 *
 * @section PageSampleAtkGroup_SectionBrief 概要
 *  Atk ライブラリでグループを使ったサウンドデータの読み込みを行うためのサンプルです。
 *
 * @section PageSampleAtkGroup_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/AtkGroup Samples/Sources/Applications/AtkGroup @endlink 以下にあります。
 *
 * @section PageSampleAtkGroup_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleAtkGroup_SectionHowToOperate 操作方法
 *  コンソールに操作方法が表示されます。
 *
 * @section PageSampleAtkGroup_SectionPrecaution 注意事項
 *  実行中のログが、画面上とコンソールに表示され、
 *  起動直後のログとして操作方法が表示されます。
 *
 *  PC 版ではウィンドウは立ち上がらず、コンソール表示のみになる点にご注意ください。
 *
 * @section PageSampleAtkGroup_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAtkGroup_SectionDetail 解説
 *
 * @subsection PageSampleAtkGroup_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  AtkGroup.cpp
 *  @includelineno AtkGroup.cpp
 *
 * @subsection PageSampleAtkGroup_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの Atk に関連している処理の流れは以下の通りです(AtkSimple と違う部分には追加要素と書かれています)。
 *
 *  - アロケータの初期化
 *    - 内部で２種類（メモリプール管理対象用とそれ以外）のアロケータを初期化。
 *  - Atk ライブラリの初期化 (InitializeAtk())
 *    - SoundSystem の初期化
 *    - SoundHeap の初期化
 *    - SoundArchive をオープンし、パラメータ情報をロード
 *    - SoundDataManager の初期化
 *    - SoundArchivePlayer で用いるストリームバッファをメモリプール管理対象用アロケータから確保し初期化
 *    - メモリプール管理対象のメモリをメモリプールにアタッチ
 *    - SoundArchivePlayer の初期化
 *  - SoundHeap に埋め込みグループとリンクグループをロード (LoadData()) (追加要素)
 *  - メモリ上にユーザー管理グループをロード (LoadData()) (追加要素)
 *  - メインループでの処理
 *    - キー操作があれば再生・停止処理を実行
 *    - メモリ上にロードしたユーザー管理グループを SoundDataManager に設定・解除 (追加要素)
 *    - SoundArchivePlayer の更新
 *  - Atk ライブラリの終了処理 (FinalizeAtk())
 *    - SoundArchivePlayer の 終了処理
 *    - メモリプール管理対象のメモリをメモリプールからデタッチ
 *    - その他の終了処理・メモリ解放
 *  - アロケータの終了処理
 *
 *
 *  キー操作を行うことで、ウェーブサウンド、シーケンスサウンドを再生します。
 *  このサンプルプログラムでは再生用のAPIとして、StartSound を使用しています。
 *  StartSoundは一度実行すると停止処理を行うか、サウンドの再生が終わるまで再生を続けます。
 *
 *  メモリプール管理対象のメモリのアタッチ・デタッチに関しては以下の点にご注意ください。
 *  - メモリプールのアタッチ・デタッチは即座に反映されないため、 SoundSystem が初期化された状態で、
 *    アタッチ・デタッチのリクエストを送り、しばらく待つ必要がある
 *  - 再生中にはメモリプールをデタッチできないため、明示的に停止させるか SoundArchivePlayer::Finalize()
 *    を実行したあとにメモリプールをデタッチする必要がある
 */

#include "Common.fsid"

#include <nn/atk.h>

#include <nns/atk/atk_SampleCommon.h>

#include <nns/nns_Log.h>

namespace
{
    const char Title[] = "AtkGroup";
    const char ArchiveRelativePath[] = "Common.bfsar";

    const int SoundHeapSize = 4 * 1024 * 1024;

    nn::atk::SoundHeap          g_SoundHeap;
    nn::atk::FsSoundArchive     g_SoundArchive;
    nn::atk::SoundArchivePlayer g_SoundArchivePlayer;
    nn::atk::SoundDataManager   g_SoundDataManager;

    nn::audio::MemoryPoolType   g_MemoryPool;

    void* g_pMemoryForSoundSystem;
    void* g_pMemoryForSoundHeap;
    void* g_pMemoryForInfoBlock;
    void* g_pMemoryForSoundDataManager;
    void* g_pMemoryForSoundArchivePlayer;
    void* g_pMemoryForStreamBuffer;

    nn::atk::SoundHandle        g_SoundHandle;
    nn::atk::SoundHandle        g_SoundHandleHold;

    const int   GroupFileCount = 3;
    const char* GroupFilesRelativePath[] =
    {
        "GROUP_USER.bfgrp",
        "GROUP_USER_AUTO_WARC.bfgrp",
        "GROUP_USER_INDIVIDUAL_WARC.bfgrp"
    };
    bool    g_IsSetGroupFile = false;
    int64_t g_GroupFileSizes[GroupFileCount];
    void*   g_pMemoryForGroupFiles[GroupFileCount];
}

void Initialize()
{
    nns::atk::InitializeHeap();
    nns::atk::InitializeFileSystem();
    nns::atk::InitializeHidDevices();
}

void Finalize()
{
    nns::atk::FinalizeHidDevices();
    nns::atk::FinalizeFileSystem();
    nns::atk::FinalizeHeap();
}

void InitializeAtk()
{
    bool isSuccess = true;

    // SoundSystem の初期化
    nn::atk::SoundSystem::SoundSystemParam param;
    std::size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize( param );
    g_pMemoryForSoundSystem = nns::atk::Allocate( memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize );
    isSuccess = nn::atk::SoundSystem::Initialize(
        param,
        reinterpret_cast<uintptr_t>( g_pMemoryForSoundSystem ),
        memSizeForSoundSystem );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundSystem" );

    // SoundHeap の初期化
    g_pMemoryForSoundHeap = nns::atk::Allocate( SoundHeapSize );
    isSuccess = g_SoundHeap.Create( g_pMemoryForSoundHeap, SoundHeapSize );
    NN_ABORT_UNLESS( isSuccess, "cannot create SoundHeap" );

    // SoundArchive の初期化
    const char* archiveAbsolutePath = nns::atk::GetAbsolutePath(ArchiveRelativePath);
    isSuccess = g_SoundArchive.Open(archiveAbsolutePath);
    NN_ABORT_UNLESS( isSuccess, "cannot open SoundArchive(%s)\n", archiveAbsolutePath );

    // SoundArchive のパラメータ情報をメモリにロード
    std::size_t infoBlockSize = g_SoundArchive.GetHeaderSize();
    g_pMemoryForInfoBlock = nns::atk::Allocate(  infoBlockSize, nn::atk::FsSoundArchive::BufferAlignSize );
    isSuccess = g_SoundArchive.LoadHeader( g_pMemoryForInfoBlock, infoBlockSize );
    NN_ABORT_UNLESS( isSuccess, "cannot load InfoBlock" );

    // SoundDataManager の初期化
    std::size_t memSizeForSoundDataManager = g_SoundDataManager.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundDataManager = nns::atk::Allocate( memSizeForSoundDataManager, nn::atk::SoundDataManager::BufferAlignSize );
    isSuccess = g_SoundDataManager.Initialize(
        &g_SoundArchive,
        g_pMemoryForSoundDataManager,
        memSizeForSoundDataManager );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundDataManager" );

    // SoundArchivePlayer で用いるストリームバッファの初期化
    // ストリームバッファはメモリプール管理されているヒープから確保する必要があります。
    std::size_t memSizeForStreamBuffer = g_SoundArchivePlayer.GetRequiredStreamBufferSize( &g_SoundArchive );
    g_pMemoryForStreamBuffer = nns::atk::AllocateForMemoryPool(memSizeForStreamBuffer);

    // 専用のヒープをメモリプールにアタッチ
    nn::atk::SoundSystem::AttachMemoryPool(&g_MemoryPool, nns::atk::GetPoolHeapAddress(), nns::atk::GetPoolHeapSize());

    // SoundArchivePlayer の初期化
    std::size_t memSizeForSoundArchivePlayer = g_SoundArchivePlayer.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundArchivePlayer = nns::atk::Allocate( memSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize);
    isSuccess = g_SoundArchivePlayer.Initialize(
        &g_SoundArchive,
        &g_SoundDataManager,
        g_pMemoryForSoundArchivePlayer, memSizeForSoundArchivePlayer,
        g_pMemoryForStreamBuffer, memSizeForStreamBuffer );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundArchivePlayer" );
}

void FinalizeAtk()
{
    g_SoundArchivePlayer.Finalize();

    // 専用のヒープをメモリプールからデタッチ
    nn::atk::SoundSystem::DetachMemoryPool(&g_MemoryPool);

    g_SoundDataManager.Finalize();
    g_SoundArchive.Close();
    g_SoundHeap.Destroy();
    nn::atk::SoundSystem::Finalize();

    nns::atk::FreeForMemoryPool(g_pMemoryForStreamBuffer);
    nns::atk::Free(g_pMemoryForSoundArchivePlayer);
    nns::atk::Free(g_pMemoryForSoundDataManager);
    nns::atk::Free(g_pMemoryForInfoBlock);
    nns::atk::Free(g_pMemoryForSoundHeap);
    nns::atk::Free(g_pMemoryForSoundSystem);

    for ( int i = 0; i < GroupFileCount; i++ )
    {
        nns::atk::FreeForMemoryPool(g_pMemoryForGroupFiles[i]);
    }
}

void LoadData()
{
    bool isSuccess = true;

    // 「埋め込み」グループのデータの読み込み
    isSuccess = g_SoundDataManager.LoadData( GROUP_EMBEDDED, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(GROUP_EMBEDDED) failed." );

    // 「リンク」グループのデータの読み込み
    isSuccess = g_SoundDataManager.LoadData( GROUP_LINKED, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(GROUP_LINKED) failed." );

    // 「ユーザー管理」グループのグループファイルの読み込み
    nn::fs::FileHandle fileHandle;
    for ( int i = 0; i < GroupFileCount; i++ )
    {
        const char* GroupFileAbsolutePath = nns::atk::GetAbsolutePath(GroupFilesRelativePath[i]);

        nn::Result result = nn::fs::OpenFile(&fileHandle, GroupFileAbsolutePath, nn::fs::OpenMode_Read);
        NN_ABORT_UNLESS( result.IsSuccess() );
        result = nn::fs::GetFileSize(&g_GroupFileSizes[i], fileHandle);
        NN_ABORT_UNLESS( result.IsSuccess() );
        g_pMemoryForGroupFiles[i] = nns::atk::AllocateForMemoryPool( static_cast<size_t>(g_GroupFileSizes[i]) );

        size_t readSize = 0;
        result = nn::fs::ReadFile(&readSize, fileHandle, 0, g_pMemoryForGroupFiles[i], static_cast<size_t>(g_GroupFileSizes[i]));

        NN_ABORT_UNLESS( result.IsSuccess() );
        nn::fs::CloseFile(fileHandle);
    }
    g_IsSetGroupFile = false;
}

void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName)
{
    g_SoundHandle.Stop( 0 );

    bool result = g_SoundArchivePlayer.StartSound( &g_SoundHandle, soundId ).IsSuccess();
    NNS_LOG("StartSound(%s) ... (%d)\n", debugLabelName, result);
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("[A]              StartSound WSD  (SE_YOSHI_FOR_LINKEDGROUP)\n");
    NNS_LOG("[X]              StartSound WSD  (SE_YOSHI) [In UserManagementGroup, need to set GroupFile.(Press R Button)]\n");
    NNS_LOG("[Y]              StartSound WSD  (SE_YOSHI_FOR_EMBEDDEDGROUP)\n");
    NNS_LOG("[B]              Stop Sound\n");
    NNS_LOG("[L]              Print Usage\n");
    NNS_LOG("[R]              Set/Unset GroupFile\n");
    NNS_LOG("[Start][Space]   Exit Application\n");
    NNS_LOG("--------------------------------------------------\n");
}

bool UpdateAtk()
{
    // StartSound / Stop
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::A >() )
    {
        PlayWithStartSound(SE_YOSHI_FOR_LINKEDGROUP, "SE_YOSHI_FOR_LINKEDGROUP");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::X >() )
    {
        // SoundDataManager にユーザー管理グループがセットされていない場合は再生に失敗します
        PlayWithStartSound(SE_YOSHI, "SE_YOSHI");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Y >() )
    {
        PlayWithStartSound(SE_YOSHI_FOR_EMBEDDEDGROUP, "SE_YOSHI_FOR_EMBEDDEDGROUP");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::B >() )
    {
        g_SoundHandle.Stop( 0 );
    }

    // Print Usage
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::L >() )
    {
        PrintUsage();
    }

    // Set/Unset Group Files
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::R >() )
    {
        if ( g_IsSetGroupFile )
        {
            // ロード済みグループファイルの登録解除
            for ( int i = 0; i < GroupFileCount; i++ )
            {
                g_SoundDataManager.ClearFileAddressInGroupFile(g_pMemoryForGroupFiles[i], static_cast<size_t>(g_GroupFileSizes[i]));
            }
            NNS_LOG("Unset GroupFile Address\n");
        }
        else
        {
            // ロード済みグループファイルの登録
            for ( int i = 0; i < GroupFileCount; i++ )
            {
                g_SoundDataManager.SetFileAddressInGroupFile(g_pMemoryForGroupFiles[i], static_cast<size_t>(g_GroupFileSizes[i]));
            }
            NNS_LOG("Set GroupFile Address\n");
        }
        g_IsSetGroupFile = !g_IsSetGroupFile;
    }

    // Exit
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Start >() )
    {
        return false;
    }

    g_SoundArchivePlayer.Update();

    return true;
}

extern "C" void nnMain()
{
    Initialize();
    InitializeAtk();

    LoadData();

    PrintUsage();

    for ( ;; )
    {
        nns::atk::UpdateHidDevices();

        if ( !UpdateAtk() )
        {
            break;
        }

        // Vsync の代わり
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    FinalizeAtk();
    Finalize();
}

