﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "AccountSelection_SequenceBase.h"

#include <cstring>
#include <memory>

#include <nn/os/os_Thread.h>
#include <nn/util/util_FormatString.h>

namespace {
class ErrorDialog final
    : public SequenceBase::DialogBase
{
public:
    explicit ErrorDialog(const char* message) NN_NOEXCEPT
    {
        auto& components = *GetComponentsPtr();
        components.Reset("Error", message, this);
        components.AddOption("OK", DialogBase::CloseHandler<decltype(*this)>);

        NNS_MIGRATION_LOG_LN("Error: %s", message);
    }
};

template <int Scale>
class Meter
{
private:
    int m_Counter;
public:
    static const int RequiredBufferSize = Scale;

    Meter() NN_NOEXCEPT
        : m_Counter(0)
    {
    }
    void Update(char* bar, size_t barSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(barSize >= RequiredBufferSize);
        NN_UNUSED(barSize);

        m_Counter = (m_Counter + 1) % (Scale * 2);
        auto offset = Scale - m_Counter;
        for (int i = 0; i < Scale; ++ i)
        {
            auto k = i + offset;
            bar[i] = (k < 0 || k >= Scale) ? ' ' : '>';
        }
        bar[Scale] = '\0';
    }
};

class SyncProgressDialog final
    : public SequenceBase::DialogBase
{
private:
    static NN_ALIGNAS(4096) char s_Stack[4 * 4096];
    static nn::os::MutexType s_Lock;

    static void ThreadFunction(void* data) NN_NOEXCEPT
    {
        auto& obj = *reinterpret_cast<SyncProgressDialog*>(data);
        obj.m_SyncResult = obj.m_Sync.context();
        obj.m_SyncFinished = true;
    }

private:
    nn::os::ThreadType m_Thread;

    Meter<32> m_Meter;
    char m_Label[128];
    SequenceBase::SyncTask m_Sync;
    nn::Result m_SyncResult;
    std::atomic<bool> m_SyncFinished;
    bool m_Done;

public:
    SyncProgressDialog(const char* label, SequenceBase::SyncTask&& sync) NN_NOEXCEPT
        : m_Sync(std::move(sync))
        , m_SyncFinished(false)
        , m_Done(false)
    {
        NN_ABORT_UNLESS(nn::os::TryLockMutex(&s_Lock), "SyncProgressDialog::s_Lock is already locked");

        nn::util::SNPrintf(m_Label, sizeof(m_Label), "%s", label);

        auto& components = *GetComponentsPtr();
        components.Reset(m_Label, "Processing function...", this);

        NNS_MIGRATION_LOG_LN("Sync: %s", label);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_Thread, ThreadFunction, this, s_Stack, sizeof(s_Stack), nn::os::DefaultThreadPriority));
        nn::os::StartThread(&m_Thread);
    }
    ~SyncProgressDialog() NN_NOEXCEPT
    {
        nn::os::DestroyThread(&m_Thread);
        nn::os::UnlockMutex(&s_Lock);
    }
    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_SyncFinished)
        {
            if (!m_Done)
            {
                m_Done = true;

                auto& components = *GetComponentsPtr();
                components.Reset(m_Label, "Synchronous function done", this);
                if (m_Sync.resultHandler(m_Sync.data, m_Sync.dataSize, m_SyncResult))
                {
                    this->RequestClose();
                    return;
                }
                components.AddOption("OK", DialogBase::CloseHandler<decltype(*this)>);
            }
        }
        else
        {
            char bar[decltype(m_Meter)::RequiredBufferSize];
            m_Meter.Update(bar, sizeof(bar));
            auto& components = *GetComponentsPtr();
            std::strncpy(components.description, bar, sizeof(components.description));
        }
    }
};

NN_ALIGNAS(4096) char SyncProgressDialog::s_Stack[4 * 4096];
nn::os::MutexType SyncProgressDialog::s_Lock = NN_OS_MUTEX_INITIALIZER(false);
} // ~namespace <anonymous>

SequenceBase::SequenceBase(Window& window) NN_NOEXCEPT
    :  m_Window(window)
    , m_DialogStackDepth(0)
    , m_ErrorOccurred(false)
    , m_CloseRequired(false)
{
    m_DialogCaller._obj = this;
}
SequenceBase::~SequenceBase() NN_NOEXCEPT
{
    while (m_DialogStackDepth > 0)
    {
        Pop();
    }
}
void SequenceBase::PushError(const char* message) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_DialogStackDepth < std::extent<decltype(m_DialogStack[0])>::value - 1);
    m_DialogStack[m_DialogStackDepth] = new ErrorDialog(message);
    ++m_DialogStackDepth;

    const auto* sp = m_DialogStack[m_DialogStackDepth - 1];
    m_Window.SetComponents(sp->GetComponentsPtr());
    m_ErrorOccurred = true;
}
void SequenceBase::PushSyncProgress(const char* label, SyncTask&& sync) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_DialogStackDepth < std::extent<decltype(m_DialogStack[0])>::value - 1);
    m_DialogStack[m_DialogStackDepth] = new SyncProgressDialog(label, std::move(sync));
    ++m_DialogStackDepth;

    const auto* sp = m_DialogStack[m_DialogStackDepth - 1];
    m_Window.SetComponents(sp->GetComponentsPtr());
}
void SequenceBase::Pop() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_DialogStackDepth > 0);
    --m_DialogStackDepth;
    delete m_DialogStack[m_DialogStackDepth];

    if (m_DialogStackDepth > 0)
    {
        const auto* sp = m_DialogStack[m_DialogStackDepth - 1];
        m_Window.SetComponents(sp->GetComponentsPtr());

        NNS_MIGRATION_LOG_LN("Popped to dialog");
    }
    else
    {
        m_Window.SetComponents(GetCurrentSceneComponentsPtr());

        NNS_MIGRATION_LOG_LN("Popped to scene");
    }
}
void SequenceBase::NotifyCurrentSceneUpdate() NN_NOEXCEPT
{
    if (m_DialogStackDepth == 0)
    {
        m_Window.SetComponents(GetCurrentSceneComponentsPtr());
    }
}
bool SequenceBase::IsErrorOccurred() const NN_NOEXCEPT
{
    return m_ErrorOccurred;
}
void SequenceBase::RequestClose() NN_NOEXCEPT
{
    m_CloseRequired = true;
}
bool SequenceBase::IsClosed() const NN_NOEXCEPT
{
    return m_CloseRequired;
}

SequenceBase::DialogCaller& SequenceBase::GetDialogCallerRef() NN_NOEXCEPT
{
    return m_DialogCaller;
}
void SequenceBase::Update() NN_NOEXCEPT
{
    if (m_DialogStackDepth > 0)
    {
        m_DialogStack[m_DialogStackDepth - 1]->Update();
        if (m_DialogStack[m_DialogStackDepth - 1]->IsClosed())
        {
            Pop();
        }
    }
    UpdateImpl();

    // window.title を設定
    if (m_DialogStackDepth > 0)
    {
        char title[256];
        int offset = 0;
        auto& components = *GetCurrentSceneComponentsPtr();
        offset += nn::util::SNPrintf(title, sizeof(title), "%s", components.title);
        for (int i = 0; i < m_DialogStackDepth && offset < sizeof(title); ++i)
        {
            auto& c = *static_cast<const SequenceBase::DialogBase*>(m_DialogStack[m_DialogStackDepth - 1])->GetComponentsPtr();
            nn::util::SNPrintf(title + offset, sizeof(title) - offset, " > %s", c.title);
        }
        m_Window.SetTitle(title);
    }
    else
    {
        m_Window.UnsetTitle();
    }
}
