﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>

#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include <nns/console/console_InputEventListener.h>
#include <nns/console/console_SimpleConsole.h>

class Logger
    : public nns::console::SimpleConsole
{
public:
    Logger() NN_NOEXCEPT
    {
        NoBlink();
    }
    void Printfln(const char* format, ...) NN_NOEXCEPT;
    void Puts(const char* str) NN_NOEXCEPT;
};

class Window
    : public nns::console::PadEventListenerBase
    , public nns::console::SimpleConsole
{
public:
    typedef void(*OptionHandler)(void*, size_t);
    typedef void(*OptionHandlerWithIndex)(int index, void*, size_t);
    struct Option
    {
        char label[128];
        bool withIndex;
        union
        {
            OptionHandler optionHandler;
            OptionHandlerWithIndex optionHandlerWithIndex;
        } u;
    };

    struct Components
    {
        char title[128];
        char description[128];

        void* optionData;
        size_t optionDataSize;
        Option options[32]; // とりあえず 32
        size_t optionCount; // 実際の個数

        void Reset(const char* _title, const char* _description) NN_NOEXCEPT;
        void Reset(const char* _title, const char* _description, void* data, size_t dataSize) NN_NOEXCEPT;
        template <typename T>
        void Reset(const char* _title, const char* _description, T* _optionArg) NN_NOEXCEPT
        {
            Reset(_title, _description, _optionArg, sizeof(*_optionArg));
        }

        void AddOption(const char* label, OptionHandler handler) NN_NOEXCEPT;
        void AddOption(const char* label, OptionHandlerWithIndex handler) NN_NOEXCEPT;
        void ClearOption() NN_NOEXCEPT;
    };

private:
    nn::os::Mutex m_Lock;

    char m_Title[256];
    const Components* m_pComponents;
    int m_Cursor;
    bool m_IsSelected;

    // 入力イベントのハンドラ
    virtual void OnButtonReleasedImpl(const nns::console::PadButtonSet& buttons) NN_NOEXCEPT final NN_OVERRIDE;

    void Flush() NN_NOEXCEPT;

public:
    Window() NN_NOEXCEPT
        : m_Lock(false)
        , m_pComponents(nullptr)
        , m_Cursor(0)
        , m_IsSelected(false)
    {
        NoBlink();

        UnsetTitle();
    }
    void SetComponents(const Components* pComponents) NN_NOEXCEPT;
    void SetTitle(const char* title) NN_NOEXCEPT
    {
        std::strncpy(m_Title, title, sizeof(m_Title));
    }
    void UnsetTitle() NN_NOEXCEPT
    {
        m_Title[0] = '\0';
    }
    void Update() NN_NOEXCEPT;

    void lock() NN_NOEXCEPT
    {
        m_Lock.Lock();
    }
    void unlock() NN_NOEXCEPT
    {
        m_Lock.Unlock();
    }
};

extern Logger g_Logger;
#define NNS_MIGRATION_LOG_LN(...) \
    do {\
        g_Logger.Printfln(__VA_ARGS__); \
    } while (NN_STATIC_CONDITION(false))

#define NNS_MIGRATION_DETAIL_DO(expression) \
    do { \
        ::nn::Result _r = (expression); \
        if (!_r.IsSuccess()) \
        { \
            NNS_MIGRATION_LOG_LN("<!> %s failed with %03d-%04d (%08lx)", #expression, _r.GetModule(), _r.GetDescription(), _r.GetInnerValueForDebug()); \
            NN_RESULT_THROW(_r); \
        } \
    } while (NN_STATIC_CONDITION(false))

#define NNS_MIGRATION_DETAIL_THROW_UNLESS(expression, result) \
    do { \
        bool _b = static_cast<bool>(expression); \
        if (!_b) \
        { \
            ::nn::Result _r = (result);\
            NNS_MIGRATION_LOG_LN("<!> %s failed, then throwing %03d-%04d (%08lx)", #expression, _r.GetModule(), _r.GetDescription(), _r.GetInnerValueForDebug()); \
            NN_RESULT_THROW(_r); \
        } \
    } while (NN_STATIC_CONDITION(false))

#define NNS_MIGRATION_DETAIL_THROW(result) \
    do { \
        ::nn::Result _r = (result);\
        NNS_MIGRATION_LOG_LN("<!> Throwing %03d-%04d (%08lx)", _r.GetModule(), _r.GetDescription(), _r.GetInnerValueForDebug()); \
        NN_RESULT_THROW(_r); \
    } while (NN_STATIC_CONDITION(false))
