﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "g3ddemo_ModelUtility.h"
#include "3dSampleViewer_Camera.h"

namespace g3ddemo = nn::g3d::demo;

enum ShaderArchiveType
{
    ShaderArchiveType_Town,
    ShaderArchiveType_Demo,
    ShaderArchiveType_VisualizeInfo,
    ShaderArchiveType_Count
};

enum ResFileType
{
    ResFileType_TownEnv,
    ResFileType_Count
};

enum DrawPassType
{
    DrawPassType_Material,
    DrawPassType_Visualize,
    DrawPassType_Count
};

enum ViewType
{
    ViewType_Camera,
    ViewType_Count
};

const int BufferingCount = 2;

g3ddemo::ResourceHolder* GetResourceHolder() NN_NOEXCEPT;
g3ddemo::CameraController* GetCameraController() NN_NOEXCEPT;
void SetProjectionMatrix(const nn::util::Matrix4x4fType& projectionMatrix) NN_NOEXCEPT;
void SetDefaultProjection() NN_NOEXCEPT;
const nn::g3d::ModelObj* GetSelectedModelObj() NN_NOEXCEPT;

class ScopedAllocator
{
    NN_DISALLOW_COPY(ScopedAllocator);
public:
    ScopedAllocator(size_t size, ptrdiff_t alignment) NN_NOEXCEPT
        : pBuffer(nullptr)
    {
        pBuffer = g3ddemo::AllocateMemory(size, alignment);
        NN_ASSERT_NOT_NULL(pBuffer);
    }

    ~ScopedAllocator() NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pBuffer);
        g3ddemo::FreeMemory(pBuffer);
    }

    void* GetAllocatedMemory()
    {
        return pBuffer;
    }

    template<typename T>
    T* GetAllocatedMemory()
    {
        return static_cast<T*>(pBuffer);
    }

private:
    void* pBuffer;
};

template<typename T>
class ScopedDynamicArray
{
    NN_DISALLOW_COPY(ScopedDynamicArray<T>);
public:
    explicit ScopedDynamicArray(int count) NN_NOEXCEPT
        : m_Allocator(sizeof(T) * count, NN_ALIGNOF(T))
        , m_Ptr(nullptr)
        , m_Count(count)
    {
        m_Ptr = m_Allocator.GetAllocatedMemory<T>();
        memset(m_Ptr, 0, sizeof(T) * count);
    }

    ~ScopedDynamicArray() NN_NOEXCEPT
    {
        m_Ptr = nullptr;
        m_Count = 0;
    }

    T& operator[] (int index) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_Count);
        return m_Ptr[index];
    }

    NN_EXPLICIT_OPERATOR T*() NN_NOEXCEPT
    {
        return m_Ptr;
    }

    NN_EXPLICIT_OPERATOR const T*() const NN_NOEXCEPT
    {
        return m_Ptr;
    }

    int GetCount() const NN_NOEXCEPT
    {
        return m_Count;
    }

private:
    ScopedAllocator m_Allocator;
    T* m_Ptr;
    int m_Count;
};
