﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/demo/pad/win/demo_KeyboardMouseDeviceWin.h>
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <windows.h>

namespace nw
{
namespace demo
{

void* KeyboardMouseDeviceWin::s_MessageHook = NULL;
bool KeyboardMouseDeviceWin::s_IsDoubleClick = false;
s32 KeyboardMouseDeviceWin::s_MouseWheel = 0;
KeyboardMouseDeviceWin* KeyboardMouseDeviceWin::s_pInstance = NULL;

//------------------------------------------------------------------------------
KeyboardMouseDeviceWin::KeyboardMouseDeviceWin()
: KeyboardMouseDevice(),
  m_CursorClientAreaPos( 0.f, 0.f ),
  m_CursorScreenPos( 0.f, 0.f ),
  m_CursorClientAreaPosCenterOrigin( 0.f, 0.f ),
  m_WheelDelta( 0 ),
  m_IsDoubleClick( false ),
  m_MainWindowHandle( NULL ),
  m_SubWindowHandle( NULL ),
  m_ActiveWindow( ACTIVE_WINDOW_NONE ),
  m_InitializedCount( 0 )
{
    if (s_pInstance != NULL)
    {
        NW_FATAL_ERROR("nw::demo::KeyboardMouseDeviceWin instance already exists.");
    }
    s_pInstance = this;
}

//------------------------------------------------------------------------------
void
KeyboardMouseDeviceWin::Initialize()
{
    if ( m_InitializedCount == 0 )
    {
        s_MessageHook = SetWindowsHookEx(
            WH_GETMESSAGE,
            (HOOKPROC)MessageHook,
            NULL,
            GetCurrentThreadId()
        );
    }

    ++m_InitializedCount;
}

//------------------------------------------------------------------------------
void
KeyboardMouseDeviceWin::Initialize( void* hWnd )
{
    if ( m_InitializedCount == 0 )
    {
        s_MessageHook = SetWindowsHookEx(
            WH_GETMESSAGE,
            (HOOKPROC)MessageHook,
            NULL,
            GetWindowThreadProcessId( reinterpret_cast<HWND>( hWnd ), NULL )
        );
    }

    ++m_InitializedCount;
}

//------------------------------------------------------------------------------
void
KeyboardMouseDeviceWin::Finalize()
{
    --m_InitializedCount;
    if ( m_InitializedCount < 0 )
    {
        m_InitializedCount = 0;
    }

    if ( m_InitializedCount == 0 )
    {
        UnhookWindowsHookEx( reinterpret_cast<HHOOK>( s_MessageHook ) );
    }
}

//------------------------------------------------------------------------------
void
KeyboardMouseDeviceWin::Update()
{
    // キーの状態を取得
    // GetAsyncKeyStateを使う

    m_Flags.SetMaskOn( MASK_KEY_ENABLE );
    ::ZeroMemory( m_VkeyTrig, V_KEY_MAX );
    ::ZeroMemory( m_VkeyRepeat, V_KEY_MAX );

    for ( int i = 0; i < V_KEY_MAX; ++i )
    {
        SHORT ret = GetAsyncKeyState( i );

        // 最下位ビットにトリガ入力またはオートリピートがあったか否かが入っている
        if ( ret & 0x0001 )
        {
            // 前のフレームのholdを見る
            if ( m_VkeyHold[ i ] == V_KEY_ON )
            {
                // holdだったらリピート
                m_VkeyRepeat[ i ] = V_KEY_ON;
            }
            else
            {
                // holdでなかったらトリガ
                m_VkeyTrig[ i ] = V_KEY_ON;
            }
        }

        // 最上位ビットに、押されているか否かが入っている
        m_VkeyHold[ i ] = ( (ret & 0x8000) != 0 ? static_cast<u8>(V_KEY_ON) : static_cast<u8>(V_KEY_OFF) );
    }

    // カーソル・ウィンドウの状態を取得
    {
        POINT point;

        m_Flags.SetMaskOff( MASK_CURSOR_ENABLE );
        m_Flags.SetMaskOff( MASK_CURSOR_ON_CLIENT_AREA );
        m_Flags.SetMaskOff( MASK_CLIENT_AREA_ENABLE );

        if ( GetCursorPos(&point) )
        {
            // カーソルの状態が取得できた
            m_Flags.SetMaskOn( MASK_CURSOR_ENABLE );
            // カーソルのスクリーン上の位置を記録
            m_CursorScreenPos.Set( static_cast<f32>(point.x), static_cast<f32>(point.y) );
        }

        // 現在アクティブなウィンドウを調査する
        HWND foregroundHWnd = GetForegroundWindow();

        if ( m_MainWindowHandle && reinterpret_cast<HWND>( m_MainWindowHandle ) == foregroundHWnd )
        {
            m_ActiveWindow = ACTIVE_WINDOW_MAIN;
        }
        else if ( m_SubWindowHandle && reinterpret_cast<HWND>( m_SubWindowHandle ) == foregroundHWnd )
        {
            m_ActiveWindow = ACTIVE_WINDOW_SUB;
        }
        else
        {
            m_ActiveWindow = ACTIVE_WINDOW_NONE;
        }

        if ( m_ActiveWindow != ACTIVE_WINDOW_NONE )
        {
            if ( IsCursorEnable() )
            {
                POINT clientPoint = point;
                if ( ScreenToClient( foregroundHWnd, &clientPoint ) )
                {
                    RECT rect;

                    // クライアント領域での座標が取得できた
                    m_Flags.SetMaskOn( MASK_CLIENT_AREA_ENABLE );

                    // カーソルがクライアント領域の上にあるか否かを取得
                    if ( GetClientRect(foregroundHWnd, &rect) )
                    {
                        if ( 0 <= clientPoint.x && clientPoint.x <= rect.right &&
                             0 <= clientPoint.y && clientPoint.y <= rect.bottom )
                        {
                            m_Flags.SetMaskOn( MASK_CURSOR_ON_CLIENT_AREA );
                        }
                    }

                    {
                        f32 clientX = static_cast<f32>( clientPoint.x );
                        f32 clientY = static_cast<f32>( clientPoint.y );

                        // カーソルのクライアント座標での位置を記録
                        m_CursorClientAreaPos.Set( clientX, clientY );
                        // 中心を原点としたクライアント座標での位置を記録
                        m_CursorClientAreaPosCenterOrigin.Set( (clientX / rect.right) * 2.f - 1.f, (clientY / rect.bottom) * 2.f - 1.f );
                    }
                }
            }

            // ダブルクリックフラグの取得
            m_IsDoubleClick = s_IsDoubleClick;
            s_IsDoubleClick = false;

            // マウスホイールの回転差分の取得
            m_WheelDelta = s_MouseWheel;
            s_MouseWheel = 0;
        }
    }
}

//------------------------------------------------------------------------------
LRESULT CALLBACK
KeyboardMouseDeviceWin::MessageHook(UINT nCode, WPARAM wParam, LPARAM lParam)
{
    if (nCode < 0)
    {
        return CallNextHookEx(reinterpret_cast<HHOOK>( s_MessageHook ), nCode, wParam, lParam);
    }

    MSG* msg = reinterpret_cast<MSG*>( lParam );

    if ( msg != NULL )
    {
        switch ( msg->message )
        {
        case WM_LBUTTONDBLCLK:
        case WM_RBUTTONDBLCLK:
        case WM_MBUTTONDBLCLK:
        case WM_XBUTTONDBLCLK:
            s_IsDoubleClick = true;
            break;

        case WM_LBUTTONUP:
        case WM_RBUTTONUP:
        case WM_MBUTTONUP:
        case WM_XBUTTONUP:
            break;

        case WM_MOUSEWHEEL:
            s_MouseWheel = GET_WHEEL_DELTA_WPARAM(msg->wParam);
            break;
        }
    }

    return CallNextHookEx(reinterpret_cast<HHOOK>( s_MessageHook ), nCode, wParam, lParam);
}

} // namespace demo
} // namespace nw

