﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/demo/pad/demo_PadBase.h>

namespace nw
{
namespace demo
{

const f32 PadBase::STICK_HOLD_THRESHOLD_DEFAULT = 0.5f;
const f32 PadBase::STICK_RELEASE_THRESHOLD_DEFAULT = 0.25f;

//------------------------------------------------------------------------------
PadBase::PadBase(int padBitMax, int leftStickCrossStartBit, int rightStickCrossStartBit, int touchKeyBit)
 : m_LeftStickHoldThreshold( STICK_HOLD_THRESHOLD_DEFAULT ),
   m_RightStickHoldThreshold( STICK_HOLD_THRESHOLD_DEFAULT ),
   m_LeftStickReleaseThreshold( STICK_RELEASE_THRESHOLD_DEFAULT ),
   m_RightStickReleaseThreshold( STICK_RELEASE_THRESHOLD_DEFAULT ),
   m_PadBitMax( padBitMax ),
   m_LeftStickCrossStartBit( leftStickCrossStartBit ),
   m_RightStickCrossStartBit( rightStickCrossStartBit ),
   m_TouchKeyBit( touchKeyBit ),
   m_Pointer( 0.f, 0.f ),
   m_PointerPrev( 0.f, 0.f ),
   m_PointerBoundMin( 0.f, 0.f ),
   m_PointerBoundMax( 0.f, 0.f ),
   m_LeftStick( 0.f, 0.f ),
   m_RightStick( 0.f, 0.f ),
   m_LeftAnalogTrigger( 0.f ),
   m_RightAnalogTrigger( 0.f )
{
    if (PAD_IDX_MAX_BASE < padBitMax)
    {
        NW_ERR("illegal padBitMax[%d]", padBitMax);
        m_PadBitMax = PAD_IDX_MAX_BASE;
    }
}

//------------------------------------------------------------------------------
void
PadBase::SetPointerWithBound(bool isOn, bool touchkeyHold, const nw::math::VEC2& pos)
{
    if ( isOn )
    {
        if ( !( m_PointerBoundMin.x >= m_PointerBoundMax.x || m_PointerBoundMin.y >= m_PointerBoundMax.y ) )
        {
            if ( m_PointerBoundMin.x <= pos.x && pos.x <= m_PointerBoundMax.x &&
                 m_PointerBoundMin.y <= pos.y && pos.y <= m_PointerBoundMax.y )
            {
                m_Pointer = pos;
            }
            else
            {
                // バウンドボックスの範囲から外れていたらoffにする
                isOn = false;
            }
        }
        else
        {
            m_Pointer = pos;
        }
    }

    m_PointerFlag.ChangeMask( POINTER_ON, isOn );

    if (m_TouchKeyBit >= 0)
    {
        m_PadHold.ChangeBit( m_TouchKeyBit, isOn && touchkeyHold );
    }
}

//------------------------------------------------------------------------------
void
PadBase::UpdateDerivativeParams(u32 prevHold, bool prevPointerOn)
{
    // 左アナログスティックを十字キーとみなしたときの値を m_PadHold に設定します。
    {
        u32 stick_hold = 0;

        if (m_LeftStickCrossStartBit >= 0)
        {
            stick_hold |= GetStickHold( prevHold, m_LeftStick, m_LeftStickHoldThreshold, m_LeftStickReleaseThreshold, m_LeftStickCrossStartBit );
        }

        if (m_RightStickCrossStartBit >= 0)
        {
            stick_hold |= GetStickHold( prevHold, m_RightStick, m_RightStickHoldThreshold, m_RightStickReleaseThreshold, m_RightStickCrossStartBit );
        }

        m_PadHold.SetDirect( (m_PadHold.GetDirect() & ~(CreateStickCrossMask())) | stick_hold );
    }

    // Hold 状態値から他の状態情報を更新します。
    m_PadProcessor.Update( m_PadHold.GetDirect() );

    // ポインタの状態を更新します。
    m_PointerFlag.ChangeMask( POINTER_ON_NOW, !prevPointerOn && m_PointerFlag.IsMaskOn( POINTER_ON ) );
    m_PointerFlag.ChangeMask( POINTER_OFF_NOW, prevPointerOn && m_PointerFlag.IsMaskOff( POINTER_ON ) );

    if ( this->IsPointerOnNow() )
    {
        // ポインタが画面外から画面内に入ったときには
        // 前フレームのポインタ位置を現ポインタ位置にリセットします。
        m_PointerPrev = m_Pointer;
    }
}

//------------------------------------------------------------------------------
void
PadBase::SetLeftStickCrossThreshold(f32 hold, f32 release)
{
    if ( hold >= release )
    {
        m_LeftStickHoldThreshold = hold;
        m_LeftStickReleaseThreshold = release;
    }
    else
    {
        NW_ERR( "hold[%f] must be larger than or equal to release[%f].", hold, release );
    }
}

//------------------------------------------------------------------------------
void
PadBase::SetRightStickCrossThreshold(f32 hold, f32 release)
{
    if (hold >= release)
    {
        m_RightStickHoldThreshold = hold;
        m_RightStickReleaseThreshold = release;
    }
    else
    {
        NW_ERR( "hold[%f] must be larger than or equal to release[%f].", hold, release );
    }
}

//------------------------------------------------------------------------------
u32
PadBase::GetStickHold(u32 prevHold, const nw::math::VEC2& stick, float holdThreshold, float releaseThreshold, int startBit)
{
    float length = stick.Length();

    if ( length < releaseThreshold ||
        ( length < holdThreshold &&
         ( prevHold & ((1 << (startBit + CROSS_UP)) | (1 << (startBit + CROSS_DOWN)) | (1 << (startBit + CROSS_LEFT)) | (1 << (startBit + CROSS_RIGHT)))) == 0 )
       )
    {
        // 何れの方向にも押されていない状態
        // length が releaseThreshold より小さい場合は、常に押されていない状態になる。
        // length が releaseThreshold より大きく、 holdThreshold より小さい場合は、前のフレームに入力がなかったら押されていない状態になる
        // それ以外は押された状態なのでelse節に行く
        return 0;
    }
    else
    {
        // スティックが倒されている方向を求める
        u32 angle = nw::math::Atan2Idx( stick.y, stick.x );

        if (angle < 0x10000000)
        {
            return (1 << (startBit + CROSS_RIGHT));
        }
        else if (angle < 0x30000000)
        {
            return ((1 << (startBit + CROSS_RIGHT)) | (1 << (startBit + CROSS_UP)));
        }
        else if (angle < 0x50000000)
        {
            return (1 << (startBit + CROSS_UP));
        }
        else if (angle < 0x70000000)
        {
            return ((1 << (startBit + CROSS_LEFT)) | (1 << (startBit + CROSS_UP)));
        }
        else if (angle < 0x90000000)
        {
            return (1 << (startBit + CROSS_LEFT));
        }
        else if (angle < 0xb0000000)
        {
            return ((1 << (startBit + CROSS_LEFT)) | (1 << (startBit + CROSS_DOWN)));
        }
        else if (angle < 0xd0000000)
        {
            return (1 << (startBit + CROSS_DOWN));
        }
        else if (angle < 0xf0000000)
        {
            return ((1 << (startBit + CROSS_RIGHT)) | (1 << (startBit + CROSS_DOWN)));
        }
        else
        {
            return (1 << (startBit + CROSS_RIGHT));
        }
    }
}

//------------------------------------------------------------------------------
u32
PadBase::CreateStickCrossMask()
{
    nw::ut::BitFlag32 mask;

    if (m_LeftStickCrossStartBit >= 0)
    {
        mask.SetBitOn(m_LeftStickCrossStartBit + CROSS_UP);
        mask.SetBitOn(m_LeftStickCrossStartBit + CROSS_DOWN);
        mask.SetBitOn(m_LeftStickCrossStartBit + CROSS_LEFT);
        mask.SetBitOn(m_LeftStickCrossStartBit + CROSS_RIGHT);
    }

    if (m_RightStickCrossStartBit >= 0)
    {
        mask.SetBitOn(m_RightStickCrossStartBit + CROSS_UP);
        mask.SetBitOn(m_RightStickCrossStartBit + CROSS_DOWN);
        mask.SetBitOn(m_RightStickCrossStartBit + CROSS_LEFT);
        mask.SetBitOn(m_RightStickCrossStartBit + CROSS_RIGHT);
    }

    return mask.GetDirect();
}

//------------------------------------------------------------------------------
bool
PadBase::IsIdle()
{
    return ( GetHoldMask() == 0 &&
             GetLeftStick().IsZero() && GetRightStick().IsZero() &&
             GetLeftAnalogTrigger() == 0.f && GetRightAnalogTrigger() == 0.f
           );
}

//------------------------------------------------------------------------------
void
PadBase::SetIdle()
{
    m_PadHold.SetAllZero();
    m_PadProcessor.Reset();

    m_LeftStick.Set( 0.f, 0.f );
    m_RightStick.Set( 0.f, 0.f );
    m_LeftAnalogTrigger = 0.f;
    m_RightAnalogTrigger = 0.f;
}

} // namespace demo
} // namespace nw

