﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/demo/cafe/demo_PerformanceMonitorCafe.h>
#include <nw/dev/dev_PrimitiveRenderer.h>

namespace nw
{
namespace demo
{

//------------------------------------------------------------------------------
/* ctor */
PerformanceMonitorCafe::PerformanceMonitorCafe( nw::dev::DevTextWriter* textWriter )
 : m_CurrentBuffer( 0 ),
   m_Count( 0 ),
   m_OverNum( 0 ),
   m_PosX( 10.f ),
   m_PosY( 10.f ),
   m_CellMinWidth( 80.f ),
   m_CellMargin( 2.f ),
   m_TextColor( nw::ut::Color4u8::WHITE ),
   m_LineColor( 255, 255, 255, 128 ),
   m_TextWriter( textWriter ),
   m_ProjMtx( NULL ),
   m_ViewMtx( NULL ),
   m_IsEnabled( true )
{
    m_SectionArray[0].Wrap( &m_SectionBuffer[0], MAX_SECTION );
    m_SectionArray[1].Wrap( &m_SectionBuffer[MAX_SECTION], MAX_SECTION );
    m_CountersNum[0] = 0;
    m_CountersNum[1] = 0;

    SetGroup(
        MMCR0_PMC1_L2_HIT,    "L2_HIT",
        MMCR0_PMC2_L2_I_MISS, "L2_I_MISS",
        MMCR1_PMC3_L2_D_MISS, "L2_D_MISS",
        MMCR1_PMC4_HOLD,      "" );
}

//------------------------------------------------------------------------------
void
PerformanceMonitorCafe::SetGroup( u32 e1, const char* label1, u32 e2, const char* label2, u32 e3, const char* label3, u32 e4, const char* label4 )
{
    if ( ! m_IsEnabled )
    {
        return;
    }

    // 計測途中で SetGroup をすることはできません。
    NW_ASSERT( m_CountersNum[m_CurrentBuffer] == 0 );
    NW_ASSERT( m_Count == 0 );
    NW_ASSERT( m_OverNum == 0 );
#if defined(NW_COMPILER_GHS)
    PMCPUSetGroup( &m_CurrentSection.counters, e1, e2, e3, e4 );
#endif
    m_Label[0] = nw::ut::SafeString(label1);
    m_Label[1] = nw::ut::SafeString(label2);
    m_Label[2] = nw::ut::SafeString(label3);
    m_Label[3] = nw::ut::SafeString(label4);
}

//------------------------------------------------------------------------------
void
PerformanceMonitorCafe::BeginMeasure( const char* name )
{
    if ( ! m_IsEnabled )
    {
        return;
    }

    if ( m_SectionArray[m_CurrentBuffer].size() <= m_CountersNum[m_CurrentBuffer] )
    {
        ++m_OverNum;
        return;
    }

    Section* section = &m_SectionArray[m_CurrentBuffer].at( m_CountersNum[m_CurrentBuffer] );
    section->counters = m_CurrentSection.counters;
    section->labelSection = nw::ut::SafeString(name);
#if defined(NW_COMPILER_GHS)
    PMCPUResetStartGroup( &section->counters );
#endif

    ++m_Count;
}

//------------------------------------------------------------------------------
void
PerformanceMonitorCafe::EndMeasure()
{
    if ( ! m_IsEnabled )
    {
        return;
    }

    if ( m_OverNum > 0 )
    {
        --m_OverNum;
        return;
    }

    --m_Count;

    Section* section = &m_SectionArray[m_CurrentBuffer].at( m_CountersNum[m_CurrentBuffer] );
#if defined(NW_COMPILER_GHS)
    PMCPUReadGroup( &section->counters );
    PMCPUStopGroup( &section->counters );
#endif

    ++m_CountersNum[m_CurrentBuffer];
}

//------------------------------------------------------------------------------
void
PerformanceMonitorCafe::DrawResult()
{
    if ( ! m_IsEnabled )
    {
        return;
    }

    NW_ASSERT_NOT_NULL( m_TextWriter );
    NW_ASSERT_NOT_NULL( m_ProjMtx );
    NW_ASSERT_NOT_NULL( m_ViewMtx );

    nw::dev::PrimitiveRenderer* renderer = nw::dev::PrimitiveRenderer::GetInstance();
    renderer->SetProjectionMtx( m_ProjMtx );
    renderer->SetViewMtx( m_ViewMtx );
    renderer->SetModelMatrix( nw::math::MTX34::Identity() );

    m_TextWriter->SetFontSize( 16.f );
    m_TextWriter->SetTextColor( m_TextColor );

    f32 cellWidth[4];
    f32 widthSection = 0.f;

    // 各イベント名を描画するの必要な幅を計算します。
    for ( int col = 0; col < 4; ++col )
    {
        cellWidth[col] = m_TextWriter->GetTextWriter().CalcFormatStringWidth( "%s", m_Label[col].c_str() );
        if ( cellWidth[col] < m_CellMinWidth && cellWidth[col] > 0.f )
        {
            cellWidth[col] = m_CellMinWidth;
        }
    }

    // 区間名を描画するのに必要な幅を計算し、最大値を取得します。
    for ( int line = 0; line < m_CountersNum[ 1 - m_CurrentBuffer ]; ++line )
    {
        Section* section = &m_SectionArray[ 1 - m_CurrentBuffer ].at( line );

        f32 width = m_TextWriter->GetTextWriter().CalcFormatStringWidth( "%s", section->labelSection.c_str() );
        if ( widthSection < width )
        {
            widthSection = width;
        }
    }

    // イベント名を描画します。
    {
        f32 currentPosX = m_PosX + widthSection + m_CellMargin;
        m_TextWriter->SetCursor( currentPosX, m_PosY );

        for ( int col = 0; col < 4; ++col )
        {
            if ( m_Label[col].calcLength() == 0 )
            {
                continue;
            }

            f32 left = cellWidth[col] - m_TextWriter->GetTextWriter().CalcFormatStringWidth( "%s", m_Label[col].c_str() );
            m_TextWriter->SetCursorX( currentPosX + left );
            m_TextWriter->Printf( "%s", m_Label[col].c_str() );
            currentPosX += ( cellWidth[ col ] + m_CellMargin );
        }
    }

    for ( int line = 0; line < m_CountersNum[ 1 - m_CurrentBuffer ]; ++line )
    {
        Section* section = &m_SectionArray[ 1 - m_CurrentBuffer ].at( line );

        // 区間名を描画します。
        m_TextWriter->SetCursor( m_PosX, m_PosY + ( ( m_TextWriter->GetTextWriter().GetLineHeight() + m_CellMargin ) * ( line + 1 ) ) );
        m_TextWriter->Printf( "%s", section->labelSection.c_str() );

        f32 currentPosX = m_PosX + widthSection + m_CellMargin;
        u32 count[4] = { section->counters.pmc1, section->counters.pmc2, section->counters.pmc3, section->counters.pmc4 };

        // 各結果を描画します。
        for ( int col = 0; col < 4; ++col )
        {
            if ( m_Label[col].calcLength() == 0 )
            {
                continue;
            }

            renderer->Begin();
            {
                renderer->DrawLine(
                    nw::math::VEC3( currentPosX, m_TextWriter->GetCursorY(), 0.f ),
                    nw::math::VEC3( currentPosX + cellWidth[col], m_TextWriter->GetCursorY(), 0.f ),
                    m_LineColor
                    );
                renderer->DrawLine(
                    nw::math::VEC3( currentPosX, m_TextWriter->GetCursorY(), 0.f ),
                    nw::math::VEC3( currentPosX, m_TextWriter->GetCursorY() + m_TextWriter->GetTextWriter().GetLineHeight(), 0.f ),
                    m_LineColor
                    );
            }
            renderer->End();

            f32 left = cellWidth[col] - m_TextWriter->GetTextWriter().CalcFormatStringWidth( "%u",  count[col] );
            m_TextWriter->SetCursorX( currentPosX + left );
            m_TextWriter->Printf( "%u", count[col] );
            currentPosX += ( cellWidth[col] + m_CellMargin );
        }
    }

    m_TextWriter->Flush();
}

//------------------------------------------------------------------------------
void
PerformanceMonitorCafe::EndFrame()
{
    if ( m_IsEnabled )
    {
        NW_ASSERT( m_Count == 0 );
        NW_ASSERT( m_OverNum == 0 );

        m_CurrentBuffer = 1 - m_CurrentBuffer;
        m_CountersNum[m_CurrentBuffer] = 0;
        m_Count = 0;
        m_OverNum = 0;
    }
}

} // namespace nw::demo
} // namespace nw
