﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEMO_KEYBOARD_MOUSE_DEVICE_WIN_H_
#define NW_DEMO_KEYBOARD_MOUSE_DEVICE_WIN_H_

#include <nw/demo/pad/demo_KeyboardMouseDevice.h>
#include <nw/ut/ut_BitFlag.h>
#include <nw/math.h>

namespace nw
{
namespace demo
{

//---------------------------------------------------------------------------
//! @brief   PC 用キーボード・マウスデバイス情報を取得するクラスです。
//!
//! @details :category     入力デバイス
//---------------------------------------------------------------------------
class KeyboardMouseDeviceWin : public KeyboardMouseDevice
{
public:
    //! @brief アクティブなウィンドウを示す列挙型です。
    enum ActiveWindow
    {
        ACTIVE_WINDOW_NONE, //!<  このアプリケーションのウィンドウはアクティブではありません。
        ACTIVE_WINDOW_MAIN, //!<  メインウィンドウがアクティブです。
        ACTIVE_WINDOW_SUB   //!<  サブウィンドウがアクティブです。
    };


    //---------------------------------------------------------------------------
    //! @brief        インスタンスを取得します。
    //!
    //! @return       シングルトンのインスタンスを返します。
    //---------------------------------------------------------------------------
    static KeyboardMouseDeviceWin* GetInstance()
    {
        static KeyboardMouseDeviceWin s_Instance;
        return &s_Instance;
    }


    //---------------------------------------------------------------------------
    //! @brief        初期化処理です。
    //---------------------------------------------------------------------------
    void Initialize();

    //---------------------------------------------------------------------------
    //! @brief        初期化処理です。
    //!
    //! @param[in]    hWnd      メッセージフックを行うウィンドウのハンドルを指定します。
    //---------------------------------------------------------------------------
    void Initialize( void* hWnd );

    //---------------------------------------------------------------------------
    //! @brief        終了処理です。
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief        デバイスを更新します。
    //---------------------------------------------------------------------------
    virtual void Update();


    //---------------------------------------------------------------------------
    //! @brief        このフレームにおいてキーの状態を取得できたかを返します。
    //!
    //! @return       キーの状態を取得できた場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsKeyEnable() const { return m_Flags.IsMaskOn( MASK_KEY_ENABLE ); }

    //---------------------------------------------------------------------------
    //! @brief        このフレームにおいてカーソルの状態を取得できたかを返します。
    //!
    //! @return       カーソルの状態を取得できた場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsCursorEnable() const { return m_Flags.IsMaskOn( MASK_CURSOR_ENABLE ); }


    //---------------------------------------------------------------------------
    //! @brief        アクティブなウィンドウを返します。
    //!
    //! @return       アクティブなウィンドウを示す列挙型を返します。
    //---------------------------------------------------------------------------
    ActiveWindow GetActiveWindow() const { return m_ActiveWindow; }

    //---------------------------------------------------------------------------
    //! @brief        アクティブなウィンドウのクライアント領域が取得できたかを返します。
    //!
    //! @return       クライアント領域を取得できた場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsClientAreaEnable() const { return m_Flags.IsMaskOn( MASK_CLIENT_AREA_ENABLE ); }

    //---------------------------------------------------------------------------
    //! @brief        カーソルがクライアント領域の上にあるかを返します。
    //!
    //! @return       カーソルがクライアント領域の上にある場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsCursorOnClientArea() const { return m_Flags.IsMaskOn( MASK_CURSOR_ON_CLIENT_AREA ); }

    //---------------------------------------------------------------------------
    //! @brief        カーソルの位置をアクティブなウィンドウのクライアント領域の相対位置で取得します。
    //!
    //! @return       クライアント領域の相対位置でのカーソルの位置を返します。
    //---------------------------------------------------------------------------
    const nw::math::VEC2& GetCursorClientAreaPos() const { return m_CursorClientAreaPos; }

    //---------------------------------------------------------------------------
    //! @brief        カーソルの位置をアクティブなウィンドウのクライアント領域の相対位置
    //!               （ただし原点中心としたもの）で取得します。
    //!
    //! @return       クライアント領域の相対位置でのカーソルの位置を返します。
    //---------------------------------------------------------------------------
    const nw::math::VEC2& GetCursorClientAreaPosCenterOrigin() const { return m_CursorClientAreaPosCenterOrigin; }

    //---------------------------------------------------------------------------
    //! @brief        カーソルの位置をスクリーン上の絶対位置で取得します。
    //!
    //! @return       スクリーン上の絶対位置でのカーソルの位置を返します。
    //---------------------------------------------------------------------------
    const nw::math::VEC2& GetCursorScreenPos() const { return m_CursorScreenPos; }

    //---------------------------------------------------------------------------
    //! @brief        ダブルクリックの有無を取得します。
    //!
    //! @return       ダブルクリックの有無を返します。
    //---------------------------------------------------------------------------
    const bool IsDoubleClick() const { return m_IsDoubleClick; }

    //---------------------------------------------------------------------------
    //! @brief        マウスホイールの回転差分を取得します。
    //!
    //! @return       マウスホイールの回転差分を返します。
    //---------------------------------------------------------------------------
    const s32 GetWheel() const { return m_WheelDelta; }


    //---------------------------------------------------------------------------
    //! @brief        操作するメインウィンドウのハンドルを設定します。
    //!
    //! @param[in]    hWnd      メインウィンドウのハンドルです。
    //---------------------------------------------------------------------------
    void SetMainWindowHandle( void* hWnd ) { m_MainWindowHandle = hWnd; }

    //---------------------------------------------------------------------------
    //! @brief        メインウィンドウのハンドルを取得します。
    //!
    //! @return       メインウィンドウのハンドルを返します。
    //---------------------------------------------------------------------------
    void* GetMainWindowHandle() const { return m_MainWindowHandle; }

    //---------------------------------------------------------------------------
    //! @brief        操作するサブウィンドウのハンドルを設定します。
    //!
    //! @param[in]    hWnd      サブウィンドウのハンドルです。
    //---------------------------------------------------------------------------
    void SetSubWindowHandle( void* hWnd ) { m_SubWindowHandle = hWnd; }

    //---------------------------------------------------------------------------
    //! @brief        サブウィンドウのハンドルを取得します。
    //!
    //! @return       サブウィンドウのハンドルを返します。
    //---------------------------------------------------------------------------
    void* GetSubWindowHandle() const { return m_SubWindowHandle; }


protected:
    enum FlagMask
    {
        MASK_KEY_ENABLE            = 1 << 0,    //!< このフレーム、キーの状態が取得できたかを表すフラグ値です。
        MASK_CURSOR_ENABLE         = 1 << 1,    //!< このフレーム、カーソルの状態が取得できたかを表すフラグ値です。
        MASK_CURSOR_ON_CLIENT_AREA = 1 << 2,    //!< カーソルがこのアプリケーションのクライアント領域の上にあるかを表すフラグ値です。
        MASK_CLIENT_AREA_ENABLE    = 1 << 3     //!< このアプリケーションのクライアント領域が取得できたかを表すフラグ値です。
    };


    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    KeyboardMouseDeviceWin();

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~KeyboardMouseDeviceWin() {}


    //! @brief メッセージフックプロシージャです。
    static _W64 long __stdcall MessageHook(unsigned int nCode, _W64 unsigned int wParam, _W64 long lParam);

    nw::math::VEC2    m_CursorClientAreaPos;                //!< カーソルのウィンドウ内の相対位置です。
    nw::math::VEC2    m_CursorClientAreaPosCenterOrigin;    //!< カーソルのウィンドウ内の相対位置（中心原点）です。
    nw::math::VEC2    m_CursorScreenPos;                    //!< カーソルの画面内の絶対位置です。
    bool              m_IsDoubleClick;                      //!< ダブルクリックフラグです。
    s32               m_WheelDelta;                         //!< マウスホイールの回転差分です。

    nw::ut::BitFlag32 m_Flags;                              //!< 各種フラグです。
    void*             m_MainWindowHandle;                   //!< ウィンドウがアクティブかを取得する際のウィンドウハンドル（メイン用）です。
    void*             m_SubWindowHandle;                    //!< ウィンドウがアクティブかを取得する際のウィンドウハンドル（サブ用）です。
    ActiveWindow      m_ActiveWindow;                       //!< 現在このアプリケーションでアクティブなウィンドウです。
    s32               m_InitializedCount;                   //!< 初期化カウンタです。

    static void*      s_MessageHook;                        //!< メッセージフックプロシージャのハンドルです。
    static bool       s_IsDoubleClick;                      //!< メッセージフックプロシージャで取得するダブルクリックフラグです。
    static s32        s_MouseWheel;                         //!< メッセージフックプロシージャで取得するマウスホイールの回転差分です。

    static KeyboardMouseDeviceWin* s_pInstance;             //!< シングルトンオブジェクトへのポインタです。
};

} // namespace demo
} // namespace nw

#endif // NW_DEMO_KEYBOARD_MOUSE_DEVICE_WIN_H_
