﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEMO_KEYBOARD_MOUSE_DEVICE_H_
#define NW_DEMO_KEYBOARD_MOUSE_DEVICE_H_

#include <nw/types.h>

namespace nw
{
namespace demo
{

//---------------------------------------------------------------------------
//! @brief        PC キーボード・マウスデバイス管理の基底クラスです。
//!
//! @details :category     入力デバイス
//---------------------------------------------------------------------------
class KeyboardMouseDevice
{
public:
    enum
    {
        V_KEY_MAX = 256 //!< 仮想キーコードの最大値です。
    };


    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    KeyboardMouseDevice()
    {
        for ( u32 i = 0; i < V_KEY_MAX; ++i )
        {
            m_VkeyHold[ i ] = 0;
            m_VkeyTrig[ i ] = 0;
            m_VkeyRepeat[ i ] = 0;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~KeyboardMouseDevice() {}


    //---------------------------------------------------------------------------
    //! @brief        デバイスを更新します。
    //---------------------------------------------------------------------------
    virtual void Update() = 0;


    //---------------------------------------------------------------------------
    //! @brief        指定した仮想キーコードのキーが押されているかを取得します。
    //!
    //! @param[in]    vkey      仮想キーコードです。
    //!
    //! @return       押されている場合、 true を返します。
    //---------------------------------------------------------------------------
    virtual bool IsVkeyHold(int vkey) const { return ( m_VkeyHold[ vkey ] == V_KEY_ON ? true : false ); }

    //---------------------------------------------------------------------------
    //! @brief        指定した仮想キーコードのキーがトリガ入力されたかを取得します。
    //!
    //! @param[in]    vkey      仮想キーコードです。
    //!
    //! @return       トリガ入力された場合、 true を返します。
    //---------------------------------------------------------------------------
    virtual bool IsVkeyTrig(int vkey) const { return ( m_VkeyTrig[ vkey ] == V_KEY_ON ? true : false ); }

    //---------------------------------------------------------------------------
    //! @brief        指定した仮想キーコードのキーでオートリピートが発生したかを取得します。
    //!
    //! @param[in]    vkey      仮想キーコードです。
    //!
    //! @return       オートリピートが発生した場合、 true を返します。
    //---------------------------------------------------------------------------
    virtual bool IsVkeyRepeat(int vkey) const { return ( m_VkeyRepeat[ vkey ] == V_KEY_ON ? true : false ); }

    //---------------------------------------------------------------------------
    //! @brief        指定した仮想キーコードのキーがトリガ入力された、もしくはオートキーリピートが
    //!               発生したかを取得します。
    //!
    //! @param[in]    vkey      仮想キーコードです。
    //!
    //! @return       トリガ入力された、もしくはオートキーリピートがあった場合、 true を返します。
    //---------------------------------------------------------------------------
    virtual bool IsVkeyTrigWithRepeat(int vkey) const { return ( IsVkeyTrig(vkey) || IsVkeyRepeat(vkey) ); }


protected:
    //! @brief 仮想キー用のフラグ値です。
    enum VKeyFlag
    {
        V_KEY_OFF = 0,
        V_KEY_ON  = 1
    };

    u8 m_VkeyHold[ V_KEY_MAX ];      //!< 仮想キーのホールド値です。
    u8 m_VkeyTrig[ V_KEY_MAX ];      //!< 仮想キーのトリガ値です。
    u8 m_VkeyRepeat[ V_KEY_MAX ];    //!< 仮想キーのリピート値です。
};

} // namespace demo
} // namespace nw

#endif // NW_DEMO_KEYBOARD_MOUSE_DEVICE_H_
