﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEMO_FRAME_BUFFER_GL_H_
#define NW_DEMO_FRAME_BUFFER_GL_H_

#include <nw/demo/demo_FrameBuffer.h>
#include <nw/demo/gl/demo_FrameBufferUtilGL.h>
#include <nw/gfnd/gfnd_GraphicsContext.h>
#include <nw/gfnd/gl/gfnd_TextureGL.h>
#include <nw/dev/dev_PrimitiveRenderer.h>
#include <gl/glew.h>

namespace nw
{
namespace demo
{

//---------------------------------------------------------------------------
//! @brief        GL 上でのフレームバッファを表すクラスです。
//---------------------------------------------------------------------------
class FrameBufferGL : public FrameBuffer
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    fboId     結びつけるフレームバッファオブジェクトの ID です。
    //! @param[in]    size      フレームバッファサイズです。
    //---------------------------------------------------------------------------
    FrameBufferGL( GLuint fboId, const nw::math::VEC2& size )
      : FrameBuffer( size ),
        m_FboId( fboId )
    {}

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~FrameBufferGL() {}


    //---------------------------------------------------------------------------
    //! @brief        フレームバッファオブジェクトの結び付けを変更します。
    //!
    //! @param[in]    fboId     フレームバッファオブジェクトの ID です。
    //---------------------------------------------------------------------------
    void SetId( GLuint fboId ){ m_FboId = fboId; }

    //---------------------------------------------------------------------------
    //! @brief        結び付けられているフレームバッファオブジェクトを取得します。
    //!
    //! @return       フレームバッファオブジェクトの ID を返します。
    //---------------------------------------------------------------------------
    GLuint GetId() const { return m_FboId; }


protected:
    //! @brief 結び付けられたフレームバッファオブジェクトをバインドします。
    virtual void BindImpl() const
    {
        glBindFramebufferEXT( GL_FRAMEBUFFER_EXT, m_FboId );
        NW_GL_ASSERT();
    }

private:
    GLuint  m_FboId;
};



//---------------------------------------------------------------------------
//! @brief       ディスプレイバッファの GL 実装クラスです。
//---------------------------------------------------------------------------
class DisplayBufferGL : public DisplayBuffer
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    DisplayBufferGL()
      : DisplayBuffer(),
        m_Rotate( DISPLAY_ROTATE_NONE )
    {}

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~DisplayBufferGL() {}


    //---------------------------------------------------------------------------
    //! @brief        実際にディスプレイへ転送します。
    //---------------------------------------------------------------------------
    virtual void CopyToDisplay()
    {
        CopyToFrameBuffer( &m_TextureGL, 0, m_Width, m_Height, m_Rotate );
    }

    //---------------------------------------------------------------------------
    //! @brief        ディスプレイ転送に用いているテクスチャを取得します。
    //!
    //! @return       テクスチャを返します。
    //---------------------------------------------------------------------------
    const nw::gfnd::Texture* GetTexture() const { return & m_TextureGL; }

    //---------------------------------------------------------------------------
    //! @brief        フレームバッファオブジェクトを取得します。
    //!
    //! @return       フレームバッファオブジェクトを返します。
    //---------------------------------------------------------------------------
    const GLuint   GetHandle()  const { return m_FBO; }

    //---------------------------------------------------------------------------
    //! @brief        ディスプレイの回転を指定します。
    //!
    //! @param [in] rotate 回転方向です。
    //---------------------------------------------------------------------------
    void SetRotate(DisplayRotateState rotate)
    {
        m_Rotate = rotate;
    }

protected:
    virtual void InitializeImpl()
    {
        const float width = m_Width;
        const float height = m_Height;

        m_TextureGL.CreateTextureImage2D(
            0,
            GL_RGBA8,
            static_cast<GLsizei>( width ),
            static_cast<GLsizei>( height ),
            0,
            GL_RGBA,
            GL_UNSIGNED_BYTE,
            NULL
        );

        glGenFramebuffersEXT( 1, &m_FBO );
        NW_ASSERT( m_FBO != 0 );
        glBindFramebufferEXT( GL_FRAMEBUFFER_EXT, m_FBO );
        glFramebufferTexture2DEXT( GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, m_TextureGL.GetID(), 0 );

        if ( glCheckFramebufferStatusEXT( GL_FRAMEBUFFER_EXT ) != GL_FRAMEBUFFER_COMPLETE_EXT )
        {
            NW_ERR( "INVALID FRAMEBUFFER\n");
        }
    }

    nw::gfnd::TextureGL m_TextureGL;
    GLuint              m_FBO;
    DisplayRotateState   m_Rotate;
};

} // namespace nw::demo
} // namespace nw

#endif // NW_DEMO_FRAME_BUFFER_GL_H_
