﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEMO_FRAME_BUFFER_H_
#define NW_DEMO_FRAME_BUFFER_H_

#include <nw/math.h>

namespace nw
{
namespace demo
{

class DisplayBuffer;

//---------------------------------------------------------------------------
//! @brief        フレームバッファを表す抽象クラスです。
//---------------------------------------------------------------------------
class FrameBuffer
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    FrameBuffer() {}

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    size     フレームバッファサイズです。
    //---------------------------------------------------------------------------
    /* ctor */ explicit FrameBuffer( const nw::math::VEC2& size ) : m_Size( size ) {}

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~FrameBuffer();


    //---------------------------------------------------------------------------
    //! @brief        このフレームバッファを描画対象とします。
    //---------------------------------------------------------------------------
    void Bind() const;

    //---------------------------------------------------------------------------
    //! @brief        フレームバッファサイズを取得します。
    //!
    //! @return       フレームバッファサイズを返します。
    //---------------------------------------------------------------------------
    const nw::math::VEC2& GetSize() const { return m_Size; }

    //---------------------------------------------------------------------------
    //! @brief        現在バインドされているフレームバッファを取得する static 関数です。
    //!
    //! @return       フレームバッファを返します。
    //---------------------------------------------------------------------------
    static const FrameBuffer* GetBoundFrameBuffer() { return s_BoundFrameBuffer; }


protected:
    //! @brief バインドのプラットフォームごとの実装です。
    virtual void BindImpl() const = 0;

private:
    nw::math::VEC2 m_Size;    //!< フレームバッファサイズです。
    static const FrameBuffer* s_BoundFrameBuffer;    //!< 現在バインドされている FrameBuffer です。
};



//---------------------------------------------------------------------------
//! @brief        実際のハードウェア上に確保されたディスプレイを表す抽象クラスです。
//---------------------------------------------------------------------------
class DisplayBuffer
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    DisplayBuffer()
       : m_Width(0),
        m_Height(0)
        {}

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~DisplayBuffer() {}

    //---------------------------------------------------------------------------
    //! @brief        初期化処理を行います。
    //!
    //! @param[in]    width    ディスプレイの幅です。
    //! @param[in]    height   ディスプレイの高さです。
    //---------------------------------------------------------------------------
    void Initialize( const float width, const float height )
    {
        m_Width = width;
        m_Height = height;

        InitializeImpl();
    }

    //---------------------------------------------------------------------------
    //! @brief        実際にディスプレイへ転送します。
    //---------------------------------------------------------------------------
    virtual void CopyToDisplay() {}

    //---------------------------------------------------------------------------
    //! @brief        ディスプレイの幅を返します。
    //!
    //! @return       幅を返します。
    //---------------------------------------------------------------------------
    float GetWidth() const { return m_Width;  }

    //---------------------------------------------------------------------------
    //! @brief        ディスプレイの高さを返します。
    //!
    //! @return       高さを返します。
    //---------------------------------------------------------------------------
    float GetHeight() const { return m_Height; }


protected:
    virtual void InitializeImpl() {}

    float m_Width;
    float m_Height;
};

} // namespace nw::demo
} // namespace nw

#endif // NW_DEMO_FRAME_BUFFER_H_
