﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nns/nac/nac_Types.h>

#if !defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
#include <nn/os.h>
#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>
#else
#include <cafe/mem.h>
#include <cafe/gx2.h>
#endif

namespace nns {
namespace nac {

//---------------------------------------------------------------------------
//! @brief        メモリ管理を行う為のクラスです。
//---------------------------------------------------------------------------
class MemoryAllocator
{
public:

    //! @brief デバックフィルの有効・無効です。
    enum DebugFill
    {
        DebugFill_Disabled,
        DebugFill_Enabled,
    };

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタ
    //---------------------------------------------------------------------------
    MemoryAllocator():
             m_IsCreated( false )
    {}

    //---------------------------------------------------------------------------
    //! @brief        デストラクタ
    //---------------------------------------------------------------------------
    virtual ~MemoryAllocator() {}

    //---------------------------------------------------------------------------
    //! @brief        ヒープを初期化します。
    //!
    //! @param[in] startAddress    管理するバッファの先頭アドレス
    //! @param[in] memSize         メモリサイズ
    //! @param[in] debugFill       デバックフィルの有効・無効
    //---------------------------------------------------------------------------
    void Initialize( void* startAddress, size_t memSize, DebugFill debugFill = DebugFill_Disabled )
    {
#if !defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )

        // デバックフィルが有効かどうか
        nn::lmem::CreationOption option = nn::lmem::CreationOption_NoOption;
        if(debugFill == DebugFill_Enabled)
        {
            option = nn::lmem::CreationOption_DebugFill;
        }

        // Debug ビルドの場合は強制的に デバックフィルを On にします。
#if defined( NN_SDK_BUILD_DEBUG )
        option = nn::lmem::CreationOption_DebugFill;
#endif

        m_HeapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(startAddress), memSize, option);
        if ( m_HeapHandle )
        {
            m_IsCreated = true;
        }
#else
        m_HeapHandle = MEMCreateExpHeapEx( startAddress, memSize, 0 );
        if ( m_HeapHandle )
        {
            m_IsCreated = true;
        }
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        ヒープ管理構造体を削除します。
    //---------------------------------------------------------------------------
    void Finalize()
    {
#if !defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
        nn::lmem::DestroyExpHeap(m_HeapHandle);
        m_IsCreated = false;
#else
        MEMDestroyExpHeap(m_HeapHandle);
        m_IsCreated = false;
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief                  メモリ確保を行います。
    //!
    //! @param[in] size         確保するメモリサイズ
    //! @return                 確保したメモリ領域へのポインタ
    //---------------------------------------------------------------------------
    virtual void* Alloc( size_t size )
    {
#if !defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
        return nn::lmem::AllocateFromExpHeap( m_HeapHandle, size );
#else
        return MEMAllocFromExpHeapEx(m_HeapHandle, size, 128);
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief                  メモリ確保を行います。
    //!
    //! @param[in] size         確保するメモリサイズ
    //! @param[in] alignment    アライメント
    //! @return                 確保したメモリ領域へのポインタ
    //---------------------------------------------------------------------------
    virtual void* Alloc( size_t size, size_t alignment )
    {
#if !defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
        int intAlignment = static_cast<int>( alignment );
        return nn::lmem::AllocateFromExpHeap( m_HeapHandle, size, intAlignment );
#else
        s32 align = (s32)GX2RoundUp( alignment, 4 );
        return MEMAllocFromExpHeapEx(m_HeapHandle, size, align);
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief          確保したメモリを破棄します。
    //!
    //!                 各ご使用の環境に合わせて派生クラス側にて実装をお願いします。
    //! @param[in] ptr  破棄するメモリアドレス
    //---------------------------------------------------------------------------
    void Free( void* ptr )
    {
        if (ptr == NULL)
        {
            return;
        }

#if !defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
        nn::lmem::FreeToExpHeap( m_HeapHandle, ptr );
#else
        MEMFreeToExpHeap(m_HeapHandle, ptr);
#endif
    }

private:
     bool                     m_IsCreated;
#if !defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
    nn::lmem::HeapHandle      m_HeapHandle;
#else
    MEMHeapHandle             m_HeapHandle;
#endif
};

}
}
