﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       TouchScreen の機能を持つコントローラに関する API の宣言
 */

#pragma once

#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/util/util_MathTypes.h>
#include <nns/hid/hid_Controller.h>
#include <nns/hid/hid_ControllerId.h>
#include <nns/hid/hid_ControllerManager.h>
#include <nns/hid/hid_TouchScreenAsset.h>

namespace nns { namespace hid {

/**
 * @brief   TouchScreen の機能を持つコントローラクラスです。
 */
class TouchScreen : public Controller
{
    NN_DISALLOW_COPY(TouchScreen);
    NN_DISALLOW_MOVE(TouchScreen);

public:
    /**
     * @brief       TouchScreen のタッチを表す構造体です。
     */
    struct TouchState
    {
        int32_t fingerId;               //!< タッチの識別子です。
        PointerPhaseSet phase;          //!< タッチのフェーズです
        nn::TimeSpanType deltaTime;     //!< タッチが最後にサンプリングされてから経過した時間です。
        nn::util::Float2 deltaPosition; //!< タッチが最後にサンプリングされてから移動した距離です。
        nn::util::Float2 position;      //!< タッチの座標です。

        /**
         * @brief       新たに検出されたタッチか否かを返します。
         *
         * @return      新たに検出されたタッチか否かを表す値です。
         */
        bool IsBegan() const NN_NOEXCEPT
        {
            return phase.Test<PointerPhase::Began>();
        }

        /**
         * @brief       タッチが離れたか否かを返します。
         *
         * @return      タッチが離れたか否かを表す値です。
         */
        bool IsEnded() const NN_NOEXCEPT
        {
            return phase.Test<PointerPhase::Ended>();
        }

        /**
         * @brief       タッチがキャンセルされたか否かを返します。
         *
         * @return      タッチがキャンセルされたか否かを表す値です。
         */
        bool IsCanceled() const NN_NOEXCEPT
        {
            return phase.Test<PointerPhase::Canceled>();
        }

        /**
         * @brief       タッチが移動したか否かを返します。
         *
         * @return      タッチが移動したか否かを表す値です。
         */
        bool IsMoved() const NN_NOEXCEPT
        {
            return (deltaPosition.x != 0 || deltaPosition.y != 0);
        }
    };

    /**
     * @brief       TouchScreen のコンストラクタです。
     *
     * @param[in]   pManager                    TouchScreen の管理者です。
     *
     * @pre
     *              - pManager != NULL
     */
    explicit TouchScreen(ControllerManager* pManager) NN_NOEXCEPT;

    /**
     * @brief       TouchScreen のデストラクタです。
     */
    virtual ~TouchScreen() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       TouchScreen の状態を更新します。
     *
     * @details     更新はフレーム毎に行う必要があります。
     */
    virtual void Update() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       TouchScreen のコントローラ識別子を返します。
     *
     * @return      TouchScreen のコントローラ識別子です。
     */
    virtual ControllerId GetControllerId() NN_NOEXCEPT NN_OVERRIDE
    {
        return ControllerId_TouchScreen;
    }

    /**
     * @brief       TouchScreen のタッチ状態を返します。
     *
     * @return      TouchScreen のタッチ状態です。
     */
    const std::vector<TouchState>& GetTouchStates() const NN_NOEXCEPT
    {
        return m_States;
    }

private:
    TouchScreenAsset* m_pTouchScreenAsset;  //!< 対応する TouchScreen のデバイスアセットです。

    bool m_IsTouched;                       //!< TouchScreen をデジタルボタンと見做した場合に押下中とするか否かを表す値です。

    std::vector<TouchState> m_States;       //!< TouchScreen のタッチ状態です。

    /**
     * @brief       TouchScreen のタッチ状態を更新します。
     */
    static void UpdateTouchState(
        TouchState* pTouchState,
        const TouchScreenAsset::TouchScreenState& touchScreenState) NN_NOEXCEPT;

    /**
     * @brief       TouchScreen のタッチ状態を更新します。
     */
    void UpdateTouchStates() NN_NOEXCEPT;
};

}} // namespace nns::hid
