﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nn/perf.h>

namespace nns
{
namespace gfx
{
namespace PrimitiveRenderer
{

//! @brief        負荷メーターを描画するクラスです。
class MeterDrawer
{
public:
    //! @brief        コンストラクタです。
    //! @param[in]    frameMeter フレーム全体時間を計測する負荷メーターです。
    explicit MeterDrawer() NN_NOEXCEPT
    {
        SetDefault();
    }

    //! @brief        デストラクタです。
    ~MeterDrawer() NN_NOEXCEPT
    {}

    //! @name 設定
    //! @{

    //! @brief    デフォルト値に設定します。
    void SetDefault() NN_NOEXCEPT
    {
        m_Width = 1260.f;
        m_MeterWidth = 1260.f;
        m_BarHeight = 24.f;
        m_WidthThreshold = 0.5f;
        m_Scale = 0;
        m_CurrentScale = 2;
        m_ScaleMax = 5;
        m_AdjustFrameCount = 10;
        m_AdjustReduceCounter = 10;
        m_AdjustAddCounter = 10;
        m_DepthCountMax = 10;
        m_VisibleNoResult = true;
        m_BorderColor = nn::util::Color4u8::Black();
        m_BackGroundColor = nn::util::Color4u8(255, 255, 255, 64);
        m_TextColor = nn::util::Color4u8::White();
        m_pDebugFontTextWriter = NULL;
        m_Position.x = 10.f;
        m_Position.y = 10.f;
        m_MeterPositionX = m_Position.x;
        SetFrameRate(60.f);
    }

    //! @brief        負荷メーターの描画位置を左上座標を設定します。
    //! @param[in]    position       設定する描画位置です。
    void SetPosition( const nn::util::Float2& position ) NN_NOEXCEPT
    {
        m_Position = position;
    }

    //! @brief        負荷メーターの描画幅を設定します。
    //! @param[in]    width     設定する描画幅です。
    void SetWidth( float width ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT( width > 0 );
        m_Width = width;
    }

    //! @brief        子負荷メーター1つあたりの高さを設定します。
    //! @param[in]    height   子負荷メーター1つあたりの高さです。
    //! @details      全体のメーターの高さは 「m_BarHeight * アタッチしたメーターの数」です。
    void SetBarHeight( float height ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT( height > 0 );
        m_BarHeight = height;
    }

    //! @brief        区間毎の計測結果を描画する最小の幅を指定します。
    //! @param[in]    threshold 描画する最小のバーの長さです。
    //! @details      設定した幅より小さく描画される結果は描画を行いません。
    void SetWidthThreshold( float threshold ) NN_NOEXCEPT
    {
        m_WidthThreshold = threshold;
    }

    //! @brief        負荷メーターの表示のために用いる基準フレームレートを設定します。初期値は 60.f です。
    //! @param[in]    frameRate 設定するフレームレートです。
    void SetFrameRate( float frameRate ) NN_NOEXCEPT
    {
        m_ReferenceFrame = nn::TimeSpan::FromMicroSeconds( static_cast<int64_t>( 1000000.f / frameRate ) );
    }

    //! @brief        基準フレームレートに対して何区切り分を描画するかを表すスケールを設定します。
    //! @param[in]    scale スケールです。
    //! @details      1 を設定すると基準フレームレートに収まる処理までしか描画しません。
    //! @n            基準フレームレートに収まらない場合は 2 以上のスケールを設定して下さい。
    //! @n            なお 0 に設定した場合、負荷に合わせてスケールを自動調整します。
    void SetScale( int scale ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(scale >= 0);
        m_Scale = scale;
    }

    //! @brief        最大スケール値を設定します。
    //! @param[in]    scaleMax    最大スケール値です。
    //! @details      0 を指定すると最大数の制限を行いません。
    void SetScaleMax( int scaleMax ) NN_NOEXCEPT
    {
        m_ScaleMax = scaleMax;
    }

    //! @brief        スケールが自動調整の場合、自動調整するまでのフレーム数を指定します。
    //! @param[in]    adjustFrameCount 自動調整するまでのフレーム数です。
    //! @details      自動調整するまでのフレーム数を指定することにより、
    //!               フレームの目盛り表示がチャタリングするのを防ぎます。
    void SetAdjustFrameCount( int adjustFrameCount ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(adjustFrameCount >= 0);
        m_AdjustFrameCount = adjustFrameCount;
    }

    //! @brief        描画する最大の入れ子の深さを指定します。
    //! @param[in]    depthCountMax 最大の入れ子の深さです。
    //! @details      指定した値以下の LoadMeter::Section::depth を持つ計測結果を描画します。
    void SetDepthCountMax(int depthCountMax) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(depthCountMax >= 0);
        m_DepthCountMax = depthCountMax;
    }

    //! @brief        計測結果を持たない負荷メーター全体の表示/非表示を設定します。
    //! @param[in]    visible   false を指定した場合、計測結果を持たない負荷メーターを表示しません。
    //! @details      初期値は true です。
    void SetVisibleNoResult( bool visible ) NN_NOEXCEPT
    {
        m_VisibleNoResult = visible;
    }

    //! @brief        枠線の色を設定します。
    //! @param[in]    color     枠線の色です。
    void SetBorderColor( const nn::util::Color4u8Type& color ) NN_NOEXCEPT
    {
        m_BorderColor = color;
    }
    //! @brief        背景の色を設定します。
    //! @param[in]    color     背景の色です。
    void SetBackGroundColor( const nn::util::Color4u8Type& color ) NN_NOEXCEPT
    {
        m_BackGroundColor = color;
    }

    //! @brief        文字描画に使用する色を設定します。
    //! @param[in]    color     文字描画に使用する色です。
    void SetTextColor( const nn::util::Color4u8Type& color ) NN_NOEXCEPT
    {
        m_TextColor = color;
    }

    //! @brief        文字描画に用いる DebugFontTextWrite を設定します。
    //! @param[in]    pTextWriter DebugFontTextWrite です。
    //! @details      設定しない場合や NULL を指定した場合は文字の描画を行いません。
    void SetDebugFontTextWriter( nn::gfx::util::DebugFontTextWriter* pTextWriter ) NN_NOEXCEPT
    {
        m_pDebugFontTextWriter = pTextWriter;
    }

    //! @}
    //! @name 取得
    //! @{

    //! @brief        負荷メーターの描画位置の左上座標を取得します。
    //! @return       描画位置を返します。
    const nn::util::Float2& GetPosition() const NN_NOEXCEPT
    {
        return m_Position;
    }

    //! @brief        負荷メーターの描画幅を取得します。
    //! @return       描画幅を返します。
    float GetWidth() const NN_NOEXCEPT
    {
        return m_Width;
    }

    //! @brief        負荷メーター全体の高さを取得します。
    //! @return       負荷メーター全体の高さを返します。
    //! @param[in]    pRootMeter   親負荷メーターです。
    float GetHeight(nn::perf::LoadMeterBase* pRootMeter) const NN_NOEXCEPT;

    //! @brief        アタッチされた子負荷メーター1つあたりの高さを取得します。
    //! @return       子負荷メーター1つあたりの高さを返します。
    float GetBarHeight() const NN_NOEXCEPT
    {
        return m_BarHeight;
    }

    //! @brief        描画する最小の計測区間の長さを指定します。
    //! @return       描画する最小の計測区間の長さを返します。
    float GetWidthThreshold() const NN_NOEXCEPT
    {
        return m_WidthThreshold;
    }

    //! @brief        設定されているスケールを取得します。
    //! @return       スケールの値を返します。
    int GetScale() const NN_NOEXCEPT
    {
        return m_Scale;
    }

    //! @brief        スケールの最大値を取得します。
    //! @return       スケールの最大値を返します。
    int GetScaleMax() const NN_NOEXCEPT
    {
        return m_ScaleMax;
    }

    //! @brief        スケールが自動調整されるまでのフレーム数を取得します。
    //! @return       フレーム数を返します。
    int GetAdjustFrameCount() const NN_NOEXCEPT
    {
        return m_AdjustFrameCount;
    }

    //! @brief        描画する最大の入れ子の深さを取得します。
    //! @return       入れ子の深さを返します。
    int GetDepthCountMax() const NN_NOEXCEPT
    {
        return m_DepthCountMax;
    }

    //! @brief        計測結果を持たない負荷メーターの表示/非表示状態を取得します。
    //! @return       非表示状態の場合 false を返します。
    bool IsVisibleNoResult() const NN_NOEXCEPT
    {
        return m_VisibleNoResult;
    }

    //! @brief        枠線の色を取得します。
    //! @return       枠線の色を返します。
    const nn::util::Color4u8& GetBorderColor() const NN_NOEXCEPT
    {
        return m_BorderColor;
    }

    //! @brief        背景の色を取得します。
    //! @return       背景の色を返します。
    const nn::util::Color4u8& GetBackGroundColor() const NN_NOEXCEPT
    {
        return m_BackGroundColor;
    }

    //! @brief        文字列描画の色を取得します。
    //! @return       文字列描画の色を返します。
    const nn::util::Color4u8& GetTextColor() const NN_NOEXCEPT
    {
        return m_TextColor;
    }

    //! @brief        設定されている DebugFontTextWrite を取得します。
    //! @return       設定されている DebugFontTextWrite を返します。
    nn::gfx::util::DebugFontTextWriter* GetDebugFontTextWriter() const NN_NOEXCEPT
    {
        return m_pDebugFontTextWriter;
    }

    //! @}

    //! @brief        負荷メーターの描画を行います。
    //! @param[in]    pCommendBuffer  描画コマンド格納先のコマンドバッファへのポインタです
    //! @param[in]    pRenderer   　　PrimitiveRenderer へのポインタです。
    //! @param[in]    pRootMeter      描画する親負荷メーターへのポインタです（アタッチされた子負荷メーターも描画されます）。
    //! @details      内部で PrimitiveRenderer に設定する model, view, projection 行列を MatrixIdentity でデフォルト値に設定します。
    void Draw( nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::PrimitiveRenderer::Renderer* pRenderer, nn::perf::LoadMeterBase* pRootMeter) NN_NOEXCEPT;

private:

    //! @brief        ある時刻 tick がメーター上の座標のどこに位置するかを計算します。
    //! @param[in]    frameBeginTick  フレームの開始時刻です
    //! @param[in]    tick            計算対象の時刻です
    float CalculateTimeLinePosition( nn::os::Tick frameBeginTick, nn::os::Tick tick ) NN_NOEXCEPT;

    //! @brief        親負荷メーターの計測結果から、表示する必要がある目盛りの数を計算します。
    //! @param[in]    pRootMeter   親負荷メーターへのポインタです
    void UpdateScale(nn::perf::LoadMeterBase* pRootMeter) NN_NOEXCEPT;

    typedef nn::util::IntrusiveList<nn::perf::LoadMeterBase, nn::util::IntrusiveListBaseNodeTraits<nn::perf::LoadMeterBase>> LoadMeterList;

    nn::util::Float2   m_Position;                //!< 負荷メーターの全体の描画位置です。
    float              m_Width;                   //!< 負荷メーターの全体の横幅です。
    float              m_MeterPositionX;          //!< フォントを除いた負荷メーターの描画位置です。
    float              m_MeterWidth;              //!< フォントを除いた負荷メーターの横幅です。
    float              m_BarHeight;               //!< 個々の子負荷メーターの高さです。全体のメーターの高さは m_BarHeight * 子負荷メーターの数です。
    float              m_WidthThreshold;          //!< 区間毎の結果を描画する最小の横幅です。

    int                m_Scale;                   //!< 何区切り分を表示するかを表すスケールです。0 の場合は自動で表示範囲を変えます。
    int                m_CurrentScale;            //!< 自動区切りとき、現在のスケールです。
    int                m_ScaleMax;                //!< スケールの最大本数です。
    int                m_AdjustFrameCount;        //!< 自動区切りの区切り数をアジャストするまでのフレーム数です。
    int                m_AdjustReduceCounter;     //!< あと何フレームでアジャストするかのカウンターです。
    int                m_AdjustAddCounter;        //!< あと何フレームでアジャストするかのカウンターです。

    nn::TimeSpan       m_ReferenceFrame;          //!< 1 フレームの基準となる時間です。

    int                m_DepthCountMax;           //!< 描画する最大の入れ子の深さです。
    bool               m_VisibleNoResult;         //!< 計測結果を持たない子メーターを表示するかです。

    nn::util::Color4u8 m_BorderColor;             //!< 枠線の色です。
    nn::util::Color4u8 m_BackGroundColor;         //!< 背景の色です。
    nn::util::Color4u8 m_TextColor;               //!< 文字列色です。

    nn::gfx::util::DebugFontTextWriter* m_pDebugFontTextWriter; //!< 文字列描画に用いる DebugFontTextWriter です。
};

} // namespace PrimitiveRenderer
} // namespace gfx
} // namespace nns

