﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/gfx.h>
#include <nn/gfx/util/gfx_PrimitiveShape.h>
#include <nn/util/util_Arithmetic.h>
#include <nn/util/util_Vector.h>
#include <nn/util/util_Matrix.h>

#include <nn/mem/mem_StandardAllocator.h>

#include <nns/gfx/gfx_GpuBuffer.h>
#include <nns/gfx/gfx_PrimitiveRendererShaderRes.h>


namespace nns
{
namespace gfx
{
namespace PrimitiveRenderer
{

//---------------------------------------------------------------------------
//! @brief 前方宣言です。
//---------------------------------------------------------------------------
class MemoryInfo;

//---------------------------------------------------------------------------
//! @brief 定数です。
//---------------------------------------------------------------------------

//! @brief 不正なオフセットです。
static const ptrdiff_t InvalidOffset = -1;

//! @brief バッファの種類です。
enum BufferType
{
    BufferType_VertexBuffer = 0,
    BufferType_IndexBuffer,
    BufferType_ConstantBuffer,

    BufferType_End,
};

//! @brief プリミティブの形状です。
enum ShapeType
{
    ShapeType_Quad              = 0,
    ShapeType_QuadTextured,
    ShapeType_QuadWired,
    ShapeType_Cube,
    ShapeType_CubeTextured,
    ShapeType_CubeWired,
    ShapeType_Sphere,
    ShapeType_SphereTextured,
    ShapeType_SphereWired,
    ShapeType_SphereCoarse,
    ShapeType_SphereTexturedCoarse,
    ShapeType_SphereWiredCoarse,
    ShapeType_Line,
    ShapeType_Point,
    ShapeType_Triangle,
    ShapeType_TriangleWired,
    ShapeType_Circle,
    ShapeType_CircleTextured,
    ShapeType_CircleWired,
    ShapeType_CircleCoarse,
    ShapeType_CircleTexturedCoarse,
    ShapeType_CircleWiredCoarse,
    ShapeType_ScreenQuad,
    ShapeType_ScreenYFlipQuad,
    ShapeType_UpperHalfSphere,
    ShapeType_UpperHalfSphereTextured,
    ShapeType_UpperHalfSphereWired,
    ShapeType_Pipe,
    ShapeType_PipeTextured,
    ShapeType_PipeWired,
    ShapeType_Cylinder,
    ShapeType_CylinderTextured,
    ShapeType_CylinderWired,
    ShapeType_Cone,
    ShapeType_ConeTextured,
    ShapeType_ConeWired,
    ShapeType_User,

    ShapeType_CountMax  // 種類の数です。
};

//! @brief サーフェイスの種類です。
enum Surface
{
    Surface_Wired = 0,
    Surface_Solid,
    Surface_Normal,

    Surface_CountMax,
};

//! @brief サーフェイスの細分化レベルです。
enum Subdiv
{
    Subdiv_Coarse = 0,
    Subdiv_Normal,
    Subdiv_Beautiful,

    Subdiv_CountMax,
};

//---------------------------------------------------------------------------
//! @brief [0, 255] -> [0, 1] 変換します。
//! @param[out]pOutValue    変換先です。
//! @param[in] source       変換元です。
//---------------------------------------------------------------------------
void ConvertUint8x4ToFloat4( nn::util::Float4* pOutValue,
                             const nn::util::Uint8x4 source );

//---------------------------------------------------------------------------
//! @brief Matrix4x3 から FloatT4x4に変換します。
//! @param[out] pOut4x4      変換先です。
//! @param[in]  inMatrix4x3  変換元です。
//---------------------------------------------------------------------------
void ConvertMatrix4x3ToFloatT4x4( nn::util::FloatT4x4* pOut4x4,
                                  const nn::util::Matrix4x3fType& inMatrix4x3 );

//---------------------------------------------------------------------------
//! @brief メッシュです。
//! @details 1つだけでなく、複数の形状を設定することが出来ます。
//---------------------------------------------------------------------------
class PrimitiveMesh
{
    NN_DISALLOW_COPY(PrimitiveMesh);
public:
    //! @brief コンストラクタです。
    PrimitiveMesh();

    //! @brief デストラクタです。
    virtual ~PrimitiveMesh();

    //----------------------------------------
    //! @name 初期化／後処理
    //@{

    //---------------------------------------------------------------------------
    //! @brief initializeを行います。
    //!
    //! @param[in] pGpuBuffer   利用するGpuBufferのポインタ
    //! @param[in] numVertices  頂点数
    //! @param[in] numIndices   インデックス数
    //! @param[in] vertexFormat 頂点バッファのフォーマット
    //! @return initializeに成功した場合に、trueを返します。
    //---------------------------------------------------------------------------
    bool Initialize( GpuBuffer* pGpuBuffer,
                     int numVertices,
                     int numIndices,
                     VertexFormat vertexFormat = VertexFormat_Default );

    //@}


    //----------------------------------------
    //! @name 設定／取得（パラメータ）
    //! @{

    //! @brief 頂点数を取得します。
    int GetVertexCount() const;

    //! @brief インデック数を取得します。
    int GetIndexCount() const;

    //! @brief 頂点属性のストライドを取得します。
    //! @param[in] attrib   頂点属性です。
    //! @return ストライドです。
    size_t GetVertexStride( VertexAttribute attrib ) const;

    //! @brief 指定された頂点属性のメモリサイズを取得します。
    //! @param[in] attrib   頂点属性です。
    //! @return メモリサイズを返します。
    size_t GetVertexMemorySize( VertexAttribute attrib ) const;

    //! @brief 頂点バッファのポインタを取得します。
    //! @param[in] attrib   頂点属性です。
    //! @return ポインタを返します。
    void* GetVertexBufferCpuAddress( VertexAttribute attribute );

    //! @brief インデックスバッファのポインタを取得します。
    //! @return ポインタを返します。
    uint32_t* GetIndexBufferCpuAddress();

    //! @brief 指定された頂点属性のメモリサイズを取得します。
    //! @param[in] attrib   頂点属性です。
    //! @param[out] pGpuAddress
    void GetVertexBufferGpuAddress( VertexAttribute attrib, nn::gfx::GpuAddress* pGpuAddress) const;

    //! @brief 指定された頂点属性が初期化済みか取得します。
    //! @param[in] attrib   頂点属性です。
    //! @param[out] trueなら初期化済。falseなら未初期化
    bool IsValidVertexBuffer( VertexAttribute attrib ) const;

    //! @brief インデックスバッファが初期化済みか取得します。
    //! @param[out] trueなら初期化済。falseなら未初期化
    bool IsValidIndexBuffer() const;

    //! @brief インデックスバッファのGpuAddressを取得します。
    //! @param[out] pGpuAddress インデックスバッファのGpuAddress
    void GetIndexBufferGpuAddress( nn::gfx::GpuAddress* pGpuAddress ) const;

    //! @brief 頂点フォーマットを取得します。
    //! @return 頂点フォーマットを返します。
    VertexFormat GetVertexFormat() const;

    //! @}

    //----------------------------------------
    //! @name コピー
    //! @{

    template<typename T>
    //! @brief 頂点属性の頂点バッファに、指定のデータ型で、指定のデータでコピーします。
    //! @param[in] attribute    頂点属性です。
    //! @param[in] pPodData     PODデータです。
    //! @param[in] size         データサイズです。
    //! @return コピー結果です。trueならコピー成功です。
    bool CopyVertexBuffer( VertexAttribute attribute, const T* pPodData, size_t size );

    //! @brief インデックスバッファに、指定データをコピーします。
    //! @param[in] pPodData     PODデータです。
    //! @param[in] size         データサイズです。
    //! @return コピー結果です。trueならコピー成功です。
    bool CopyIndexBuffer( const uint32_t* pPodData, size_t size );

    //! @}
private:

    // インタリーブではなく、頂点属性ごとにバッファを持ちます。
    // 未使用の頂点属性は、アロケーションしません。
    GpuBuffer* m_pGpuBuffer;
    void* m_VertexBufferPointer[ VertexAttribute_CountMax ];
    void* m_IndexBufferPointer;
    int m_VertexCount;
    int m_IndexCount;
};

//---------------------------------------------------------------------------
//! @brief メッシュの集合です。
//---------------------------------------------------------------------------
class MeshSet
{
public:
    //! @brief コンストラクタです。
    MeshSet();

    //! @brief デストラクタです。
    ~MeshSet();

    //! @brief  メッシュセットのデータを初期化します。
    bool Initialize(GpuBuffer* pGpuBuffer, nn::util::BytePtr& pMemory, size_t memorySize);

    //! @brief メッシュを取得します。
    //! @param[in] shapeType    メッシュ形状の種類です。
    //! @return メッシュです。
    PrimitiveMesh* Get( ShapeType shapeType );

    //! @brief メッシュを取得します。
    //! @param[in] shapeType    メッシュ形状の種類です。
    //! @return メッシュです。
    nn::gfx::util::PrimitiveShape* GetShape( ShapeType shapeType );

    //! @brief MeshSetに必要なMemoryPoolのサイズを計算します。
    //! @param[in] alignment    アライメントのサイズ
    //! @return 必要なMemoryPoolのサイズ
    static size_t CalculateMemoryPoolSize( const size_t alignment );

    //! @breif MeshSet で動的に確保されるされる可能性のあるメモリ量を取得します。
    //!
    //! @return メモリサイズです。
    static size_t GetRequiredMemorySize();

    //! @breif nn::gfx::util::PrimitiveShapeの頂点とインデックス数を取得します。
    //! @param[out] vertexCount  頂点数です。
    //! @param[out] indexCount   インデックス数です。
    //! @param[in]  shapeType    メッシュ形状の種類です。
    static void GetVertexIndexCount( int* vertexCount, int* indexCount, ShapeType shapeType );

private:

    //! @brief  pCreatedShape で指定された領域に指定されたパラメータのシェイプインスタンスを作成します。
    //!
    //! @tparam T   作成する PrimitiveShape から派生したシェイプを表すクラス。
    //! @param[out] pCreatedShape   作成したインスタンスのポインタを書き込む領域を指定する。
    //! @param[in]  format  頂点フォーマット
    //! @param[in]  topology    トポロジー
    //! @param[in]  pGpuBuffer  頂点バッファとインデックスバッファを確保するための GPU バッファ。
    //! @param[in]  pMemory     インスタンスのメモリを確保する領域。
    //!
    template <typename T>
    void CreateShape(
        nn::gfx::util::PrimitiveShape*& pCreatedShape,
        nn::gfx::util::PrimitiveShapeFormat format,
        nn::gfx::PrimitiveTopology topology,
        GpuBuffer* pGpuBuffer,
        nn::util::BytePtr& pMemory)
    {
        pCreatedShape = new(pMemory.Get()) T(format, topology);
        pMemory.Advance(sizeof(T));

        return AllocateShapeGpuMemory(pCreatedShape, pGpuBuffer);
    }

    //! @brief  pCreatedShape で指定された領域に指定されたパラメータのシェイプインスタンスを作成します。
    //!
    //! @tparam T   作成する PrimitiveShape から派生したシェイプを表すクラス。
    //! @tparam Param1   PrimitiveShape への追加引数。
    //! @param[out] pCreatedShape   作成したインスタンスのポインタを書き込む領域を指定する。
    //! @param[in]  format  頂点フォーマット
    //! @param[in]  topology    トポロジー
    //! @param[in]  pGpuBuffer  頂点バッファとインデックスバッファを確保するための GPU バッファ。
    //! @param[in]  pMemory     インスタンスのメモリを確保する領域。
    //! @param[in]  param1  T のコンストラクタへのパラメータ。
    //!
    template <typename T, typename Param1>
    void CreateShape(
        nn::gfx::util::PrimitiveShape*& pCreatedShape,
        nn::gfx::util::PrimitiveShapeFormat format,
        nn::gfx::PrimitiveTopology topology,
        GpuBuffer* pGpuBuffer,
        nn::util::BytePtr& pMemory,
        Param1 param1)
    {
        pCreatedShape = new(pMemory.Get()) T(format, topology, param1);
        pMemory.Advance(sizeof(T));

        AllocateShapeGpuMemory(pCreatedShape, pGpuBuffer);
    }

    //! @brief  pCreatedShape で指定された領域に指定されたパラメータのシェイプインスタンスを作成します。
    //!
    //! @tparam T   作成する PrimitiveShape から派生したシェイプを表すクラス。
    //! @tparam Param1   PrimitiveShape への追加引数。
    //! @tparam Param2   PrimitiveShape への追加引数。
    //! @param[out] pCreatedShape   作成したインスタンスのポインタを書き込む領域を指定する。
    //! @param[in]  format  頂点フォーマット
    //! @param[in]  topology    トポロジー
    //! @param[in]  pGpuBuffer  頂点バッファとインデックスバッファを確保するための GPU バッファ。
    //! @param[in]  pMemory     インスタンスのメモリを確保する領域。
    //! @param[in]  param1  T のコンストラクタへのパラメータ。
    //! @param[in]  param2  T のコンストラクタへのパラメータ。
    //!
    template <typename T, typename Param1, typename Param2>
    void CreateShape(
        nn::gfx::util::PrimitiveShape*& pCreatedShape,
        nn::gfx::util::PrimitiveShapeFormat format,
        nn::gfx::PrimitiveTopology topology,
        GpuBuffer* pGpuBuffer,
        nn::util::BytePtr& pMemory,
        Param1 param1,
        Param2 param2)
    {
        pCreatedShape = new(pMemory.Get()) T(format, topology, param1, param2);
        pMemory.Advance(sizeof(T));

        AllocateShapeGpuMemory(pCreatedShape, pGpuBuffer);
    }

    //! @brief  シェイプの GPU メモリ確保処理の共通化。
    //!
    //! @param[in] pShape GPU メモリ確保処理を行うシェイプインスタンス。
    //! @param[in] pGpuBuffer  頂点バッファとインデックスバッファを確保するための GPU バッファ。
    //!
    void AllocateShapeGpuMemory(
        nn::gfx::util::PrimitiveShape* pShape,
        GpuBuffer* pGpuBuffer);

    // nn::gfx::util::PrimitiveShape を使用して作成されたメッシュデータのポインタリスト。
    nn::gfx::util::PrimitiveShape* m_pShapes[ShapeType_CountMax];
    // PrimitiveRenderer が独自に作成したメッシュリスト。
    // m_pShapes でメッシュデータが見つからなかったときにこちらが参照されます。
    PrimitiveMesh m_Mesh[ ShapeType_CountMax ];
};



} // namespace PrimitiveRenderer
} // namespace gfx
} // namespace nns
