﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/image/image_Common.h>
#include <nn/mem/mem_StandardAllocator.h>

#include <nns/gfx/gfx_GraphicsFramework.h>

namespace nns {
namespace console {
template <typename T>
class ConsoleManager;
} // ~namespace nns::console
}

namespace nns {
namespace console {
namespace detail {

// コンソールの標準描画設定。毎フレームこの値でリセットされる。
struct ConsoleStyle
{
    int tabWidth;
    float scaleH;
    float scaleV;
    nn::util::Color4u8Type color;
    bool fixedWidthEnabled;
    float fixedWidth;
};

// そのデフォルト
const ConsoleStyle DefaultConsoleStyle = {
    4,      // tabWidth in charactor count
    1.0f,   // scaleH
    1.0f,   // scaleV
    {{223, 223, 223, 255}}, // color
    true,   // fixedWidthEnabled
    16.0f,  // fixedWidth
};

// コンソールの内部機構
class ConsoleBase
{
    template <typename T>
    friend class nns::console::ConsoleManager;

private:
    ConsoleStyle m_Style;

    bool m_Initialized;
    nn::gfx::Device* m_pDevice;
    nn::gfx::ViewportScissorState m_ViewportScissorState;
    nn::image::Dimension m_Dimension;

    void Initialize(nn::gfx::Device* pDevice, const nn::gfx::ViewportStateInfo& viewport, const nn::gfx::ScissorStateInfo& scissor) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;
    void MakeCommand(nn::gfx::util::DebugFontTextWriter* pTextWriter, nn::gfx::CommandBuffer* pCommandBuffer) const NN_NOEXCEPT;
    void Update() NN_NOEXCEPT;

protected:
    virtual void MakeCommandImpl(nn::gfx::util::DebugFontTextWriter* pTextWriter) const NN_NOEXCEPT = 0;
    virtual void UpdateImpl() NN_NOEXCEPT = 0;

protected:
    ConsoleBase() NN_NOEXCEPT;
    explicit ConsoleBase(const ConsoleStyle& style) NN_NOEXCEPT;
    virtual ~ConsoleBase() NN_NOEXCEPT;

    nn::image::Dimension GetDimension() const NN_NOEXCEPT;
};

// グラフィクスリソースの管理体
class ConsoleGraphicsFramework final
{
    NN_DISALLOW_COPY(ConsoleGraphicsFramework);
    NN_DISALLOW_MOVE(ConsoleGraphicsFramework);

private:
    static void* AllocationFunction(size_t size, size_t alignment, void* pUserData) NN_NOEXCEPT;
    static void DeallocationFunction(void* ptr, void* pUserData) NN_NOEXCEPT;
    static void CommandMakeFunction(nns::gfx::GraphicsFramework* pFramework, int bufferIndex, void* pUserData) NN_NOEXCEPT;

    bool m_Initialized;
    void(*m_Callback)(nn::gfx::CommandBuffer* pCommandBuffer, void* callbackUserData);
    void* m_CallbackUserData;
    nn::util::Color4u8Type m_ClearColor;

    nn::mem::StandardAllocator m_Allocator;
    nns::gfx::GraphicsFramework m_Framework;

    void* Allocate(size_t size, size_t alignment) NN_NOEXCEPT;
    void* Allocate(size_t size) NN_NOEXCEPT;
    void Deallocate(void* ptr) NN_NOEXCEPT;

public:
    ConsoleGraphicsFramework(void *buffer, size_t bufferSize) NN_NOEXCEPT;
    ~ConsoleGraphicsFramework() NN_NOEXCEPT;

    void Initialize(
        void(*callback)(nn::gfx::CommandBuffer* pCommandBuffer, void* callbackUserData), void* callbackUserData,
        size_t width, size_t height, const nn::util::Color4u8Type& clearColor,
        size_t memorySize) NN_NOEXCEPT;

    void InitializeDebugTextWriter(
        nn::gfx::util::DebugFontTextWriter* pWriter,
        size_t width, size_t height, int charactorCountMax) NN_NOEXCEPT;
    void FinalizeDebugTextWriter(nn::gfx::util::DebugFontTextWriter* pWriter) NN_NOEXCEPT;
    nn::gfx::Device* GetDevice() NN_NOEXCEPT;
    void ProcessFrame() NN_NOEXCEPT;
};

} // ~namespace nns::console::detail
}
}
