﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nns/ble/ble_Types.h>
#include <nns/ble/ble_Hid.h>

namespace nns { namespace ble {

/**
 * @brief   BLE 関連の初期化を行います
 */
void Initialize() NN_NOEXCEPT;

/**
 * @brief   BLE 関連の終了処理を行います
 */
void Finalize() NN_NOEXCEPT;

/**
 * @brief   BLE スキャンを開始します
 *
 * @param[in]   pFilter             スキャンに使用するフィルタクラス
 *
 * @pre
 *  - Initialize 実行済み
 *  - @a pFilter != nullptr
 */
void StartScan(const uint8_t (&clientId)[nn::bluetooth::BLE_NN_ADV_MANU_CLIENT_ID_SIZE], const uint8_t (&serverId)[nn::bluetooth::BLE_NN_ADV_MANU_SERVER_ID_SIZE]) NN_NOEXCEPT;

/**
 * @brief   BLE スキャンを停止します
 *
 * @pre
 *  - Initialize 実行済み
 */
void StopScan() NN_NOEXCEPT;

/**
 * @brief   BLE スキャンを実行中か判定します
 *
 * @pre
 *  - Initialize 実行済み
 */
bool IsScanning() NN_NOEXCEPT;

/**
 * @brief   BLE スキャンの結果を取得します
 *
 * @param[out]  pOutResults     スキャン結果の格納先
 * @param[in]   resultCountMax  格納するスキャン結果の最大数
 *
 * @return  有効なスキャン結果の数
 *
 * @pre
 *  - @ref Initialize 実行済み
 *  - @a pOutResults != nullptr
 */
int GetScanResult(
    nn::bluetooth::BleScanResult* pOutResults,
    int resultCountMax) NN_NOEXCEPT;

/**
 * @brief   指定した GATT Server と接続します
 *
 * @detail  @ref GetScanResults で取得した GATT Server と接続します。
 *
 * @param[in]   serverAddress   GATT Server の Bluetooth アドレス
 *
 * @pre
 *  - Initialize 実行済み
 */
void Connect(nn::bluetooth::Address serverAddress) NN_NOEXCEPT;

/**
 * @brief   指定した GATT Client を GATT Server から切断します
 *
 * @param[in]   clientIndex     切断するクライアントの番号
 *
 * @pre
 *  - Initialize 実行済み
 */
void Disconnect(uint8_t clientIndex) NN_NOEXCEPT;

/**
 * @brief   指定した GATT Client の MTU を更新します
 *
 * @param[in]   clientIndex     対象のクライアントの番号
 * @param[in]   mtu             設定する MTU 値
 *
 * @pre
 *  - Initialize 実行済み
 */
void UpdateMtu(uint8_t clientIndex, uint16_t mtu) NN_NOEXCEPT;

/**
 * @brief   GATT Client を取得します
 *
 * @param[out]  pOutClient      GATT Client の格納先
 * @param[in]   clientIndex     対象のクライアントの番号
 *
 * @pre
 *  - Initialize 実行済み
 *  - @a pOutClient != nullptr
 */
void GetGattClientStatus(nns::ble::GattClientStatus* pOutClient, uint8_t clientIndex) NN_NOEXCEPT;

}}  // nns::ble
