﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NW_CONSOLE_ENABLE // Release 版でも NW_LOG を有効にするため

#include <nw/snd.h>
#include <nw/sndutil.h>
#include <nw/demo.h>
#include "common.fsid"

#if defined( NW_PLATFORM_CAFE )
#include <cafe.h>
#elif defined( NW_USE_NINTENDO_SDK )
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <nn/nn_Windows.h>
#include <nn/fs.h>
#else
#include <winext/cafe.h>
#include <Windows.h>
#endif

namespace
{
#if defined(NW_USE_NINTENDO_SDK)
const char MOUNT_NAME[] = "content";
const char SOUND_ARC_PATH[] = "content:snd/common/common.bfsar";
#else
const char CONTENT_DIR[] = "/vol/content";
const char SOUND_ARC_PATH[] = "snd/common/common.bfsar";
#endif

nw::snd::FsSoundArchive     s_SoundArchive;
nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
nw::snd::SoundDataManager   s_SoundDataManager;
nw::snd::SoundHandle        s_SoundHandle;
nw::snd::SoundHandle        s_SoundHandleHold;

void* s_pMemoryForSoundSystem;
void* s_pMemoryForInfoBlock;
void* s_pMemoryForSoundDataManager;
void* s_pMemoryForSoundArchivePlayer;
void* s_pMemoryForStreamBuffer;

#if defined(NW_PLATFORM_CAFE)
FSClient* s_pFsClient;
void* s_pMemoryForFsSoundArchive;
#endif

const int GROUP_FILE_COUNT = 4;
void* s_pMemoryForGroupFiles[GROUP_FILE_COUNT];
size_t s_GroupFileSizes[GROUP_FILE_COUNT];

const char* GROUP_FILES_PATH[] =
{
    "/vol/content/snd/common/GROUP_USER.bfgrp",
    "/vol/content/snd/common/GROUP_USER_AUTO_WARC.bfgrp",
    "/vol/content/snd/common/GROUP_USER_INDIVIDUAL_WARC.bfgrp",
    "/vol/content/snd/common/GROUP_USER_TEST_PLAYERHEAP.bfgrp"
};

#if defined(NW_USE_NINTENDO_SDK)
void* Allocate(size_t size)
{
    nw::demo::DefaultAllocator allocator;
    return allocator.Alloc(size);
}

void Deallocate(void* memory, size_t size)
{
    NW_UNUSED_VARIABLE(size);

    nw::demo::DefaultAllocator allocator;
    allocator.Free(memory);
}
#endif

void InitializeNwSound()
{
    nw::demo::DefaultAllocator allocator;

    // サウンドシステムの初期化
    nw::snd::SoundSystem::SoundSystemParam param;
    size_t workMemSize = nw::snd::SoundSystem::GetRequiredMemSize( param );
    s_pMemoryForSoundSystem = allocator.Alloc( workMemSize, 4 );

    nw::snd::SoundSystem::Initialize(
            param,
            reinterpret_cast<uptr>( s_pMemoryForSoundSystem ),
            workMemSize );

    // サウンドアーカイブの初期化
    {
    #if defined(NW_PLATFORM_CAFE)
        size_t size = s_SoundArchive.GetRequiredMemSize();
        s_pMemoryForFsSoundArchive = allocator.Alloc(size);
        if ( ! s_SoundArchive.Open(
                    s_pFsClient, SOUND_ARC_PATH, s_pMemoryForFsSoundArchive, size) )
        {
            NW_ASSERTMSG( 0, "cannot open bfsar(%s)\n", SOUND_ARC_PATH );
        }
    #else
        if ( ! s_SoundArchive.Open(SOUND_ARC_PATH) )
        {
            NW_ASSERTMSG( 0, "cannot open bfsar(%s)\n", SOUND_ARC_PATH );
        }
    #endif
    }

    // INFO ブロックのロード
    {
        size_t infoBlockSize = s_SoundArchive.GetHeaderSize();
        s_pMemoryForInfoBlock = allocator.Alloc( infoBlockSize, nw::snd::FsSoundArchive::BUFFER_ALIGN_SIZE );
        if ( ! s_SoundArchive.LoadHeader( s_pMemoryForInfoBlock, infoBlockSize ) )
        {
            NW_ASSERTMSG( 0, "cannot load infoBlock(%s)", SOUND_ARC_PATH );
        }
    }

    // サウンドデータマネージャーの初期化
    {
        size_t setupSize = s_SoundDataManager.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundDataManager = allocator.Alloc( setupSize, 4 );
        s_SoundDataManager.Initialize( &s_SoundArchive, s_pMemoryForSoundDataManager, setupSize );
    }

    // サウンドアーカイブプレイヤーの初期化
    {
        size_t setupSize = s_SoundArchivePlayer.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundArchivePlayer = allocator.Alloc( setupSize, 32 );
        size_t setupStrmBufferSize =
            s_SoundArchivePlayer.GetRequiredStreamBufferSize( &s_SoundArchive );
        s_pMemoryForStreamBuffer = allocator.Alloc( setupStrmBufferSize, 8 );
        bool result = s_SoundArchivePlayer.Initialize(
                &s_SoundArchive,
                &s_SoundDataManager,
                s_pMemoryForSoundArchivePlayer, setupSize,
                s_pMemoryForStreamBuffer, setupStrmBufferSize );
        NW_ASSERT( result );
    }

    // FileDeviceManager の初期化 (グループファイルロードに利用)
    {
        nw::dev::FileDeviceManager* fileDeviceManager = nw::dev::FileDeviceManager::GetInstance();
        nw::dev::FileDeviceManager::CreateArg arg;
        arg.allocator = &allocator;
        fileDeviceManager->Initialize(arg);
    }

    // グループファイルのロード
    for ( int i = 0; i < GROUP_FILE_COUNT; i++ )
    {
        nw::dev::FileDevice fileDevice;
        nw::dev::FileHandle fileHandle;

        {
            void* result = fileDevice.Open( &fileHandle, GROUP_FILES_PATH[i] );
            NW_ASSERT_NOT_NULL(result);
        }

        {
            s_GroupFileSizes[i] = fileDevice.GetFileSize(&fileHandle);
            s_pMemoryForGroupFiles[i] = allocator.Alloc(s_GroupFileSizes[i], 64);

            s32 readSize;
            readSize = fileDevice.Read(
                    &fileHandle,
                    reinterpret_cast<u8*>(s_pMemoryForGroupFiles[i]),
                    s_GroupFileSizes[i]);
            NW_LOG("[%d] Read size(%7d) path(%s)\n", i, readSize, GROUP_FILES_PATH[i]);
        }

        {
            bool result = fileDevice.Close(&fileHandle);
            NW_ASSERT(result);
        }
    }
}

} // namespace

int
NwDemoMain(int argc, char **argv)
{
#if defined( NW_PLATFORM_CAFE )
    AXInit();

#else
    NW_UNUSED_VARIABLE(argc);
    NW_UNUSED_VARIABLE(argv);
#endif

    nw::demo::DefaultAllocator allocator;
    nw::demo::DemoSystem::CreateArg demoSystemArg;
    demoSystemArg.allocator = &allocator;
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    demoSystemArg.waitVBlank = 0; // DemoSystem の SwapBuffer は呼ばないため、VBlank 待ちはしない。
#endif
    nw::demo::DemoSystem* pDemo =
        new( allocator.Alloc( sizeof( nw::demo::DemoSystem ) ) )
        nw::demo::DemoSystem( demoSystemArg );
    pDemo->Initialize();
    pDemo->InitializeGraphicsSystem();

#if defined(NW_PLATFORM_CAFE)
    // FS 初期化
    {
        FSInit();
        s_pFsClient = reinterpret_cast<FSClient*>(allocator.Alloc(sizeof(FSClient)));
        std::memset(s_pFsClient, 0x00, sizeof(FSClient));
        NW_ASSERT_NOT_NULL(s_pFsClient);

        FSAddClient(s_pFsClient, FS_RET_NO_ERROR);
    }

    // ディレクトリ移動
    {
        FSCmdBlock* block = reinterpret_cast<FSCmdBlock*>(allocator.Alloc(sizeof(FSCmdBlock)));
        NW_ASSERT_NOT_NULL(block);
        FSInitCmdBlock(block);

        FSStatus status = FSChangeDir(s_pFsClient, block, CONTENT_DIR, FS_RET_NO_ERROR);
        NW_ASSERT(status == FS_STATUS_OK);

        allocator.Free(block);
    }
#elif defined( NW_USE_NINTENDO_SDK )
    // nn::fs の初期化
    nn::fs::SetAllocator(Allocate, Deallocate);
    // CONTENT_DIRのマウント
    {
        static const int cFileNameLen = 512;
        char buf[cFileNameLen];
        if ( GetEnvironmentVariableA("CAFE_CONTENT_DIR", buf, cFileNameLen) > 0 )
        {
            nn::fs::MountHost(MOUNT_NAME, buf);
        }
        else
        {
            NW_LOG("cannot read CAFE_CONTENT_DIR\n");
            return -1;
        }
    }
#else
    // ディレクトリ移動
    {
        static const int cFileNameLen = 512;
        char buf[cFileNameLen];
        if ( GetEnvironmentVariableA("CAFE_CONTENT_DIR", buf, cFileNameLen) > 0 )
        {
            NW_LOG("buf(%s)\n", buf);
            SetCurrentDirectoryA(buf);
        }
        else
        {
            NW_LOG("cannot read CAFE_CONTENT_DIR\n");
            return -1;
        }
    }
#endif

    InitializeNwSound();

    NW_LOG("----------------------------------------\n");
    NW_LOG("NW4F Demo/snd/loadGroupManually\n");
    NW_LOG("----------------------------------------\n");
    NW_LOG("[A] SEQ_MARIOKART\n");
    NW_LOG("[X] SEQ_TEST\n");
    NW_LOG("[Y] SE_WIHAHO\n");
    NW_LOG("[L] SE_YOSHI_PLAYERHEAP_FOR_USERGROUP\n");
    NW_LOG("[R] Set/Unset GroupFile\n");
    NW_LOG("[B] Stop Sound\n");
#if defined( NW_PLATFORM_CAFE )
    NW_LOG("[HOME]  Exit Application\n");
#elif defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    NW_LOG("[S] Exit Application\n");
#endif
    NW_LOG("---------------------------------------\n");

    bool setGroupFile = true;

    while ( !pDemo->IsExiting() )
    {
#if defined( NW_PLATFORM_CAFE )
        GX2WaitForVsync();
#else
        pDemo->WaitForVBlank();
#endif

        pDemo->UpdatePad();
        nw::demo::Pad* pad = pDemo->GetPad();

        // StartSound / Stop
        if ( pad->IsTrig( nw::demo::Pad::MASK_A ) ) {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer.StartSound(
                    &s_SoundHandle, SEQ_MARIOKART ).IsSuccess();
            NW_LOG("[SEQ_MARIOKART] ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_X ) ) {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer.StartSound(
                    &s_SoundHandle, SEQ_TEST ).IsSuccess();
            NW_LOG("[SEQ_TEST]      ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_Y ) ) {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer.StartSound(
                    &s_SoundHandle, SE_WIHAHO ).IsSuccess();
            NW_LOG("[SE_WIHAHO]     ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_L ) ) {
            s_SoundHandle.Stop( 0 );

            // [ユーザー管理] のグループにのみ含まれるデータは、
            // プレイヤーヒープを利用するサウンドであっても再生できません。
            // (ただし、プレイヤーヒープ利用時のサウンドは、
            //  データの参照解決が遅延されるので、
            //  サウンドインスタンスさえ確保出来れば、
            //  StartSound().IsSuccess() 自体は true を返します)
            //
            // GROUP_USER_TEST_PLAYERHEAP.bfgrp をロードし、かつ、
            // SetFileAddressInGroupFile を呼び出しておく必要があります。
            bool result = s_SoundArchivePlayer.StartSound(
                    &s_SoundHandle, SE_YOSHI_PLAYERHEAP_FOR_USERGROUP ).IsSuccess();
            NW_LOG("[SE_YOSHI_PLAYERHEAP_FOR_USERGROUP] ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_B ) ) {
            s_SoundHandle.Stop( 0 );
        }

        if ( pad->IsTrig( nw::demo::Pad::MASK_R ) ) {
            if ( setGroupFile )
            {
                // ロード済みグループファイルの登録
                for ( int i = 0; i < GROUP_FILE_COUNT; i++ )
                {
                    s_SoundDataManager.SetFileAddressInGroupFile(
                            s_pMemoryForGroupFiles[i], s_GroupFileSizes[i]);
                }
                NW_LOG("Set GroupFile Address\n");
            }
            else
            {
                // ロード済みグループファイルの登録解除
                for ( int i = 0; i < GROUP_FILE_COUNT; i++ )
                {
                    s_SoundDataManager.ClearFileAddressInGroupFile(
                            s_pMemoryForGroupFiles[i],
                            s_GroupFileSizes[i]);
                }
                NW_LOG("Unset GroupFile Address\n");
            }
            setGroupFile = !setGroupFile;
        }

        // Exit
        if ( pad->IsTrig( nw::demo::Pad::MASK_START ) ) {
            break;
        }

        s_SoundArchivePlayer.Update();
    }

    s_SoundArchivePlayer.Finalize();
    s_SoundDataManager.Finalize();
    s_SoundArchive.Close();
    nw::snd::SoundSystem::Finalize();

    allocator.Free( s_pMemoryForSoundArchivePlayer );
    allocator.Free( s_pMemoryForStreamBuffer );
    allocator.Free( s_pMemoryForSoundDataManager );
    allocator.Free( s_pMemoryForInfoBlock );
    allocator.Free( s_pMemoryForSoundSystem );

    for ( int i = 0; i < GROUP_FILE_COUNT; i++ )
    {
        allocator.Free(s_pMemoryForGroupFiles[i]);
    }

#if defined( NW_PLATFORM_CAFE )
    allocator.Free( s_pMemoryForFsSoundArchive );
    FSDelClient(s_pFsClient, FS_RET_NO_ERROR);
    allocator.Free(s_pFsClient);
    s_pFsClient = NULL;
    FSShutdown();

    AXQuit();
#endif

    pDemo->FinalizeGraphicsSystem();
    pDemo->Finalize();
    allocator.Free( pDemo );

#if defined ( NW_USE_NINTENDO_SDK )
    nn::fs::Unmount(MOUNT_NAME);
#endif

    return 0;
}
