﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "common.fsid"

#if defined(NW_PLATFORM_WIN32)
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <windows.h>
#endif

#include <nw/demo.h>

#include <nw/snd/util/sndutil_FileReader.h>
#include <nw/snddw.h>

#include "main.h"
#include "common/SampleUtility.h"
#include "common/NwSoundSetupUtility.h"

#if defined(NW_PLATFORM_WIN32)
#define DEMO_WIN
using namespace nw::internal::winext;
#endif

//#define CPU_RENDERING

namespace
{

    const char DEMO_TITLE[] = "snddw";
    const s32 SOUND_HEAP_SIZE = 4 * 1024 * 1024;

    const f32 WINDOW_WIDTH = 1280.0f;
    const f32 WINDOW_HEIGHT = 720.0f;

#if defined( CPU_RENDERING )
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_PPC;
#else
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_DSP;
#endif

    nw::snd::FsSoundArchive     s_SoundArchive;
    nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
    nw::snd::SoundDataManager   s_SoundDataManager;
    nw::snd::SoundHeap          s_SoundHeap;

    nw::snd::SoundHandle s_SoundHandle;
    nw::snd::SoundHandle s_SoundHandleHold;

    // パッド
#if defined( NW_USE_NINTENDO_SDK )
    nn::hid::DebugPadState s_PadStatus;
#else
    PADStatus s_PadStatus;
#endif

    // サウンドデバッグウィンドウフレームワーク
    nw::snd::dw::SoundWindowSystem s_SoundWindowSystem;

    void* s_pFontBinary;

    // ウィンドウ
    nw::snd::dw::VoicesWindow s_VoicesWindow;
    nw::snd::dw::PerformanceWindow s_PerformanceWindow;
    nw::snd::dw::MasterOutputWindow s_MasterOutputWindow;

    void* s_pMemoryForSoundWindowSystem;
    void* s_pMemoryForVoicesWindow;
    void* s_pMemoryForPerformanceWindow;
    void* s_pMemoryForMasterOutputWindow;

    void InitializeNwSound(nw::ut::IAllocator& allocator)
    {
        char soundArchivePath[512];
        snddemo::ConvertToPlatformDependentPath(snddemo::GetCommmonSoundArchivePath(), soundArchivePath);

        snddemo::InitializeSoundSystem(allocator);
        snddemo::InitializeFsSoundArchive(s_SoundArchive, soundArchivePath, allocator);
        snddemo::InitializeSoundDataManager(s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundArchivePlayer(s_SoundArchivePlayer, s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundHeap(s_SoundHeap, SOUND_HEAP_SIZE, allocator);

        // データロード
        if ( ! s_SoundDataManager.LoadData( SEQ_MARIOKART, &s_SoundHeap ) )
        {
            NW_ASSERTMSG( false, "LoadData(SEQ_MARIOKART) failed." );
        }
        if ( ! s_SoundDataManager.LoadData( SE_YOSHI, &s_SoundHeap ) )
        {
            NW_ASSERTMSG( false, "LoadData(SE_YOSHI) failed." );
        }
    }

    void FinalizeNwSound(nw::ut::IAllocator& allocator)
    {
        snddemo::FinalizeSoundArchivePlayer(s_SoundArchivePlayer, allocator);
        snddemo::FinalizeSoundDataManager(s_SoundDataManager, allocator);
        snddemo::FinalizeFsSoundArchive(s_SoundArchive, allocator);
        snddemo::FinalizeSoundHeap(s_SoundHeap, allocator);
        snddemo::FinalizeSoundSystem(allocator);
    }

    //------------------------------------------------------------------------------
    // デバッグウィンドウ
    void InitializeDebugWindow(nw::ut::IAllocator& allocator)
    {
        // フォントバイナリの読み込み
        nw::snd::util::FileReader fileReader;
        nw::snd::util::FileReader::Result result = fileReader.Initialize(
    #if defined(NW_PLATFORM_CAFE)
            snddemo::GetFsClientPointer()
    #endif
        );
        NW_ASSERT(result == nw::snd::util::FileReader::RESULT_SUCCESS);

#if defined(NW_PLATFORM_CAFE)
        result = fileReader.Open("common/fonts/nintendo_NTLG-DB_002.bffnt");
#else
        result = fileReader.Open("common/fonts/nintendo_NTLG-DB_002_Nw4f.bffnt");
#endif
        NW_ASSERT(result == nw::snd::util::FileReader::RESULT_SUCCESS);

        u32 fontBinarySize = fileReader.GetSize();
        s_pFontBinary = allocator.Alloc(fontBinarySize, nw::font::RESOURCE_ALIGNMENT);
        s32 readSize = 0;
        fileReader.Read(s_pFontBinary, fontBinarySize, &readSize);
        NW_ASSERT(fontBinarySize == static_cast<u32>(readSize));

        result = fileReader.Close();
        NW_ASSERT(result == nw::snd::util::FileReader::RESULT_SUCCESS);
        result = fileReader.Finalize();
        NW_ASSERT(result == nw::snd::util::FileReader::RESULT_SUCCESS);

        // SoundWindowSystemの初期化
        {
            u32 memorySize = s_SoundWindowSystem.GetRequiredSize();
            s_pMemoryForSoundWindowSystem = allocator.Alloc(memorySize);

            nw::snd::dw::SoundWindowSystem::Param param;
            param.memory = s_pMemoryForSoundWindowSystem;
            param.memorySize = memorySize;
            param.fontBinary = s_pFontBinary;
            param.fontBinarySize = fontBinarySize;
            param.width = WINDOW_WIDTH;
            param.height = WINDOW_HEIGHT;

            s_SoundWindowSystem.Initialize(param);

            //s_SoundWindowSystem.SetModifierKey(nw::demo::Pad::MASK_R | nw::demo::Pad::MASK_L);
        }

        // VoicesWindowの初期化
        {
            u32 memorySize = s_VoicesWindow.GetRequiredSize();
            s_pMemoryForVoicesWindow = allocator.Alloc(memorySize);
            s_VoicesWindow.Initialize(s_pMemoryForVoicesWindow, memorySize);
        }

        // PerformanceWindowの初期化
        {
            u32 memorySize = s_PerformanceWindow.GetRequiredSize();
            s_pMemoryForPerformanceWindow = allocator.Alloc(memorySize);
            s_PerformanceWindow.Initialize(s_pMemoryForPerformanceWindow, memorySize);
        }

        // MasterOutputWindowの初期化
        {
            u32 memorySize = s_MasterOutputWindow.GetRequiredSize();
            s_pMemoryForMasterOutputWindow = allocator.Alloc(memorySize);
            s_MasterOutputWindow.Initialize(s_pMemoryForMasterOutputWindow, memorySize);
        }

        // ウィンドウを生成
        f32 margin = 5.0f;
        s_SoundWindowSystem.AddWindow(
            margin,
            WINDOW_HEIGHT - s_PerformanceWindow.GetHeight() - margin,
            s_PerformanceWindow
        );
        s_SoundWindowSystem.AddWindow(
            WINDOW_WIDTH - s_MasterOutputWindow.GetWidth() - margin,
            margin,
            s_MasterOutputWindow
        );
        s_SoundWindowSystem.AddWindow(
            WINDOW_WIDTH - s_VoicesWindow.GetWidth() - margin,
            WINDOW_HEIGHT - s_VoicesWindow.GetHeight() - margin,
            s_VoicesWindow
        );
    }

    void FinalizeDebugWindow(nw::ut::IAllocator& allocator)
    {
        s_MasterOutputWindow.Finalize();
        s_PerformanceWindow.Finalize();
        s_VoicesWindow.Finalize();
        s_SoundWindowSystem.Finalize();
        allocator.Free(s_pMemoryForMasterOutputWindow);
        allocator.Free(s_pMemoryForPerformanceWindow);
        allocator.Free(s_pMemoryForVoicesWindow);
        allocator.Free(s_pMemoryForSoundWindowSystem);

        allocator.Free(s_pFontBinary);
    }

    void PrintUsage()
    {
        NW_LOG("-------------------------------------------------------------------------------\n");
        NW_LOG("NintendoWare %s Sample\n", DEMO_TITLE);
        NW_LOG("-------------------------------------------------------------------------------\n");
        NW_LOG("[A]                                         StartSound SEQ  (SEQ_MARIOKART)\n");
        NW_LOG("[X]                                         StartSound STRM (STRM_MARIOKART)\n");
        NW_LOG("[B]                                         Stop Sound\n");
        NW_LOG("[R + ZR]                                    Toggle visibility of all windows\n");
        NW_LOG("[R + Left Stick]                            Move Cursor\n");
        NW_LOG("[R + Right Stick]                           Adjust transparency of the window\n");
        NW_LOG("[R + A(On a Window) + Left Stick]           Move the window\n");
        NW_LOG("[R + X(On a Window) + Left Stick]           Adjust the window size\n");
        NW_LOG("[R + A(On a Window) + UP/DOWN/LEFT/RIGHT]   Move the window \n");
        NW_LOG("[R + Y(On a Window)]                        Change WindowMode\n");
        NW_LOG("[R + UP/DOWN]                               Change ActiveWindow\n");
        NW_LOG("[R + UP/DOWN + B]                           Toggle visibility of selected window\n");
        NW_LOG("[R + A + UP/DOWN]                           Change order of the window\n");
        NW_LOG("-------------------------------------------------------------------------------\n");
    }

    bool Process(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::Pad* pad = pDemo->GetPad();
        pad->GetPadStatus(&s_PadStatus);
        s_SoundWindowSystem.UpdateInputs(s_PadStatus);

        // StartSound / Stop
        if ( pad->IsTrig( nw::demo::Pad::MASK_A ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, SEQ_MARIOKART ).IsSuccess();
            NW_LOG("[SEQ] StartSound(SEQ_MARIOKART) ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_X ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, SE_YOSHI ).IsSuccess();
            NW_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_B ) )
        {
            s_SoundHandle.Stop( 0 );
        }

        // Exit
        if ( pad->IsTrig( nw::demo::Pad::MASK_START ) )
        {
            nw::gfnd::Graphics::GetInstance()->LockDrawContext();
            {
                // 画面のクリア
                pDemo->ClearFrameBuffers();
            }
            nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

            // バッファのスワップ
            pDemo->SwapBuffer();

            return false;
        }

        s_SoundWindowSystem.Update();
        s_SoundArchivePlayer.Update();

        // 描画処理
        {
            nw::gfnd::Graphics::GetInstance()->LockDrawContext();
            {
                // 画面のクリア
                pDemo->ClearFrameBuffers();

                s_SoundWindowSystem.Draw();

#if defined(DEMO_WIN)
                NW_GL_ASSERT();
#endif
            }
            nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

            // バッファのスワップ
            pDemo->SwapBuffer();
        }

        return true;
    }
}

namespace snddemo
{

    void SnddwDemo(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::DefaultAllocator allocator;

        // SDK 層のサウンドの初期化
        snddemo::InitializeSdkSound(RENDERER_SELECT);

        // NW 層のサウンドの初期化
        InitializeNwSound(allocator);

        InitializeDebugWindow(allocator);

        PrintUsage();

        // メインループ
        while ( !pDemo->IsExiting() )
        {
            snddemo::WaitForVBlank(pDemo);

            pDemo->UpdatePad();
            if (!Process(pDemo))
            {
                break;
            }
        }

        FinalizeDebugWindow(allocator);

        // NW 層のサウンドの終了処理
        FinalizeNwSound(allocator);

        // SDK 層のサウンドの終了処理
        snddemo::FinalizeSdkSound();
    }

}
