﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/demo.h>

#include "common/SampleSelector.h"
#include "common/SampleUtility.h"
#include "main.h"

namespace
{

    nw::demo::DemoSystem* CreateDemoSystem(nw::ut::IAllocator& allocator)
    {
        nw::demo::DemoSystem::CreateArg arg;
        arg.allocator = &allocator;
        arg.waitVBlank = 0;

        nw::demo::DemoSystem* pDemo =
            new( allocator.Alloc( sizeof( nw::demo::DemoSystem ) ) )
            nw::demo::DemoSystem( arg );
        pDemo->Initialize();
        pDemo->InitializeGraphicsSystem();

        return pDemo;
    }

    void DestroyDemoSystem(nw::demo::DemoSystem* pDemo, nw::ut::IAllocator& allocator)
    {
        pDemo->FinalizeGraphicsSystem();
        pDemo->Finalize();
        allocator.Free( pDemo );
    }

}

int
NwDemoMain(int argc, char **argv)
{
    NW_UNUSED_VARIABLE(argc);
    NW_UNUSED_VARIABLE(argv);

    enum SampleKind
    {
        SampleKind_Edit,
        SampleKind_Effect,
        SampleKind_ExternalFile,
        SampleKind_Filter,
        SampleKind_FinalMix,
        SampleKind_LabelString,
        SampleKind_LoadGroupManually,
        SampleKind_MemorySoundArchive,
        SampleKind_MultiListener,
        SampleKind_PlayerHeap,
        SampleKind_Simple,
        SampleKind_Sound3d,
        SampleKind_SoundActor,
        SampleKind_Snddw,
#if defined( NW_PLATFORM_CAFE )
        SampleKind_Aac,
        SampleKind_Drc,
        SampleKind_Profile,
        SampleKind_Remote,
        SampleKind_SubCore,
#endif

        SampleKind_Exit,
        SampleKind_Num
    };

    const char* sampleNames[] =
    {
        "edit",
        "effect",
        "externalFile",
        "filter",
        "finalMix",
        "labelString",
        "loadGroupManually",
        "memorySoundArchive",
        "multiListener",
        "playerHeap",
        "simple",
        "sound3d",
        "soundActor",
        "snddw",
#if defined( NW_PLATFORM_CAFE )
        "aac",
        "drc",
        "profile",
        "remote",
        "subCore",
#endif

        "exit sample"
    };

    snddemo::SampleSelector selector;

    nw::demo::DefaultAllocator allocator;

    // DemoSystem の初期化
    nw::demo::DemoSystem* pDemo = CreateDemoSystem(allocator);

    // FS の初期化
    if (!snddemo::InitializeFs(allocator))
    {
        return -1;
    }

    bool isSampleEnd = false;
    while ( !isSampleEnd )
    {
        SampleKind kind = static_cast<SampleKind>(selector.Select(pDemo, sampleNames, SampleKind_Num));

        switch(kind)
        {
        case SampleKind_Edit:
            snddemo::EditDemo(pDemo);
            break;
        case SampleKind_Effect:
            snddemo::EffectDemo(pDemo);
            break;
        case SampleKind_ExternalFile:
            snddemo::ExternalFileDemo(pDemo);
            break;
        case SampleKind_Filter:
            snddemo::FilterDemo(pDemo);
            break;
        case SampleKind_FinalMix:
            snddemo::FinalMixDemo(pDemo);
            break;
        case SampleKind_LabelString:
            snddemo::LabelStringDemo(pDemo);
            break;
        case SampleKind_LoadGroupManually:
            snddemo::LoadGroupManuallyDemo(pDemo);
            break;
        case SampleKind_MemorySoundArchive:
            snddemo::MemorySoundArchiveDemo(pDemo);
            break;
        case SampleKind_MultiListener:
            snddemo::MultiListenerDemo(pDemo);
            break;
        case SampleKind_PlayerHeap:
            snddemo::PlayerHeapDemo(pDemo);
            break;
        case SampleKind_Simple:
            snddemo::SimpleDemo(pDemo);
            break;
        case SampleKind_Sound3d:
            snddemo::Sound3dDemo(pDemo);
            break;
        case SampleKind_SoundActor:
            snddemo::SoundActorDemo(pDemo);
            break;
        case SampleKind_Snddw:
            //snddemo::SnddwDemo(pDemo);
            break;
#if defined( NW_PLATFORM_CAFE )
        case SampleKind_Aac:
            snddemo::AacDemo(pDemo);
            break;
        case SampleKind_Drc:
            snddemo::DrcDemo(pDemo);
            break;
        case SampleKind_Profile:
            snddemo::ProfileDemo(pDemo);
            break;
        case SampleKind_Remote:
            snddemo::RemoteDemo(pDemo);
            break;
        case SampleKind_SubCore:
            snddemo::SubCoreDemo(pDemo);
            break;
#endif

        case SampleKind_Exit:
            isSampleEnd = true;
            break;
        default:
            break;
        }
    }

    // FS の終了処理
    snddemo::FinalizeFs(allocator);

    // DemoSystem の終了処理
    DestroyDemoSystem(pDemo, allocator);

    return 0;
}

