﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "common.fsid"

#include <nw/demo.h>
#include <nw/snd/util/sndutil_FileReader.h>

#include "main.h"
#include "common/SampleUtility.h"
#include "common/NwSoundSetupUtility.h"

//#define CPU_RENDERING

namespace
{

    const char DEMO_TITLE[] = "loadGroupManually";

#if defined( CPU_RENDERING )
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_PPC;
#else
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_DSP;
#endif

    nw::snd::FsSoundArchive     s_SoundArchive;
    nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
    nw::snd::SoundDataManager   s_SoundDataManager;

    nw::snd::SoundHandle s_SoundHandle;

    const int GROUP_FILE_COUNT = 4;
    void* s_pMemoryForGroupFiles[GROUP_FILE_COUNT];
    size_t s_GroupFileSizes[GROUP_FILE_COUNT];

    bool s_SetGroupFile = true;

    const char* GROUP_FILES_PATH[] =
    {
        "snd/common/GROUP_USER.bfgrp",
        "snd/common/GROUP_USER_AUTO_WARC.bfgrp",
        "snd/common/GROUP_USER_INDIVIDUAL_WARC.bfgrp",
        "snd/common/GROUP_USER_TEST_PLAYERHEAP.bfgrp"
    };

    void InitializeNwSound(nw::ut::IAllocator& allocator)
    {
        char soundArchivePath[512];
        snddemo::ConvertToPlatformDependentPath(snddemo::GetCommmonSoundArchivePath(), soundArchivePath);

        snddemo::InitializeSoundSystem(allocator);
        snddemo::InitializeFsSoundArchive(s_SoundArchive, soundArchivePath, allocator);
        snddemo::InitializeSoundDataManager(s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundArchivePlayer(s_SoundArchivePlayer, s_SoundDataManager, s_SoundArchive, allocator);
    }

    void FinalizeNwSound(nw::ut::IAllocator& allocator)
    {
        snddemo::FinalizeSoundArchivePlayer(s_SoundArchivePlayer, allocator);
        snddemo::FinalizeSoundDataManager(s_SoundDataManager, allocator);
        snddemo::FinalizeFsSoundArchive(s_SoundArchive, allocator);
        snddemo::FinalizeSoundSystem(allocator);
    }

    void LoadGroupFile(nw::ut::IAllocator& allocator)
    {
        nw::snd::util::FileReader fileReader;
        nw::snd::util::FileReader::Result result = fileReader.Initialize(
#if defined(NW_PLATFORM_CAFE)
            snddemo::GetFsClientPointer()
#endif
        );
        NW_ASSERT(result == nw::snd::util::FileReader::RESULT_SUCCESS);

        for ( int i = 0; i < GROUP_FILE_COUNT; i++ )
        {
            char groupFilePath[512];
            snddemo::ConvertToPlatformDependentPath(GROUP_FILES_PATH[i], groupFilePath);
            fileReader.Open( groupFilePath );
            s_GroupFileSizes[i] = fileReader.GetSize();
            s_pMemoryForGroupFiles[i] = allocator.Alloc(s_GroupFileSizes[i], 64);

            int readSize = 0;
            fileReader.Read( s_pMemoryForGroupFiles[i], s_GroupFileSizes[i], &readSize );
            NW_LOG("[%d] Read size(%7d) path(%s)\n", i, readSize, GROUP_FILES_PATH[i]);

            fileReader.Close();
        }

        fileReader.Finalize();
    }

    void UnloadGroupFile(nw::ut::IAllocator& allocator)
    {
        for ( int i = 0; i < GROUP_FILE_COUNT; i++ )
        {
            allocator.Free(s_pMemoryForGroupFiles[i]);
        }
    }

    void PrintUsage()
    {
        NW_LOG("----------------------------------------\n");
        NW_LOG("NintendoWare %s Sample\n", DEMO_TITLE);
        NW_LOG("----------------------------------------\n");
        NW_LOG("[A] SEQ_MARIOKART\n");
        NW_LOG("[X] SEQ_TEST\n");
        NW_LOG("[Y] SE_WIHAHO\n");
        NW_LOG("[L] SE_YOSHI_PLAYERHEAP_FOR_USERGROUP\n");
        NW_LOG("[R] Set/Unset GroupFile\n");
        NW_LOG("[B] Stop Sound\n");
#if defined( NW_PLATFORM_CAFE )
        NW_LOG("[HOME]  Exit Application\n");
#elif defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        NW_LOG("[S] Exit Application\n");
#endif
        NW_LOG("---------------------------------------\n");
    }

    bool Process(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::Pad* pad = pDemo->GetPad();

        // StartSound / Stop
        if ( pad->IsTrig( nw::demo::Pad::MASK_A ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, SEQ_MARIOKART ).IsSuccess();
            NW_LOG("[SEQ_MARIOKART] ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_X ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, SEQ_TEST ).IsSuccess();
            NW_LOG("[SEQ_TEST]      ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_Y ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, SE_WIHAHO ).IsSuccess();
            NW_LOG("[SE_WIHAHO]     ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_L ) )
        {
            s_SoundHandle.Stop( 0 );
            // [ユーザー管理] のグループにのみ含まれるデータは、
            // プレイヤーヒープを利用するサウンドであっても再生できません。
            // (ただし、プレイヤーヒープ利用時のサウンドは、
            //  データの参照解決が遅延されるので、
            //  サウンドインスタンスさえ確保出来れば、
            //  StartSound().IsSuccess() 自体は true を返します)
            //
            // GROUP_USER_TEST_PLAYERHEAP.bfgrp をロードし、かつ、
            // SetFileAddressInGroupFile を呼び出しておく必要があります。
            bool result = s_SoundArchivePlayer
                .StartSound(&s_SoundHandle, SE_YOSHI_PLAYERHEAP_FOR_USERGROUP ).IsSuccess();
            NW_LOG("[SE_YOSHI_PLAYERHEAP_FOR_USERGROUP] ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_B ) )
        {
            s_SoundHandle.Stop( 0 );
        }

        // グループファイルの登録と登録解除
        if ( pad->IsTrig( nw::demo::Pad::MASK_R ) )
        {
            if ( s_SetGroupFile )
            {
                // ロード済みグループファイルの登録
                for ( int i = 0; i < GROUP_FILE_COUNT; i++ )
                {
                    s_SoundDataManager.SetFileAddressInGroupFile(
                            s_pMemoryForGroupFiles[i], s_GroupFileSizes[i]);
                }
                NW_LOG("Set GroupFile Address\n");
            }
            else
            {
                // ロード済みグループファイルの登録解除
                for ( int i = 0; i < GROUP_FILE_COUNT; i++ )
                {
                    s_SoundDataManager.ClearFileAddressInGroupFile(
                            s_pMemoryForGroupFiles[i],
                            s_GroupFileSizes[i]);
                }
                NW_LOG("Unset GroupFile Address\n");
            }
            s_SetGroupFile = !s_SetGroupFile;
        }

        // Exit
        if ( pad->IsTrig( nw::demo::Pad::MASK_START ) ) {
            return false;
        }

        s_SoundArchivePlayer.Update();

        return true;
    }
}

namespace snddemo
{

    void LoadGroupManuallyDemo(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::DefaultAllocator allocator;

        // SDK 層のサウンドの初期化
        snddemo::InitializeSdkSound(RENDERER_SELECT);

        // NW 層のサウンドの初期化
        InitializeNwSound(allocator);

        LoadGroupFile(allocator);

        PrintUsage();

        s_SetGroupFile = true;

        // メインループ
        while ( !pDemo->IsExiting() )
        {
            snddemo::WaitForVBlank(pDemo);

            pDemo->UpdatePad();
            if (!Process(pDemo))
            {
                break;
            }
        }

        UnloadGroupFile(allocator);

        // NW 層のサウンドの終了処理
        FinalizeNwSound(allocator);

        // SDK 層のサウンドの終了処理
        snddemo::FinalizeSdkSound();
    }

}
