﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*

  CustomControlデモ：

  ビューアを拡張してユーザ独自のコントロールをプレビューできる環境を作ります。
  ユーザ独自のコントロールの例としてスクロールバーを実装しています。


  【実行方法】

  ・本プログラムを実機用またはPC用にビルドし、起動します。

  ・LayoutEditor で以下の操作を行います。
    ・SampleData/Layout/flyt/scrollBar/ScrollBar.flytを開きます。
    ・[ツール]メニューの[ビューアターゲット]を適切に選択します。
    ・[ツール]メニューの[ビューアに転送]を実行します。

  ・本プログラムでサンプルデータが表示されます。
    矢印ボタンを押すとサムボタンがスクロールします。
    サムボタンはドラッグできます。

    本プログラムは、HostIO 機能を利用します。
    Release ビルドでは動作しないので注意してください。

  【概要】

  レイアウトの構築時に、カスタマイズした ControlCreator を指定することで、
  ユーザ独自のコントロールを組み込むことができます。

  本プログラムではレイアウトの管理に nw::vwrlyt::Viewer クラスを使用しているので、
  Viewer::SetControlCreator() でユーザ独自の ControlCreator を指定します。

  LayoutEditor からプレビュー要求が来ると、上記で指定した ControlCreator が
  Layout::Build() に渡されます。


  【レイアウトデータ解説】

  データは SampleData/Layout/flyt/scrollBar/ フォルダに置かれています。

  ・ScrollBar.flct
    スクロールバーのための独自コントロールの定義ファイルです。
    HScrollBarとVScrollBarの2つの独自コントロールを定義しています。

  ・ScrollBar.flyt
    水平、垂直スクロールバーを利用したサンプルデータです。

    <ペイン階層>
        RootPane
          + body_HScrollBar : Parts -- parts_HScrollBar.flyt
          + body_VScrollBar : Parts -- parts_VScrollBar.flyt
          + Window_00 : Window

  ・parts_HScrollBar.flyt
    水平スクロールバーの部品レイアウトです。
    [コントロールの種類]に独自コントロールの"HScrollBar"を指定しています。

    <ペイン階層>
        RootPane
          + ResizeRoot
              + Minus : Parts -- parts_ScrollArrow.flyt
                矢印ボタンです。
              + Plus : Parts -- parts_ScrollArrow.flyt
                矢印ボタンです。
              + ScrollBox : Window
                サムボタンの移動範囲の見た目を定義します。
                  + ScrollArea : Null
                    サムボタンの移動範囲を定義します
                      + Thumb : Window
                        サムボタンの見た目を定義します。
                        ハイライト、押し下げ動作のアニメーションがついています。
                      + Hit : Bounding
                        サムボタンの当たりを定義します。

    矢印ボタンには parts_ScrollArrow 部品レイアウトを使用します。
    UV設定を上書きして、矢印の向きを変更しています。

    部品ペインのサイズ倍率に応じてスクロールバーが適切に拡縮されるように
    以下のような設定を行っています。

        ResizeRoot
          [部品倍率の影響]を"サイズをスケール倍に設定"に指定。
          ボディレイアウトでの部品のサイズと一致するように拡縮されます。

        Plus, Minus
          [親ペインでの原点位置]を"右", "左"に指定して、ResizeRootの拡縮に
          位置が追従するようにします。
          [部品倍率の影響]を"無視"に指定して、サイズが変化しないようにします。

        ScrollBox, ScrollArea
          [部品倍率の影響]を"自分を含む部品外枠との距離を維持"に設定して、
          ResizeRootと同じように拡縮するようにしています。

  ・parts_VScrollBar.flyt
    垂直スクロールバーの部品レイアウトです。
    [コントロールの種類]に独自コントロールの"VScrollBar"を指定しています。
    構造はparts_HScrollBarと同様です。

  ・parts_ScrollArrow.flyt
    矢印ボタンのための部品レイアウトです。
    [コントロールの種類]は標準の"NormalButton"です。
    矢印ボタンのハイライト、押し下げ動作のアニメーションがついています。


  【クラス解説】

  ・CustomControlCreator クラス
    nw::ctrl::DefaultControlCreator の派生クラスです。
    独自コントロールを生成します。

  ・HScrollBar クラス
    nw::ctrl::AnimButton の派生クラスです。
    水平スクロールバーを実現する独自コントロールです。

  ・VScrollBar クラス
    HScrollBarの派生クラスです。
    垂直スクロールバーを実現する独自コントロールです。
    サムボタンの移動方向を縦に変更します。

  コントロールの当たり判定ではペインの回転をサポートしないので、
  水平スクロールバーと垂直スクロールバーを別のクラスとしました。

  部品ペインのサイズに応じて判断する、拡張ユーザデータで向きを指定するなど、
  別の実装方法も考えられます。

 *---------------------------------------------------------------------------*/

#define NW_CONSOLE_ENABLE

#include <nw/types.h>

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
#include <winext/cafe/env.h>
#include <winext/cafe/fs.h>
#else
#include <cafe/env.h>
#include <cafe/fs.h>
#endif
#include <nw/demo.h>
#include <nw/gfnd.h>
#include <nw/dw.h>

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
#include <nw/dev/win/dev_PadWin.h>
#include <nw/dev/win/dev_WPadWin.h>
#include <nw/dev/win/dev_VPadWin.h>
#include <nw/dev/win/dev_MouseWin.h>
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <windows.h>
#else
#include <nw/dev/cafe/dev_PadCafe.h>
#include <nw/dev/cafe/dev_WPadCafe.h>
#include <nw/dev/cafe/dev_VPadCafe.h>
#include <nw/dev/cafe/dev_MouseCafe.h>
#endif
#include <nw/lyt.h>
#include <nw/ctrl.h>
#include <nw/vwrlyt.h>

namespace {

const u32 SCREEN_WIDTH = 1280;
const u32 SCREEN_HEIGHT = 720;

#if defined(NW_PLATFORM_CAFE)
const s32 PAD_CHANNEL = 0;
const s32 WPAD_CHANNEL = 0;
#else
const s32 JOYPAD_NUMBER = 0;
#endif

#if defined(NW_PLATFORM_CAFE) && defined(NW_MCS_ENABLE)
#define USE_MCS_INPUT
#endif

#if defined(USE_MCS_INPUT)
const u32 MCS_INPUT_DEVICE_BUF_SIZE = 64 * 1024; //!< mcsHID のバッファサイズです。
u8* s_InputDeviceBuf = NULL;                     //!< mcsHID のバッファです。
#endif

#if defined(NW_PLATFORM_CAFE)
static const char* const DEMO_FONT_FILENAME = "nintendo_NTLG-DB_002.bffnt";
#else
static const char* const DEMO_FONT_FILENAME = "nintendo_NTLG-DB_002_Nw4f.bffnt";
#endif

//-----------------------------------------------------------------------------
//! @brief        メモリの初期化を実行します。
//!
//! @param[out]   heap          メインメモリのヒープです。
//-----------------------------------------------------------------------------
void
InitializeMemory( nw::ut::MemoryAllocator* heap )
{
    const u32 MEM2_SIZE         = 256 * 1024 * 1024;

  #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    void* memory = malloc( MEM2_SIZE );
  #else
    void* memory = MEMAllocFromDefaultHeap( MEM2_SIZE );
  #endif

    heap->Initialize( memory, MEM2_SIZE );
}

//-----------------------------------------------------------------------------
//! @brief 水平スクロールバー全体を統括するカスタムコントロールです。
//!
//! @details
//! 水平スクロールバーはおもに以下の部品から構成されます。
//!
//! @li Plusボタン -- 押されるとThumbボタンをプラス方向へ移動させます。
//! @li Minusボタン -- 押されるとThumbボタンをマイナス方向へ移動させます。
//! @li Thumbボタン -- ドラッグできます。
//!
//! Thumbボタンが押されている間はドラッグ状態となり、
//! Thumbボタンをポインタ位置に追従させます。
//!
//! 矢印ボタン(Plus, Minus)が押されるとスクロール状態となり、
//! Thumbボタンをスクロールさせます。
//!
//! 矢印ボタンには標準のボタンコントロールを利用します。
//! 状態変化を通知するコールバックを設定し、ボタンの押し下げを検知します。
//-----------------------------------------------------------------------------
class HScrollBar : public nw::ctrl::AnimButton
{
private:
    typedef nw::ctrl::AnimButton Base;
    typedef HScrollBar Self;

protected:
    enum ScrollState
    {
        SCROLL_IDLE,
        SCROLL_PLUS,
        SCROLL_MINUS,
        SCROLL_DRAG,
    };

public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO(Base);

    HScrollBar()
        : m_pThumbPane(NULL)
        , m_TotalLength(100.f)
        , m_PageLength(20.f)
        , m_ScrollLength(20.f)
        , m_CurrentPosition(0.f)
        , m_ScrollFrames(10.f)
        , m_CurrentFrame(0.f)
        , m_ScrollState(SCROLL_IDLE)
    {
        this->SetDragMode(nw::ctrl::AnimButton::DRAG_MODE_ON_DOWN);
    }

    //!-------------------------------------------------------------------------
    //! @brief コントロールを構築します。
    //!
    //! @param[in] controlSrc       ControlSrc です。
    //! @param[in] layout           Layout です。
    //! @param[in] pButtonGroup     ButtonGroup です。
    //!-------------------------------------------------------------------------
    void Build(const nw::lyt::ControlSrc& controlSrc, nw::lyt::Layout* layout, nw::ctrl::ButtonGroup* pButtonGroup)
    {
        Base::Build(controlSrc, layout);

        nw::lyt::Pane* pPartsPane = layout->GetRootPane();
        NW_ASSERT_NOT_NULL(pPartsPane);
        if (!pPartsPane)
        {
            return;
        }

        // ペインの取得
        m_pThumbPane = nw::ut::DynamicCast<nw::lyt::Window*>(pPartsPane->FindPaneByName(controlSrc.FindFunctionalPaneName("Thumb")));

        // 相互作用の設定
        nw::ctrl::AnimButton* pPlusButton = pButtonGroup->FindButtonByNameReverse(controlSrc.FindFunctionalPaneName("Plus"));
        if (pPlusButton)
        {
            pPlusButton->SetStateChangeCallback(ButtonDownCallback<&Self::PlusDown>, this);
        }

        nw::ctrl::AnimButton* pMinusButton = pButtonGroup->FindButtonByNameReverse(controlSrc.FindFunctionalPaneName("Minus"));
        if (pMinusButton)
        {
            pMinusButton->SetStateChangeCallback(ButtonDownCallback<&Self::MinusDown>, this);
        }
    }

protected:
    //!-------------------------------------------------------------------------
    //! @brief 毎フレームの更新処理を行います。
    //!-------------------------------------------------------------------------
    /* override */
    virtual void Update()
    {
        Base::Update();

        switch (m_ScrollState)
        {
        case SCROLL_IDLE:
            break;

        case SCROLL_PLUS:
        case SCROLL_MINUS:
            this->UpdateScroll();
            break;

        case SCROLL_DRAG:
            this->UpdateDrag();
            break;
        }

        this->UpdateThumb();
    }

    //!-------------------------------------------------------------------------
    //! @brief サムがスクロールするように現在の表示位置を更新します。
    //!-------------------------------------------------------------------------
    virtual void UpdateScroll()
    {
        if (m_ScrollState == SCROLL_PLUS)
        {
            m_CurrentPosition = m_StartPosition + m_ScrollLength * m_CurrentFrame / m_ScrollFrames;
        }
        else
        {
            m_CurrentPosition = m_StartPosition - m_ScrollLength * m_CurrentFrame / m_ScrollFrames;
        }

        m_CurrentPosition = nw::ut::Clamp(m_CurrentPosition, 0.f, m_TotalLength - m_PageLength);

        m_CurrentFrame += 1.0f;
        if (m_CurrentFrame > m_ScrollFrames)
        {
            m_ScrollState = SCROLL_IDLE;
        }
    }

    //!-------------------------------------------------------------------------
    //! @brief サムがポインタ位置にくるように現在の表示位置を更新します。
    //!-------------------------------------------------------------------------
    virtual void UpdateDrag()
    {
        f32 hitWidth = m_HitBoxTopRight.x - m_HitBoxBottomLeft.x;
        if (hitWidth >= 1.f)
        {
            f32 hitCenter = (m_HitBoxTopRight.x + m_HitBoxBottomLeft.x) / 2;
            f32 diff = m_HitPoint.x - hitCenter;

            m_CurrentPosition = nw::ut::Clamp(m_CurrentPosition + diff * m_PageLength / hitWidth, 0.f, m_TotalLength - m_PageLength);
            m_CurrentPosition = nw::math::FFloor(m_CurrentPosition + 0.5f);
        }
    }

    //!-------------------------------------------------------------------------
    //! @brief 現在の表示位置にあわせてサムの位置と幅を調整します。
    //!-------------------------------------------------------------------------
    virtual void UpdateThumb()
    {
        if (m_HitPane)
        {
            nw::lyt::Pane* pScrollArea = m_HitPane->GetParent();
            if (pScrollArea)
            {
                // スクロールボックスのペイン中心は左上であると仮定します。

                nw::lyt::Size areaSize = pScrollArea->GetSize();
                nw::lyt::Size hitSize = m_HitPane->GetSize();
                nw::math::VEC3 hitTrans = m_HitPane->GetTranslate();

                // サムの幅をページサイズに合わせます。
                hitSize.width = areaSize.width * m_PageLength / m_TotalLength;
                if (m_pThumbPane)
                {
                    nw::lyt::WindowFrameSize frameSize;
                    m_pThumbPane->CalcFrameSize(&frameSize);
                    hitSize.width = nw::ut::Max(hitSize.width, frameSize.l + frameSize.r);
                }

                // スクロールボックスのX方向がスクロールバーのプラスの移動方向になります。
                hitTrans.x = m_CurrentPosition * areaSize.width / m_TotalLength;

                NW_ASSERT(m_HitPane->GetParentRelativePositionH() == nw::lyt::HORIZONTALPOSITION_LEFT);
                NW_ASSERT(m_HitPane->GetBasePositionH() == nw::lyt::HORIZONTALPOSITION_LEFT);
                m_HitPane->SetSize(hitSize);
                m_HitPane->SetTranslate(hitTrans);

                if (m_pThumbPane)
                {
                    NW_ASSERT(m_pThumbPane->GetParentRelativePositionH() == nw::lyt::HORIZONTALPOSITION_LEFT);
                    NW_ASSERT(m_pThumbPane->GetBasePositionH() == nw::lyt::HORIZONTALPOSITION_LEFT);
                    m_pThumbPane->SetSize(hitSize);
                    m_pThumbPane->SetTranslate(hitTrans);
                }
            }
        }
    }

    //!-------------------------------------------------------------------------
    //! @brief ドラッグ開始時のポインタの位置を記録します。
    //!-------------------------------------------------------------------------
    /* override */
    virtual void InitDragPosition(const nw::math::VEC2& point)
    {
        m_HitPoint = point;
    }

    //!-------------------------------------------------------------------------
    //! @brief ドラッグ中のポインタの位置を記録します。
    //!-------------------------------------------------------------------------
    /* override */
    virtual void UpdateDragPosition(const nw::math::VEC2* point)
    {
        if (point)
        {
            m_HitPoint = *point;
        }
    }

    //!-------------------------------------------------------------------------
    //! @brief サムのドラッグを開始します。
    //!-------------------------------------------------------------------------
    /* override */
    virtual void StartDown()
    {
        Base::StartDown();
        m_ScrollState = SCROLL_DRAG;
    }

    //!-------------------------------------------------------------------------
    //! @brief サムのドラッグを終了します。
    //!-------------------------------------------------------------------------
    /* override */
    virtual void FinishCancel()
    {
        Base::FinishCancel();
        m_ScrollState = SCROLL_IDLE;
    }

private:
    //!-------------------------------------------------------------------------
    //! @brief ボタンが押されたときにメンバー関数を呼び出すコールバック関数です。
    //!-------------------------------------------------------------------------
    template <void (Self::*MemberFunc)()>
    static void ButtonDownCallback(
        nw::ctrl::AnimButton* button,
        nw::ctrl::ButtonBase::State prevState,
        nw::ctrl::ButtonBase::State nextState,
        void* param)
    {
        NW_UNUSED_VARIABLE(button);
    	NW_UNUSED_VARIABLE(prevState);

        if (nextState == button->STATE_DOWN)
        {
            (static_cast<Self*>(param)->*MemberFunc)();
        }
    }

    //!-------------------------------------------------------------------------
    //! @brief サムボタンをプラス方向に移動します。
    //!-------------------------------------------------------------------------
    void PlusDown()
    {
        m_ScrollState = SCROLL_PLUS;
        m_StartPosition = nw::math::FFloor(m_CurrentPosition + 0.5f);
        m_CurrentFrame = 0.f;
    }

    //!-------------------------------------------------------------------------
    //! @brief サムボタンをマイナス方向に移動します。
    //!-------------------------------------------------------------------------
    void MinusDown()
    {
        m_ScrollState = SCROLL_MINUS;
        m_StartPosition = nw::math::FFloor(m_CurrentPosition + 0.5f);
        m_CurrentFrame = 0.f;
    }

protected:
    nw::lyt::Window* m_pThumbPane;

    //! @brief コンテント全体の長さです。
    f32 m_TotalLength;

    //! @brief コンテントのうち、表示されているエリアの長さです。
    f32 m_PageLength;

    //! @brief プラス、マイナス・ボタンが押されたときに何単位スクロールするかを表します。
    f32 m_ScrollLength;

    //! @brief 現在の表示エリアの位置です。
    mutable f32 m_CurrentPosition;

    //! @brief スクロールにかかるフレーム数です。
    f32 m_ScrollFrames;

    //! @brief 現在のフレームです。
    f32 m_CurrentFrame;

    //! @brief スクロール開始時の表示エリアの位置です。
    f32 m_StartPosition;

    //! @brief スクロールバーの状態です。
    ScrollState m_ScrollState;

    //! @brief 当たり判定時のポインタの位置です。
    mutable nw::math::VEC2 m_HitPoint;
};

//-----------------------------------------------------------------------------
//! @brief 垂直スクロールバー全体を統括するカスタムコントロールです。
//!
//! @details
//! HScrollBar に対して、処理を縦方向に変更する差分だけを定義します。
//-----------------------------------------------------------------------------
class VScrollBar : public HScrollBar
{
private:
    typedef HScrollBar Base;
    typedef VScrollBar Self;

public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO(Base);

protected:
    //!-------------------------------------------------------------------------
    //! @brief サムがポインタ位置にくるように現在の表示位置を更新します。
    //!-------------------------------------------------------------------------
    /* override */
    virtual void UpdateDrag()
    {
        f32 hitHeight = m_HitBoxTopRight.y - m_HitBoxBottomLeft.y;
        if (hitHeight >= 1.f)
        {
            f32 hitCenter = (m_HitBoxTopRight.y + m_HitBoxBottomLeft.y) / 2;
            f32 diff = m_HitPoint.y - hitCenter;

            m_CurrentPosition = nw::ut::Clamp(m_CurrentPosition - diff * m_PageLength / hitHeight, 0.f, m_TotalLength - m_PageLength);
            m_CurrentPosition = nw::math::FFloor(m_CurrentPosition + 0.5f);
        }
    }

    //!-------------------------------------------------------------------------
    //! @brief 現在の表示位置にあわせてサムの位置と幅を調整します。
    //!-------------------------------------------------------------------------
    /* override */
    virtual void UpdateThumb()
    {
        if (m_HitPane)
        {
            nw::lyt::Pane* pScrollArea = m_HitPane->GetParent();
            if (pScrollArea)
            {
                nw::lyt::Size areaSize = pScrollArea->GetSize();
                nw::lyt::Size hitSize = m_HitPane->GetSize();
                nw::math::VEC3 hitTrans = m_HitPane->GetTranslate();

                // サムの高さをページサイズに合わせます。
                hitSize.height = areaSize.height * m_PageLength / m_TotalLength;
                if (m_pThumbPane)
                {
                    nw::lyt::WindowFrameSize frameSize;
                    m_pThumbPane->CalcFrameSize(&frameSize);
                    hitSize.height = nw::ut::Max(hitSize.height, frameSize.t + frameSize.b);
                }

                // スクロールボックスの-Y方向がスクロールバーのプラスの移動方向になります。
                hitTrans.y = -m_CurrentPosition * areaSize.height / m_TotalLength;

                NW_ASSERT(m_HitPane->GetParentRelativePositionV() == nw::lyt::VERTICALPOSITION_TOP);
                NW_ASSERT(m_HitPane->GetBasePositionV() == nw::lyt::VERTICALPOSITION_TOP);
                m_HitPane->SetSize(hitSize);
                m_HitPane->SetTranslate(hitTrans);

                if (m_pThumbPane)
                {
                    NW_ASSERT(m_pThumbPane->GetParentRelativePositionV() == nw::lyt::VERTICALPOSITION_TOP);
                    NW_ASSERT(m_pThumbPane->GetBasePositionV() == nw::lyt::VERTICALPOSITION_TOP);
                    m_pThumbPane->SetSize(hitSize);
                    m_pThumbPane->SetTranslate(hitTrans);
                }
            }
        }
    }
};

//-----------------------------------------------------------------------------
//! @brief カスタムコントロールを生成するクラスです。
//!
//! @details
//! スクロールバーのための独自コントロールを生成します。
//! それ以外のコントロールの生成は DefaultControlCreator に任せます。
//-----------------------------------------------------------------------------
class CustomControlCreator : public nw::ctrl::DefaultControlCreator
{
private:
    typedef nw::ctrl::DefaultControlCreator Base;

protected:
    //!-------------------------------------------------------------------------
    //! @brief コントロールを作成します。
    //!
    //! @param[in] controlSrc   ControlSrc です。
    //! @param[in] layout       Layout です。
    //!
    //! @details
    //! スクロールバーを構成する部品にはカスタムコントロールを作成します。
    //!-------------------------------------------------------------------------
    /* override */
    virtual void CreateControl(const nw::lyt::ControlSrc& controlSrc, nw::lyt::Layout* layout)
    {
        NW_ASSERT_NOT_NULL(this->GetButtonGroup());

        if (!this->GetButtonGroup())
        {
            return;
        }

        nw::ctrl::AnimButton* button = NULL;

        if (std::strcmp("HScrollBar", controlSrc.GetControlName()) == 0)
        {
            button = nw::lyt::Layout::NewObj<HScrollBar>();
            static_cast<HScrollBar*>(button)->Build(controlSrc, layout, this->GetButtonGroup());
        }
        else if (std::strcmp("VScrollBar", controlSrc.GetControlName()) == 0)
        {
            button = nw::lyt::Layout::NewObj<VScrollBar>();
            static_cast<VScrollBar*>(button)->Build(controlSrc, layout, this->GetButtonGroup());
        }

        if (button)
        {
            this->GetButtonGroup()->GetButtonList().PushBack(button);
        }
        else
        {
            Base::CreateControl(controlSrc, layout);
        }
    }
};

//-----------------------------------------------------------------------------
//! @brief        デモシステムです。
//-----------------------------------------------------------------------------
class TestDemoSystem : public nw::demo::DemoSystemDRC
{
public:
    typedef nw::ut::FixedSafeString<NW_DEV_FS_MAX_FULLPATH_SIZE> PathString;

    //! @brief コンストラクタに渡す生成パラメータです。
    struct CreateArg : public nw::demo::DemoSystemDRC::CreateArg
    {
        //! NINTENDO_SDK_ROOT環境変数が示すディレクトリのFSにおけるパスです。
        const char* nwRootPath;

#if defined(NW_PLATFORM_CAFE)
        //! レイアウトの描画に使用するシェーダのパスです。
        const char* lytShaderPath;
#endif

        CreateArg()
            : nwRootPath(NULL)
#if defined(NW_PLATFORM_CAFE)
            , lytShaderPath(NULL)
#endif
        {}
    };

    //-------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //-------------------------------------------------------------------------
    /* ctor */
    TestDemoSystem(const CreateArg& arg)
      : nw::demo::DemoSystemDRC( arg )
      , m_Arg(arg)
    {
        NW_ASSERT(s_pInstance == NULL);
        s_pInstance = this;
    }

    //-------------------------------------------------------------------------
    //! @brief        nw::dev::FileDeviceManagerを初期化します。
    //-------------------------------------------------------------------------
    static void InitFileDeviceManager(nw::ut::IAllocator* pAllocator)
    {
        nw::dev::FileDeviceManager::CreateArg createArg;
        createArg.allocator = pAllocator;
        createArg.useMount = true;
        nw::dev::FileDeviceManager* pFileSystem = NW_INSTANCE(nw::dev::FileDeviceManager);
        pFileSystem->Initialize(createArg);
    }

    //-------------------------------------------------------------------------
    //! @brief        nw::dev::FileDeviceManagerを解放します。
    //-------------------------------------------------------------------------
    static void FinalizeFileDeviceManager()
    {
        nw::dev::FileDeviceManager* pFileSystem = NW_INSTANCE(nw::dev::FileDeviceManager);
        pFileSystem->Finalize();
    }

    //-------------------------------------------------------------------------
    //! @brief        NINTENDO_SDK_ROOT環境変数の値を取得します。
    //!
    //! @param[out] nwRoot  値が格納されます。
    //!
    //! @return       適切なパスが取得できた場合には true を返します。
    //!
    //! @details
    //! 環境変数の値を FS のパスに変換して nwRoot に格納します。
    //-------------------------------------------------------------------------
    static bool GetNwRoot(nw::ut::BufferedSafeString& nwRoot)
    {
#if defined(NW_VWRLYT_ENABLE)
        PathString buff;

        if (0 != ENVGetEnvironmentVariable("NINTENDO_SDK_ROOT", buff.getBuffer(), buff.getBufferSize()))
        {
            return false;
        }

        nw::dev::FileDeviceManager* pFileSystem = NW_INSTANCE(nw::dev::FileDeviceManager);
        return nw::vwrlyt::Viewer::ConvertHostToFsPath(&nwRoot, pFileSystem->GetFSMountPath(), buff.c_str());
#else
        NW_UNUSED_VARIABLE(nwRoot);
        return false;
#endif
    }

    //-------------------------------------------------------------------------
    //! @brief        決め打ちのサンプルデータを読み込みます。
    //-------------------------------------------------------------------------
    void LoadSampleData();

    //-------------------------------------------------------------------------
    //! @brief        デモの初期化処理を行います。
    //-------------------------------------------------------------------------
    virtual void ProcInit()
    {
        nw::dev::FileDeviceManager* pFileSystem = NW_INSTANCE(nw::dev::FileDeviceManager);

#if defined( NW_PLATFORM_CAFE )
        // 入力デバイスを初期化します。
        {
            nw::dev::PadDeviceCafe::GetInstance()->Initialize();
            nw::dev::WPadDeviceCafe::GetInstance()->Initialize( this->GetAllocator() );
        }

        // パッドを初期化します。
        {
            // デバッグ用パッドにデバイスを設定します。
            m_Pad = new( m_Arg.allocator->Alloc( sizeof( nw::dev::PadCafe ) ) ) nw::dev::PadCafe( PAD_CHANNEL );
            m_Pad->SetPadDevice( nw::dev::PadDeviceCafe::GetInstance() );
            // WPad にデバイスを設定します。
            m_WPad = new( m_Arg.allocator->Alloc( sizeof( nw::dev::WPadCafe ) ) ) nw::dev::WPadCafe( WPAD_CHANNEL );
            m_WPad->SetWPadDevice( nw::dev::WPadDeviceCafe::GetInstance() );
            // VPad にデバイスを設定します。
            m_VPad = new( m_Arg.allocator->Alloc( sizeof( nw::dev::VPadCafe ) ) ) nw::dev::VPadCafe();
            m_VPad->SetVPadDevice( nw::dev::VPadDeviceCafe::GetInstance() );
            // Mouse にデバイスを設定します。
            m_Mouse = new( m_Arg.allocator->Alloc( sizeof( nw::dev::MouseCafe ) ) ) nw::dev::MouseCafe();
            m_Mouse->SetKeyboardMouseDevice( nw::dev::KeyboardMouseDeviceCafe::GetInstance() );
            m_Mouse->SetPointerBound( nw::math::VEC2( 0.f, 0.f ), nw::math::VEC2( static_cast<f32>( m_Arg.width ), static_cast<f32>( m_Arg.height ) ) );
        }

#else
        // 入力デバイスを初期化します。
        {
            // ジョイパッドデバイスを初期化します。
            nw::dev::JoyPadDeviceWin::GetInstance()->Initialize();

            // キーボード・マウスデバイスを初期化します。
            nw::dev::KeyboardMouseDeviceWin::GetInstance()->Initialize();
            // マウス入力を取得するウィンドウのハンドルを設定します。
            nw::dev::KeyboardMouseDeviceWin::GetInstance()->SetMainWindowHandle( this->GetWindowHandle() );
        }

        // パッドを初期化します。
        {
            // デバッグ用パッドにデバイスを設定します。
            m_Pad = new( m_Arg.allocator->Alloc( sizeof( nw::dev::PadWin ) ) ) nw::dev::PadWin( JOYPAD_NUMBER );
            m_Pad->SetKeyboardMouseDevice( nw::dev::KeyboardMouseDeviceWin::GetInstance() );
            m_Pad->SetJoyPadDevice( nw::dev::JoyPadDeviceWin::GetInstance() );

            // WPad にデバイスを設定します。
            m_WPad = new( m_Arg.allocator->Alloc( sizeof( nw::dev::WPadWin ) ) ) nw::dev::WPadWin();
            m_WPad->SetKeyboardMouseDevice( nw::dev::KeyboardMouseDeviceWin::GetInstance() );
            // ポインタの座標を中心原点にします。
            m_WPad->SetPointerCenterOrigin( true );

            // VPad にデバイスを設定します。
            m_VPad = new( m_Arg.allocator->Alloc( sizeof( nw::dev::VPadWin ) ) ) nw::dev::VPadWin( JOYPAD_NUMBER );
            m_VPad->SetKeyboardMouseDevice( nw::dev::KeyboardMouseDeviceWin::GetInstance() );
            m_VPad->SetJoyPadDevice( nw::dev::JoyPadDeviceWin::GetInstance() );
            // Mouse にデバイスを設定します。
            m_Mouse = new( m_Arg.allocator->Alloc( sizeof( nw::dev::MouseWin ) ) ) nw::dev::MouseWin();
            m_Mouse->SetKeyboardMouseDevice( nw::dev::KeyboardMouseDeviceWin::GetInstance() );
        }
#endif

        // Cafeではシェーダバイナリを読み込んで初期化する必要があります。
        nw::ut::MemoryRange fontShaderBinary;
        nw::ut::MemoryRange lytShaderBinary;

        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.allocator = this->GetAllocator();

        PathString path;

        // フォントシェーダの読み込み
#if defined(NW_PLATFORM_CAFE)
        {
            loadArg.path = m_Arg.fontShaderPath;
            u8* binary = pFileSystem->Load(loadArg);
            fontShaderBinary = nw::ut::MakeMemoryRange(binary, loadArg.readSize);
        }
#endif

        // フォントの読み込み
        {
            nw::dev::FileDevice::LoadArg myLoadArg = loadArg;
            myLoadArg.path = m_Arg.fontPath;
            myLoadArg.alignment = nw::font::RESOURCE_ALIGNMENT;
            u8* binary = pFileSystem->Load(myLoadArg);
            m_FontBinary = nw::ut::MakeMemoryRange(binary, myLoadArg.readSize);
        }

#if defined(NW_PLATFORM_CAFE)
        // レイアウトシェーダの読み込み
        {
            loadArg.path = m_Arg.lytShaderPath;
            u8* binary = pFileSystem->Load(loadArg);
            lytShaderBinary = nw::ut::MakeMemoryRange(binary, loadArg.readSize);
        }
#endif

        NW_INSTANCE(nw::gfnd::Graphics)->LockDrawContext();
        {
            const u32 CharMax = 512;

#if defined(NW_PLATFORM_CAFE)
            m_GraphicsResource.Setup(CharMax, fontShaderBinary);
            m_ShaderSetupHelper.Setup(&m_GraphicsResource, lytShaderBinary);
#endif

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
            m_GraphicsResource.Setup(CharMax);
#endif
        }
        NW_INSTANCE(nw::gfnd::Graphics)->UnlockDrawContext();

        m_DwTextRenderer.Initialize(
            *this->GetAllocator(),
            m_FontBinary.Begin(),
            m_FontBinary.Size(),
            fontShaderBinary.Begin(),
            fontShaderBinary.Size());
        m_DwUIRenderer.SetTextRenderer(&m_DwTextRenderer);
        m_DwUIRenderer.SetPrimitiveRenderer(NW_INSTANCE(nw::dev::PrimitiveRenderer));

        pFileSystem->Unload(loadArg, fontShaderBinary.Begin());
        pFileSystem->Unload(loadArg, lytShaderBinary.Begin());

#if defined(NW_VWRLYT_ENABLE)

        m_LayoutViewer
            .SetUIRenderer(&m_DwUIRenderer)
            .SetDefaultPreviewPath(m_Arg.nwRootPath)
            .Initialize(
                &m_GraphicsResource,
                this->GetAllocator(),
                nw::lyt::Size(f32(this->GetWidth()), f32(this->GetHeight())));

        m_Inputs
            .SetMainViewportSize(this->GetWidth(), this->GetHeight())
            .SetVPadSrc(this->m_VPad)
            .SetWPadSrc(this->m_WPad)
            .SetPadSrc(this->m_Pad)
            .SetMouseSrc(this->m_Mouse);

        m_CustomControlCreator.SetButtonGroup(&m_LayoutViewer.GetButtonGroup());
        m_LayoutViewer.SetControlCreator(&m_CustomControlCreator);
#endif // NW_VWRLYT_ENABLE

        this->LoadSampleData();

        m_GfxContext.SetDepthEnable(false, false);
        m_GfxContext.SetCullingMode(nw::gfnd::Graphics::CULLING_MODE_NONE);

        // 負荷メーターの枠の色を変えます。
        this->GetMeterDrawer().SetBorderColor(nw::ut::Color4u8(nw::ut::Color4u8::X_DARK_GRAY));
    }

    //-------------------------------------------------------------------------
    //! @brief        デモの終了処理です。
    //-------------------------------------------------------------------------
    virtual void ProcFin()
    {
        // 入力デバイスの終了処理を行います。
        {
            nw::dev::FileDevice::LoadArg loadArg;
            loadArg.allocator = this->GetAllocator();
            nw::dev::FileDeviceManager* pFileSystem = NW_INSTANCE(nw::dev::FileDeviceManager);
            pFileSystem->Unload(loadArg, m_FontBinary.Begin());

#if defined( NW_PLATFORM_CAFE )
            nw::dev::PadDeviceCafe::GetInstance()->Finalize();
            nw::dev::WPadDeviceCafe::GetInstance()->Finalize();
            nw::dev::KeyboardMouseDeviceCafe::GetInstance()->Finalize();
#else
            nw::dev::JoyPadDeviceWin::GetInstance()->Finalize();
            nw::dev::KeyboardMouseDeviceWin::GetInstance()->Finalize();
#endif

            nw::ut::SafeFreeWithDestruct(m_Pad, m_Arg.allocator);
            nw::ut::SafeFreeWithDestruct(m_VPad, m_Arg.allocator);
            nw::ut::SafeFreeWithDestruct(m_WPad, m_Arg.allocator);
            nw::ut::SafeFreeWithDestruct(m_Mouse, m_Arg.allocator);
        }

        m_GraphicsResource.Finalize();

#if defined(NW_PLATFORM_CAFE)
        m_ShaderSetupHelper.Finalize();
#endif

        m_DwTextRenderer.Finalize();

#if defined(NW_VWRLYT_ENABLE)
        m_LayoutViewer.Finalize();
#endif
    }

    //-------------------------------------------------------------------------
    //! @brief        デモの計算処理です。
    //-------------------------------------------------------------------------
    virtual void ProcCalc()
    {
#if defined(NW_VWRLYT_ENABLE)

        // デバイスを更新します。
        {
#if defined(NW_PLATFORM_CAFE)
            nw::dev::PadDeviceCafe::GetInstance()->Update();
            nw::dev::WPadDeviceCafe::GetInstance()->Update();
            nw::dev::VPadDeviceCafe::GetInstance()->Update();
            nw::dev::KeyboardMouseDeviceCafe::GetInstance()->Update();
#else
            nw::dev::KeyboardMouseDeviceWin::GetInstance()->Update();
            nw::dev::JoyPadDeviceWin::GetInstance()->Update();
#endif
           // パッドを更新します。
            m_Pad->Update();
            m_WPad->Update();
            m_VPad->Update();
            m_Mouse->Update();
        }

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        if (m_LayoutViewer.IsScreenShotDone())
        {
            m_Exit = true;
            return;
        }
#endif
        m_LayoutViewer.UpdateSystem();
        m_Inputs.Update();
        m_LayoutViewer.UpdateInputs(m_Inputs);
        m_LayoutViewer.UpdateMenu();

        this->GetMeterCalc().BeginMeasure();
        m_LayoutViewer.UpdateLayout();
        this->GetMeterCalc().EndMeasure();

        // FPSの切り替え
        nw::vwrlyt::Viewer::Fps fps = m_LayoutViewer.GetFps();
        if (m_Fps != fps)
        {
            m_Fps = fps;

            bool isFps60 = (fps == nw::vwrlyt::Viewer::FPS_60);

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
            this->SetFps(isFps60? 60.f : 30.f);
#endif

#if defined(NW_PLATFORM_CAFE)
            this->SetVBlankWaitInterval(isFps60? 1 : 2);
#endif
        }

#endif // NW_VWRLYT_ENABLE
    }

    //-------------------------------------------------------------------------
    //! @brief        TV画面の描画処理です。
    //-------------------------------------------------------------------------
    virtual void ProcDraw()
    {
        {
            nw::math::Matrix44 projMatrix;
            projMatrix.SetOrtho(0.f, f32(this->GetWidth()), f32(this->GetHeight()), 0.f, 0.0f, 1.f);
            m_DwUIRenderer.SetProjectionMatrix(projMatrix);
        }


        this->BeginDrawTV();
        this->SetViewport(0.f, 0.f, f32(this->GetWidth()), f32(this->GetHeight()), 0.f, 1.f);
        this->SetScissor(0.f, 0.f, f32(this->GetWidth()), f32(this->GetHeight()));
        this->ClearFrameBuffers();

        this->GetMeterGPU().BeginMeasure();
        this->GetMeterDraw().BeginMeasure();
#if defined(NW_VWRLYT_ENABLE)
        m_LayoutViewer.DrawLayout();
#endif
        this->GetMeterDraw().EndMeasure();
        this->GetMeterGPU().EndMeasure();

#if defined(NW_VWRLYT_ENABLE)
        m_LayoutViewer.DrawSystem();

        if (m_Arg.usePointerCursor &&
            m_Inputs.GetMouse() &&
            m_Inputs.GetMouse()->IsPointerOn())
        {
            nw::math::Vector2 pos = m_Inputs.GetMouse()->GetPointer();
            this->DrawPointerCursor(
                pos.x * 2.f / this->GetWidth() -1.f,
                pos.y * 2.f / this->GetHeight() - 1.f,
                f32(this->GetWidth()),
                f32(this->GetHeight()));
        }
#endif

        this->DrawLoadMeters();

        m_DrawOnce = true;

        this->EndDrawTV();
    }

    //-------------------------------------------------------------------------
    //! @brief        DRCの描画処理です。
    //-------------------------------------------------------------------------
    virtual void ProcDrawDRC()
    {
        this->BeginDrawDRC();
        this->SetViewport(0.f, 0.f, f32(this->GetDRCWidth()), f32(this->GetDRCHeight()), 0.f, 1.f);
        this->SetScissor(0.f, 0.f, f32(this->GetDRCWidth()), f32(this->GetDRCHeight()));
        this->ClearFrameBuffers();

#if defined(NW_VWRLYT_ENABLE)
        m_LayoutViewer.DrawLayoutDRC();
#endif

        this->EndDrawDRC();
    }

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    static LRESULT CALLBACK WindowMessageCallback(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
    {
        NW_UNUSED_VARIABLE(hWnd);
        NW_UNUSED_VARIABLE(message);
        NW_UNUSED_VARIABLE(wParam);
        NW_UNUSED_VARIABLE(lParam);
#if defined(NW_VWRLYT_ENABLE)
        if (GetInstance()->m_LayoutViewer.PreviewByMessageProcCallback(hWnd, message, wParam, lParam))
        {
            return 1;
        }
#endif

        return 0;
    }
#endif

    //-------------------------------------------------------------------------
    //! @brief        インスタンスを取得します。
    //-------------------------------------------------------------------------
    static TestDemoSystem* GetInstance()
    {
        return s_pInstance;
    }

private:
    //! グラフィックスコンテキストです。
    nw::gfnd::GraphicsContext   m_GfxContext;

    //! レイアウトグラフィックスリソースです。
    nw::lyt::GraphicsResource m_GraphicsResource;

    //! フォントバイナリです。
    nw::ut::MemoryRange m_FontBinary;

#if defined(NW_PLATFORM_CAFE)
    //! レイアウトシェーダ・セットアップヘルパーです。
    nw::lyt::ShaderSetupHelper m_ShaderSetupHelper;
#endif

#if defined(NW_VWRLYT_ENABLE)
    //! レイアウトビューアです。
    nw::vwrlyt::Viewer m_LayoutViewer;
    //! 入力情報です。
    nw::vwrlyt::Inputs m_Inputs;
    //! カスタムコントロールクリエータです。
    CustomControlCreator m_CustomControlCreator;
    //!
    nw::vwrlyt::Viewer::Fps m_Fps;
#endif

    //!
    bool m_DrawOnce;

    nw::internal::dw::NwTextRenderer m_DwTextRenderer;
    nw::internal::dw::NwUIRenderer m_DwUIRenderer;
    nw::internal::dw::ControlWindow m_DwControlWindow;
    CreateArg m_Arg;

#if defined( NW_PLATFORM_CAFE )
    nw::dev::PadCafe*          m_Pad;    // デバッグ用パッドを管理します。
    nw::dev::WPadCafe*         m_WPad;   // WPAD を管理します。
    nw::dev::VPadCafe*         m_VPad;   // VPAD を管理します。
    nw::dev::MouseCafe*        m_Mouse;  // Mouse を管理します。
#else
    nw::dev::PadWin*           m_Pad;    // デバッグ用パッドを管理します。
    nw::dev::WPadWin*          m_WPad;   // WPAD をマウスを用いて PC で再現します。
    nw::dev::VPadWin*          m_VPad;   // VPAD を PC で再現します。
    nw::dev::MouseWin*         m_Mouse;  // Mouse を PC で再現します。
#endif

    static TestDemoSystem* s_pInstance;
};

TestDemoSystem* TestDemoSystem::s_pInstance = NULL;

//-----------------------------------------------------------------------------
void
TestDemoSystem::LoadSampleData()
{
}

//-----------------------------------------------------------------------------

} // anonymous namespace

//-----------------------------------------------------------------------------
//! @brief        メイン関数です。
//-----------------------------------------------------------------------------
int
NwDemoMain(int argc, char **argv)
{

#if defined(NW_RELEASE)
    NW_LOG("NintendoWare --- Can't execute this demo as release-build.\n");
    for(;;) { } // Release 版では ホスト通信機能 が利用できないため実行できません。
#endif

    NW_UNUSED_VARIABLE(argc);
    NW_UNUSED_VARIABLE(argv);

    nw::ut::MemoryAllocator demoAllocator;
    InitializeMemory( &demoAllocator );

    nw::ut::MemoryAllocator* allocator = &demoAllocator;

    nw::lyt::Initialize(allocator);

    // ファイルシステムを初期化します。
    TestDemoSystem::InitFileDeviceManager(allocator);

    // NINTENDO_SDK_ROOT 環境変数を取得します。
    TestDemoSystem::PathString nwRoot;
    if (!TestDemoSystem::GetNwRoot(nwRoot))
    {
        NW_LOG("Can not get valid NINTENDO_SDK_ROOT environment variable.");
    	NW_HALT;
    }

#if defined(USE_MCS_INPUT)
    // マウス入力キャプチャのために mcs を初期化します。
    {
        nw::mcs::Mcs_Initialize();

        u32 errorCode = nw::mcs::Mcs_Open();
        NW_ASSERT( errorCode == nw::mcs::MCS_ERROR_SUCCESS );

        // mcsHID を初期化します。
        nw::mcs::McsHID_Initialize();
        if ( nw::mcs::McsHID_GetRegisteredBuffer() == NULL )
        {
            // 登録済みのバッファがない場合には、バッファを生成して登録します。
            s_InputDeviceBuf = reinterpret_cast<u8*>( allocator->Alloc( MCS_INPUT_DEVICE_BUF_SIZE ) );
            nw::mcs::McsHID_RegisterBuffer( s_InputDeviceBuf, MCS_INPUT_DEVICE_BUF_SIZE );
        }
    }
#endif // #if defined(USE_MCS_INPUT)

    {
        // デモシステムの初期化を行います。
        TestDemoSystem::CreateArg demoSystemArg;
        demoSystemArg.allocator = allocator;
        demoSystemArg.width     = SCREEN_WIDTH;
        demoSystemArg.height    = SCREEN_HEIGHT;
        demoSystemArg.drawMeter = true;
#if defined( NW_PLATFORM_CAFE )
        demoSystemArg.usePointerCursor = true;
#endif
#if defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        demoSystemArg.primitiveRendererInitialize = true;
#endif

        demoSystemArg.nwRootPath = nwRoot.c_str();

        TestDemoSystem::PathString fontPath;
        fontPath.format("%s/Samples/Resources/font/%s",
            nwRoot.c_str(), DEMO_FONT_FILENAME);
        demoSystemArg.fontPath = fontPath.c_str();

#if defined( NW_PLATFORM_CAFE )
        TestDemoSystem::PathString fontShaderPath;
        fontShaderPath.format(
            "%s/Resources/FontShaders/font_BuildinShader.gsh",
            nwRoot.c_str());
        demoSystemArg.fontShaderPath = fontShaderPath.c_str();

        TestDemoSystem::PathString lytShaderPath;
        lytShaderPath.format(
            "%s/Resources/LytShaders/lyt_BuildinShader.gsh",
            nwRoot.c_str());
        demoSystemArg.lytShaderPath = lytShaderPath.c_str();

        TestDemoSystem::PathString primRendererPath;
        primRendererPath.format(
            "%s/Resources/DevShaders/dev_PrimitiveRenderer.gsh",
            nwRoot.c_str());
        demoSystemArg.primitiveRendererShaderPath = primRendererPath.c_str();
#endif
        TestDemoSystem* pDemoSystem = new (allocator->Alloc(sizeof(TestDemoSystem))) TestDemoSystem( demoSystemArg );
        NW_ASSERT_NOT_NULL(pDemoSystem);

        // CAFE版ビルド時のワーニング抑制
        (void)TestDemoSystem::GetInstance();

        pDemoSystem->Initialize();
        pDemoSystem->InitializeGraphicsSystem();

#if defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        // プレビュー要求のWindowsメッセージを受信するために、
        // ウィンドウタイトルに"Layout Viewer"を含むように設定します。
        pDemoSystem->SetWindowTitle("Custom Layout Viewer");
        pDemoSystem->SetMsgProcCallback(TestDemoSystem::WindowMessageCallback);
#endif

        // 自動テスト用のブロックです。
        NW_DEMO_TEST_BLOCK( allocator )
        {
            pDemoSystem->ProcInit();

            // 描画ループです。
            do
            {
                pDemoSystem->BeginFrame();

                pDemoSystem->ProcCalc();

                nw::gfnd::Graphics::GetInstance()->LockDrawContext();
                {
#if defined(NW_PLATFORM_CAFE)
                    // GPUのreadキャッシュを全域クリアする
                    GX2Invalidate(static_cast<GX2InvalidateType>(GX2_INVALIDATE_ATTRIB_BUFFER | GX2_INVALIDATE_TEXTURE | GX2_INVALIDATE_CONSTANT_BUFFER | GX2_INVALIDATE_SHADER),
                        0x00000000, 0xffffffff );
#endif
                    pDemoSystem->ProcDraw();
                    pDemoSystem->ProcDrawDRC();
                }
                nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

                pDemoSystem->SwapBuffer();

                pDemoSystem->EndFrame();

                pDemoSystem->WaitForVBlank();
            } while ( ! pDemoSystem->IsExiting() );

            pDemoSystem->ProcFin();
        }

        // デモシステムの終了処理を行います。
        pDemoSystem->FinalizeGraphicsSystem();
        pDemoSystem->Finalize();

#if defined(USE_MCS_INPUT)
    // mcs の終了処理を行います。
    {
        if ( s_InputDeviceBuf != NULL && nw::mcs::McsHID_GetRegisteredBuffer() == s_InputDeviceBuf )
        {
            nw::mcs::McsHID_UnregisterBuffer();
            allocator->Free( s_InputDeviceBuf );
        }

        nw::mcs::McsHID_Finalize();
        nw::mcs::Mcs_Finalize();
    }
#endif // #if defined(USE_MCS_INPUT)

        TestDemoSystem::FinalizeFileDeviceManager();
        nw::ut::SafeFreeWithDestruct(pDemoSystem, allocator);
    }

    return 0;
}
