﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/fnd/g3d_GfxObject.h>
#include <nw/g3d/fnd/g3d_GfxState.h>
#include <nw/g3d/fnd/g3d_GfxShader.h>
#include <nw/g3d/fnd/g3d_GfxManage.h>
#include <nw/g3d/fnd/g3d_EglUtility.h>
#include <g3ddemo_DemoUtility.h>

namespace nw { namespace g3d { namespace res {

class ResFile;
class ResShaderProgram;

}}} // namespace nw::g3d

namespace nw { namespace g3d { namespace demo {

//--------------------------------------------------------------------------------------------------
// Graphics

enum BufferType
{
    COLOR_BUFFER        = 0x1 << 0,
    AUX_BUFFER          = 0x1 << 1,
    DEPTH_BUFFER        = 0x1 << 2,
    HIZ_BUFFER          = 0x1 << 3,
    COLOR_AUX_BUFFER    = COLOR_BUFFER | AUX_BUFFER,
    DEPTH_HIZ_BUFFER    = DEPTH_BUFFER | HIZ_BUFFER,
    ALL_BUFFER          = COLOR_AUX_BUFFER | DEPTH_HIZ_BUFFER
};

struct ColorBufferTexture
{
    nw::g3d::GfxTexture texture;
    nw::g3d::GfxColorBuffer renderBuffer;
    bool isFTV;

    void Setup();
    void Cleanup();

    void Alloc(FuncAlloc funcAlloc, bit32 bufferType = ALL_BUFFER);
    void Free(FuncFree funcFree, bit32 bufferType = ALL_BUFFER);
};

struct DepthBufferTexture
{
    nw::g3d::GfxTexture texture;
    nw::g3d::GfxDepthBuffer renderBuffer;

    void Setup();
    void Cleanup();

    void Alloc(FuncAlloc funcAlloc, bit32 bufferType = ALL_BUFFER);
    void Free(FuncFree funcFree, bit32 bufferType = ALL_BUFFER);
};

class FrameBuffer
{
public:
    enum
    {
        MAX_WIDTH = 8192,
        MAX_HEIGHT = 8192,
        MAX_RENDER_TARGET = 8
    };

    FrameBuffer() {}

    struct InitArg;
    static size_t CalcSize(const InitArg& arg);
    bool Init(const InitArg& arg, void* pBuffer, size_t bufferSize);

    void Setup();
    void Cleanup();

    void Alloc(FuncAlloc funcAlloc, bit32 bufferType = ALL_BUFFER);
    void Free(FuncFree funcFree, bit32 bufferType = ALL_BUFFER);

    void Load();

    u32 GetWidth() const { return m_Width; }
    u32 GetHeight() const { return m_Height; }

    void SetMipLevel(u32 mipLevel);

    ColorBufferTexture* GetColorBufferTexture(GX2RenderTarget target)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(target, m_NumRenderTarget);
        return &m_pColorBuffer[target];
    }

    const ColorBufferTexture* GetColorBufferTexture(GX2RenderTarget target) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(target, m_NumRenderTarget);
        return &m_pColorBuffer[target];
    }

    DepthBufferTexture* GetDepthBufferTexture() { return m_pDepthBuffer; }
    const DepthBufferTexture* GetDepthBufferTexture() const { return m_pDepthBuffer; }

    nw::g3d::GfxViewport& GetViewport() { return m_Viewport; }
    const nw::g3d::GfxViewport& GetViewport() const { return m_Viewport; }

    nw::g3d::GfxScissor& GetScissor() { return m_Scissor; }
    const nw::g3d::GfxScissor& GetScissor() const { return m_Scissor; }

    void* GetBufferPtr()
    {
        return m_pColorBuffer ?
            static_cast<void*>(m_pColorBuffer) :
            static_cast<void*>(m_pDepthBuffer);
    }

private:
    u32 m_Handle;
    u32 m_Width;
    u32 m_Height;
    int m_NumRenderTarget;
    ColorBufferTexture* m_pColorBuffer;
    DepthBufferTexture* m_pDepthBuffer;
    nw::g3d::GfxViewport m_Viewport;
    nw::g3d::GfxScissor m_Scissor;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(FrameBuffer);
};

struct FrameBuffer::InitArg
{
    InitArg(u32 width, u32 height)
        : width(width)
        , height(height)
        , colorBufferCount(1)
        , colorBufferFormat(GX2_SURFACE_FORMAT_TC_R11_G11_B10_FLOAT)
        , colorBufferFTV(false)
        , useDepthBuffer(true)
    {
    }

    u32 width;
    u32 height;
    int colorBufferCount;
    GX2SurfaceFormat colorBufferFormat;

    bool colorBufferFTV;
    bool useDepthBuffer;
};

class SimpleShader : public nw::g3d::GfxShader
{
public:
    struct Path
    {
        const char* gsh;
        const char* vertexShader;
        const char* fragmentShader;
    };

    void Setup(const Path& path);
    void Cleanup();
};

}}} // namespace nw::g3d::demo

