﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d.h>
#include <nw/g3d/g3d_edit.h>
#include <g3ddemo_DemoUtility.h>
#include <g3ddemo_GfxUtility.h>
#include <g3ddemo_UserModel.h>
#include <g3ddemo_ModelUtility.h>

#ifdef _WIN32
#include <nw/g3d/fnd/g3d_WinUtility.h>
#include <gl/glew.h>
#else
#include <cafe/vpad.h>
#endif

#include <float.h>
#include <stdio.h>

namespace g3ddemo = nw::g3d::demo;

namespace {

enum
{
    // モデルに使用するテクスチャのアライメントの最大値を設定します。
    // リソースファイルごとのアライメントを事前に取得しておくことで、
    // 必要なメモリのアライメントを最小限に抑えてメモリを節約することも可能です。
    TEXTURE_ALIGNMENT   = 8192
};

#define SHADER_PATH "shader/"
#define MODEL_PATH  ""

const char* BFSHA_PATH[] = {
    SHADER_PATH "town.bfsha",  // 3D Editor 用
    SHADER_PATH "shadow.bfsha",
    SHADER_PATH "gbuffer.bfsha",
    SHADER_PATH "demo.bfsha",  // 3D Editor 用
    SHADER_PATH "tex_average.bfsha",
    NULL
};
const char* BFRES_PATH[] = {
    MODEL_PATH "town_env.bfres",
    MODEL_PATH "bg_WhiteTown.bfres",
    MODEL_PATH "Duck.bfres",
    MODEL_PATH "FemaleA.bfres",
    MODEL_PATH "Fish.bfres",
    MODEL_PATH "MaleA.bfres",
    MODEL_PATH "WhiteTown_Water_Caustics.bfres",
    MODEL_PATH "WhiteTown_Water_Flat.bfres",
    MODEL_PATH "Light.bfres",
    NULL
};

#ifdef _WIN32
const char* MODEL_SHADER = "town_pc.bfsha";
const char* s_pRootPath = NULL;
#else
const char* MODEL_SHADER = "town.bfsha";
#endif

nw::g3d::Vec3 s_CameraPosition;
nw::g3d::Vec3 s_CameraUp;
nw::g3d::Vec3 s_CameraTarget;
nw::g3d::Vec3 s_LightPosition;

enum
{
    VIEW_TV = 0,
    VIEW_WATER,
    VIEW_LIGHT,
    VIEW_LIGHT1,
    VIEW_LIGHT2,
    VIEW_LIGHT3,

    NUM_VIEW
};

g3ddemo::ResourceHolder s_Holder;
g3ddemo::Vector<g3ddemo::UserModel*> s_Lights;

g3ddemo::ScreenInfo s_Screen;
g3ddemo::ProcessMeter s_Meter;

g3ddemo::UserView s_View;
nw::g3d::Mtx34 s_TexMtxEnv;
nw::g3d::Mtx34 s_TexMtxProj;

nw::g3d::GfxTexture s_WaterTexture;
nw::g3d::GfxTexture s_ShadowTexture;
nw::g3d::GfxTexture s_ColorTexture;
nw::g3d::GfxTexture s_NormalTexture;
nw::g3d::GfxTexture s_LightTexture;
nw::g3d::GfxTexture s_DepthTexture;

int s_TileDimX = 4;
int s_TileDimY = 4;

int s_SelectedModel;

const float s_PointLightMaxRadius = 400.0f;

bool s_IsNormalMode;

const int s_nCascade = 4;
const float s_CascadeDepth[s_nCascade + 1] = { 10.0f, 800.0f, 1600.0f, 4500.0f, 40000.0f };

g3ddemo::DisplayList s_FrameDL;

g3ddemo::ModelAssign* SetupModelAssign(nw::g3d::ResModel* pResModel,
    nw::g3d::ResShaderArchive** pShaderArchiveArray = NULL, int ShaderArchiveCount = 0,
    nw::g3d::ResShaderArchive* pModelShader = NULL);
g3ddemo::UserModel* SetupUserModel(nw::g3d::ModelObj* pModelObj);
void SetupUserShadingModel(nw::g3d::ResShaderArchive* pShader);
void SetupWaterVariation(nw::g3d::ModelObj* pModelObj);
void AddAnim(const char* pModelName, const char* pAnimName, float step);
void ChangeAnimStep(g3ddemo::UserModel* pUserModel, float step);
void UpdateCamera(const g3ddemo::Pad& pad);
void CalcLightProj(nw::g3d::Mtx44* pOutLightProj, const nw::g3d::Mtx34* pLightView,
                   const nw::g3d::Mtx34* pCameraView, const nw::g3d::Mtx44* pCameraProj);

#if NW_G3D_CONFIG_USE_HOSTIO
bool s_AssignChanged = false;
void AttachShader(const char* pArchiveName);
#endif

enum EnableFlag
{
    ENABLE_LIGHT_ANIMATION = 0x01 << 0,
    ENABLE_WATER_RENDERING = 0x01 << 1,
    ENABLE_GEOMETRY_RENDERING = 0x01 << 2,
    ENABLE_LIGHT_RENDERING = 0x01 << 3,
    ENABLE_SHADOW_RENDERING = 0x01 << 4,
    ENABLE_MODEL_ANIMATION = 0x01 << 5,
    ENABLE_WIREFRAME = 0x01 << 6,
    ENABLE_ICE = 0x01 << 7,
    ENABLE_DRC_RENDERING = 0x01 << 8,
    ENABLE_TEXTURE_COMPUTE = 0x01 << 9,

    INITIAL_FLAG =
        ENABLE_WATER_RENDERING |
        ENABLE_GEOMETRY_RENDERING |
        ENABLE_LIGHT_RENDERING |
        ENABLE_SHADOW_RENDERING |
        ENABLE_MODEL_ANIMATION |
        ENABLE_DRC_RENDERING |
        ENABLE_TEXTURE_COMPUTE
};
bit32 s_EnableFlag;

#if NW_G3D_CONFIG_USE_HOSTIO

g3ddemo::EditAllocator s_Allocator;

class Editor : public nw::g3d::edit::EditCallback
{
public:
    virtual nw::g3d::ResFile* LoadFile(const nw::g3d::edit::LoadFileArg& arg)
    {
        // 3DEditor が開いたモデルがリソースとして送られてきます。
        // リソースファイルはアプリケーションが管理する必要あるため、コピーします。
        void* pFile = g3ddemo::AllocMem2(arg.fileSize, arg.align);
        memcpy(pFile, arg.resFile, arg.fileSize);
        // CPU がコピーしたので、GPU が参照できるようキャッシュをフラッシュします。
        nw::g3d::CPUCache::Flush(pFile, arg.fileSize);

        NW_G3D_ASSERT(nw::g3d::ResFile::IsValid(pFile));
        nw::g3d::ResFile* pResFile = nw::g3d::ResFile::ResCast(pFile);
        pResFile->Setup();

        s_Holder.files.PushBack(pResFile);

        // モデルインスタンス構築処理は、アプリケーションで使用している構築処理を
        // そのまま使用することができます。
        // このデモではコールバック内でモデルを構築、登録していますが、
        // コールバック外で遅延処理しても構いません。
        for (int idxModel = 0, numModel = pResFile->GetModelCount();
            idxModel < numModel; ++idxModel)
        {
            nw::g3d::ResModel* pResModel = pResFile->GetModel(idxModel);

            // シェーダ割り当ての構築。
            g3ddemo::ModelAssign* pModelAssign = SetupModelAssign(pResModel);
            s_Holder.assigns.PushBack(pModelAssign);

            // モデルインスタンスの構築。
            g3ddemo::CreateModelObjArg createModelObjArg(pResModel);
            createModelObjArg.initArg.ViewCount(NUM_VIEW);
            createModelObjArg.initArg.EnableBounding();
            nw::g3d::ModelObj* pModelObj = g3ddemo::CreateModelObj(createModelObjArg);
            g3ddemo::UserModel* pUserModel = SetupUserModel(pModelObj);
            pUserModel->EnableEditCalcSkeletalAnimations();
            pUserModel->EnableFrustumCulling();
            s_Holder.models.PushBack(pUserModel);

            SetupWaterVariation(pModelObj);

            // エディットへの登録。
            // エディットライブラリのために専用のクラスなどを作成する必要がありません。
            pUserModel->AttachToEdit();
        }

        return pResFile;
    }

    virtual void UnloadFile(const nw::g3d::edit::UnloadFileArg& arg)
    {
        // 3DEditor でファイルが閉じられたので、関連するデータを破棄します。
        // GPU が参照中のデータを破棄してしまわないよう、アプリケーションの
        // フレームワークに応じて描画完了待ちを行ったり、描画をスキップするなど、
        // 必要な対策を講じてください。
        // このデモでは GPU による描画が完了してから呼ばれているため、
        // 特に対策は行っていません。
        nw::g3d::ResFile* pResFile = arg.resFile;

        for (int idxResMdl = 0, numResMdl = pResFile->GetModelCount();
            idxResMdl < numResMdl; ++idxResMdl)
        {
            nw::g3d::ResModel* pResModel = pResFile->GetModel(idxResMdl);

            // モデルインスタンスの破棄。
            for (int idxModel = 0, numModel = s_Holder.models.Size();
                idxModel < numModel; ++idxModel)
            {
                g3ddemo::UserModel* pUserModel = s_Holder.models[idxModel];
                nw::g3d::ModelObj* pModelObj = pUserModel->GetModelObj();
                if (pModelObj->GetResource() == pResModel)
                {
                    pUserModel->DestroyAnimObj();
                    g3ddemo::DestroyUserModel(pModelObj);
                    g3ddemo::DestroyModelObj(pModelObj);
                    s_Holder.models.Erase(idxModel);
                    break;
                }
            }

            // シェーダ割り当ての破棄。
            for (int idxAssign = 0, numAssign = s_Holder.assigns.Size();
                idxAssign < numAssign; ++idxAssign)
            {
                g3ddemo::ModelAssign* pModelAssign = s_Holder.assigns[idxAssign];
                if (pModelAssign->GetResModel() == pResModel)
                {
                    pModelAssign->Cleanup();
                    g3ddemo::DestroyShaderAssign(pResModel);
                    s_Holder.assigns.Erase(idxAssign);
                    break;
                }
            }

            // リソースファイルの破棄。
            for (int idxFile = 0, numFile = s_Holder.files.Size();
                idxFile < numFile; ++idxFile)
            {
                if (s_Holder.files[idxFile] == pResFile)
                {
                    pResFile->Cleanup();
                    g3ddemo::FreeMem2(pResFile);
                    s_Holder.files.Erase(idxFile);
                    break;
                }
            }
        }
    }

    virtual void UpdateMaterials(const nw::g3d::edit::UpdateMaterialsArg& arg)
    {
        // UpdateShaderAssign で行われた変更に伴って ModelObj に変更が必要な場合は
        // このコールバックで処理を行います。
        // 編集のために使用するヒープを区別したい場合は arg.state で状態を判定します。
        // UpdateShaderAssign() で初期化したリソースに従って MaterialObj や ShapeObj が
        // エディットライブラリによって作り直されているので、必要な初期化を行います。

        nw::g3d::ModelObj* pModelObj = arg.modelObj;

        // シェーダが変更された場合のためにシェーダインスタンスを再構築します。
        for (int idxShape = 0, numShape = pModelObj->GetShapeCount();
            idxShape < numShape; ++idxShape)
        {
            nw::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(idxShape);
            g3ddemo::UserShape* pUserShape = pShapeObj->GetUserPtr<g3ddemo::UserShape>();
            pUserShape->Cleanup();
            pUserShape->Setup();
        }

        // マテリアルを更新します。
        g3ddemo::SetupMaterialsArg argSetup(pModelObj);
        g3ddemo::SetupMaterials(argSetup);

        SetupWaterVariation(pModelObj);
    }

    virtual void UpdateShaderAssign(const nw::g3d::edit::UpdateShaderAssignArg& arg)
    {
        // モデルリソースの構造変更を伴う編集が 3DEditor で行われたため、
        // 古いモデルリソースの破棄と新しいモデルリソースの構築を行います。
        // モデルインスタンスが新しいリソースを参照するように、
        // エディットライブラリが参照関係を変更します。
        g3ddemo::ModelAssign* pModelAssign = arg.oldResModel->GetUserPtr<g3ddemo::ModelAssign>();
        switch (arg.state)
        {
        case nw::g3d::edit::UpdateShaderAssignArg::STATE_MODEL_UPDATE:
        case nw::g3d::edit::UpdateShaderAssignArg::STATE_MODEL_END:
            {
                // モデルリソースにユーザ領域を関連付けて使用していた場合は破棄します。
                NW_G3D_ASSERT_NOT_NULL(pModelAssign);
                pModelAssign->Cleanup();
                g3ddemo::DestroyShaderAssign(arg.oldResModel);
            }
            break;
        default:
            break;
        }
        switch(arg.state)
        {
        case nw::g3d::edit::UpdateShaderAssignArg::STATE_MODEL_UPDATE:
        case nw::g3d::edit::UpdateShaderAssignArg::STATE_MODEL_BEGIN:
            {
                // モデルリソースにユーザ領域を関連付けて管理します。
                // アプリケーションで使用している構築処理をそのまま使用することができます。
                for (int idxShader = 0; idxShader < arg.numShaderArchive; ++idxShader)
                {
                    SetupUserShadingModel(arg.shaderArchivePtrs[idxShader]);
                }
                SetupModelAssign(arg.newResModel, arg.shaderArchivePtrs, arg.numShaderArchive);
            }
            break;
        default:
            break;
        }
        s_AssignChanged = true;
    }

    virtual void UpdateBoneBind(const nw::g3d::edit::UpdateBoneBindArg& arg)
    {
        // 3DEditor でプレビュー配置のバインド設定が行われた際にモデルに通知を行います。
        // 配置自体はここで行わずにアプリケーションのバインド機能によってバインドします。
        g3ddemo::UserModel* pUserModel = arg.modelObj->GetUserPtr<g3ddemo::UserModel>();
        pUserModel->BindTo(arg.parentModelObj, arg.parentBoneIndex);
    }

    virtual void UpdateModelLayout(const nw::g3d::edit::UpdateModelLayoutArg& arg)
    {
        g3ddemo::UserModel* pUserModel = arg.modelObj->GetUserPtr<g3ddemo::UserModel>();
        pUserModel->GetScale().Set(*nw::g3d::Vec3::Cast(arg.scale.a));
        pUserModel->GetLayoutMtx().SetR(*nw::g3d::Vec3::Cast(arg.rotate.a));
        pUserModel->GetLayoutMtx().SetT(*nw::g3d::Vec3::Cast(arg.translate.a));
    }

    virtual bool SendModelLayout(nw::g3d::edit::SendModelLayoutData* outputData,
        const nw::g3d::edit::SendModelLayoutArg& arg)
    {
        const g3ddemo::UserModel* pUserModel = arg.modelObj->GetUserPtr<g3ddemo::UserModel>();
        *nw::g3d::Vec3::Cast(outputData->scale.a) = pUserModel->GetScale();
        const nw::g3d::Mtx34& layoutMtx = pUserModel->GetLayoutMtx();
        float sy = -layoutMtx.m20;
        // Y 軸周りの回転は ±90°の範囲として計算します。
        outputData->rotate.y = nw::g3d::Math::Asin(sy);
        if (nw::g3d::Math::Abs(sy) > 0.9999f)
        {
            // Y 軸周りの回転が ±90°の場合は角度が一意に決まらないため、
            // X 軸周りの回転は 0°として扱います。
            outputData->rotate.x = 0.0f;
            // Z 軸周りの回転は ±180°の範囲として計算します。
            outputData->rotate.z = nw::g3d::Math::Atan2(layoutMtx.m12, layoutMtx.m02);
        }
        else
        {
            // X 軸および Z 軸周りの回転は ±180°の範囲として計算します。
            outputData->rotate.x = nw::g3d::Math::Atan2(layoutMtx.m21, layoutMtx.m22);
            outputData->rotate.z = nw::g3d::Math::Atan2(layoutMtx.m10, layoutMtx.m00);
        }
        outputData->translate.x = layoutMtx.m03;
        outputData->translate.y = layoutMtx.m13;
        outputData->translate.z = layoutMtx.m23;
        return true;
    }

    virtual void BindSceneAnim(const nw::g3d::edit::BindSceneAnimArg& arg)
    {
        (void)arg;
    }

    virtual void UnbindSceneAnim(const nw::g3d::edit::UnbindSceneAnimArg& arg)
    {
        (void)arg;
    }

    virtual void ApplySceneAnim(const nw::g3d::edit::ApplySceneAnimArg& arg)
    {
        for (int viewIdx = 0; viewIdx < NUM_VIEW && viewIdx < arg.numCameraAnimObj; ++viewIdx)
        {
            const nw::g3d::CameraAnimObj* pCameraAnimObj = arg.cameraAnimObjPtrs[viewIdx];
            const nw::g3d::CameraAnimResult* result = pCameraAnimObj->GetResult();
            nw::g3d::Mtx34 viewMtx;
            viewMtx.LookAt(
                nw::g3d::Vec3::Make(result->view.pos[0], result->view.pos[1], result->view.pos[2]),
                nw::g3d::Vec3::Make(0.0f, 1.0f, 0.0f),
                nw::g3d::Vec3::Make(result->view.aim[0], result->view.aim[1], result->view.aim[2]));
            nw::g3d::Mtx44 projMtx;
            projMtx.Perspective(result->proj.fovy, result->proj.aspect, result->proj.nearZ, result->proj.farZ);
            memcpy(&s_View.GetViewMtx(viewIdx), &viewMtx, sizeof(s_View.GetViewMtx(viewIdx)));
            memcpy(&s_View.GetProjMtx(viewIdx), &projMtx, sizeof(s_View.GetProjMtx(viewIdx)));
        }
    }

    virtual void ShapeUpdated(const nw::g3d::edit::ShapeUpdatedArg& arg)
    {
        g3ddemo::UserModel* pUserModel = arg.pModelObj->GetUserPtr<g3ddemo::UserModel>();

        // シェイプが更新されているのでワールド行列を計算し直します。
        pUserModel->CalcWorld();

        s_AssignChanged = true;
    }

} s_Editor;

#endif

struct CalcLod : public nw::g3d::ICalcLodLevelFunctor
{
public:
    CalcLod(const nw::g3d::Mtx34* pViewMtx, float cotFovY, const float* pThreasholds, int numThreasholds)
       : m_pViewMtx(pViewMtx), m_CotFovY(cotFovY), m_pThresholds(pThreasholds), m_NumThreasholds(numThreasholds)
    {
    }

    virtual int operator()(const nw::g3d::AABB& bounding, const nw::g3d::ShapeObj& )
    {
        nw::g3d::Vec3 center;
        center.Add(bounding.min, bounding.max);
        center.Mul(center, 0.5f);
        float viewZ = m_pViewMtx->m20 * center.x + m_pViewMtx->m21 * center.y
            + m_pViewMtx->m22 * center.z + m_pViewMtx->m23;
        float ratio = (viewZ != 0.0f) ? nw::g3d::math::Math::Abs(
            nw::g3d::Vec3::Distance(center, bounding.min) * m_CotFovY / viewZ) : 1.0f;
        int lodLevel = 0;
        for (int idxLevel = 1; idxLevel < m_NumThreasholds; ++idxLevel)
        {
            if (ratio < m_pThresholds[idxLevel])
            {
                lodLevel = idxLevel;
            }
        }
        return lodLevel;
    }

private:
    const nw::g3d::Mtx34* m_pViewMtx;
    float m_CotFovY;
    const float* m_pThresholds;
    int m_NumThreasholds;
};

class TownContext : public g3ddemo::IUserContext
{
public:
    struct Context
    {
        nw::g3d::Mtx44 lightViewProj[s_nCascade];
        nw::g3d::Vec4 cascadeDepth;

        nw::g3d::Vec2 tanFov;
        nw::g3d::Vec2 nearFar;

        float diffIntensity;
    };

    TownContext(){}

    void Setup()
    {
        size_t size = sizeof(Context);
        size_t alignedSize = nw::g3d::Align(size, GX2_UNIFORM_BLOCK_ALIGNMENT);
        void* ptr = g3ddemo::AllocMem2(alignedSize, GX2_UNIFORM_BLOCK_ALIGNMENT);
        NW_G3D_ASSERT_NOT_NULL(ptr);

        m_ContextBlock.SetData(ptr, size);
        m_ContextBlock.Setup();
    }

    void Cleanup()
    {
        m_ContextBlock.Cleanup();
        g3ddemo::FreeMem2(m_ContextBlock.GetData());
    }

    void Calc()
    {
        Context* pContext = static_cast<Context*>(m_ContextBlock.GetData());
        nw::g3d::Copy32<true>(pContext, &m_Context, sizeof(m_Context) >> 2);
        m_ContextBlock.DCFlush();
    }

    const nw::g3d::GfxBuffer& GetContextBlock() const { return m_ContextBlock; }
    Context& GetContext() { return m_Context; }
    //const Context& GetContext() const { return m_Context; }

private:
    nw::g3d::GfxBuffer m_ContextBlock;
    Context m_Context;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(TownContext);

} s_Context;

class TexAverageShader
{
public:
    TexAverageShader() :
        m_pResShadingModel(NULL),
        m_pShadingModelObj(NULL),
        m_pSelector(NULL),
        m_pTargetSampler(NULL)
    {
    }

    void Cleanup()
    {
        if (m_pShadingModelObj)
        {
            g3ddemo::DestroyShadingModelObj(m_pShadingModelObj);
            m_pShadingModelObj = NULL;
        }
        if (m_pSelector)
        {
            g3ddemo::DestroyShaderSelector(m_pSelector);
            m_pSelector = NULL;
        }
        if (m_pTargetSampler)
        {
            m_pTargetSampler->Cleanup();
            g3ddemo::FreeMem2(m_pTargetSampler);
            m_pTargetSampler = NULL;
        }
    }

    void Setup(nw::g3d::ResShadingModel* pResShadingModel)
    {
        NW_G3D_ASSERT_NOT_NULL(pResShadingModel);
        m_pResShadingModel = pResShadingModel;
        {
            nw::g3d::ShadingModelObj::InitArg arg(m_pResShadingModel);
            m_pShadingModelObj = g3ddemo::CreateShadingModelObj(arg);
            NW_G3D_ASSERT_NOT_NULL(m_pShadingModelObj);
        }
        {
            nw::g3d::ShaderSelector::InitArg arg(m_pShadingModelObj);
            m_pSelector = g3ddemo::CreateShaderSelector(arg);
            NW_G3D_ASSERT_NOT_NULL(m_pSelector);
        }
        {
            void* pBuffer = g3ddemo::AllocMem2(sizeof(nw::g3d::GfxSampler));
            m_pTargetSampler = new(pBuffer) nw::g3d::GfxSampler();
            NW_G3D_ASSERT_NOT_NULL(m_pTargetSampler);
            m_pTargetSampler->SetMinFilter(GX2_TEX_XY_FILTER_POINT);
            m_pTargetSampler->SetMagFilter(GX2_TEX_XY_FILTER_POINT);
            m_pTargetSampler->Setup();
        }
        m_IdxTileResolutionX = m_pResShadingModel->GetDynamicOptionIndex("tile_resolution_x");
        m_IdxTileResolutionY = m_pResShadingModel->GetDynamicOptionIndex("tile_resolution_y");
        NW_G3D_ASSERT(m_IdxTileResolutionX >= 0);
        NW_G3D_ASSERT(m_IdxTileResolutionY >= 0);
        m_IdxTargetSampler = m_pResShadingModel->GetSamplerIndex("TargetTexture");
        NW_G3D_ASSERT(m_IdxTargetSampler >= 0);
    }

    void SetTileResolution(uint x, uint y)
    {
        char str[16];
        int choice;
        sprintf(str, "%d", x);
        choice = m_pSelector->GetDynamicOption(m_IdxTileResolutionX)->GetChoiceIndex(str);
        NW_G3D_ASSERT(choice >= 0);
        m_pSelector->WriteDynamicKey(m_IdxTileResolutionX, choice);
        sprintf(str, "%d", y);
        choice = m_pSelector->GetDynamicOption(m_IdxTileResolutionY)->GetChoiceIndex(str);
        NW_G3D_ASSERT(choice >= 0);
        m_pSelector->WriteDynamicKey(m_IdxTileResolutionY, choice);
    }

    void LoadTextureSampler(const nw::g3d::GfxTexture* pTexture)
    {
        g3ddemo::LoadTextureSampler(
            pTexture, m_pTargetSampler, m_pSelector->GetProgram(), m_IdxTargetSampler);
    }
    void Update() { m_pSelector->UpdateVariation(); }
    void Load() { m_pSelector->GetProgram()->Load(); }

private:
    nw::g3d::ResShadingModel* m_pResShadingModel;
    nw::g3d::ShadingModelObj* m_pShadingModelObj;
    nw::g3d::ShaderSelector* m_pSelector;
    nw::g3d::GfxSampler* m_pTargetSampler;
    int m_IdxTargetSampler;
    int m_IdxTileResolutionX;
    int m_IdxTileResolutionY;
} s_TexAverageShader;

} // anonymous namespace

int TownMain(int argc, const char* argv[])
{
    (void)argc;
    (void)argv;

    s_SelectedModel = 6; // サーモン

    s_IsNormalMode = true;

    s_EnableFlag = INITIAL_FLAG;

#ifdef _WIN32
    char* pPath = NULL;
    if ((pPath = getenv("NW4F_G3D_ROOT")) != NULL)
    {
        s_pRootPath = pPath;
    }
    else if ((pPath = getenv("NW4F_ROOT")) != NULL)
    {
        s_pRootPath = pPath;
    }
#endif

    // 初期化処理。
    g3ddemo::Init();
    g3ddemo::InitDisplayArg initDisplayArg;
#ifdef _WIN32
    initDisplayArg.modeDRC = GX2_DRC_NONE;
#endif
    g3ddemo::InitDisplay(initDisplayArg);

    g3ddemo::Pad pad;
    if (!pad.Reset())
    {
        g3ddemo::PostQuitMsg();
    }

    // グラフィックスリソースの初期化処理。
    nw::g3d::GfxContext::Prepare(); // 構築に GL コンテクストが必要。
    nw::g3d::GfxContext* pCtx = nw::g3d::demo::AllocMem2<nw::g3d::GfxContext>(
        sizeof(*pCtx), GX2_CONTEXT_STATE_ALIGNMENT);
    pCtx->Setup();
    pCtx->TempPrepare();

    g3ddemo::FrameBuffer frameBufferTV;
    {
        g3ddemo::FrameBuffer::InitArg initArg(1280, 720);
        initArg.colorBufferFTV = true;
        size_t bufferSize = g3ddemo::FrameBuffer::CalcSize(initArg);
        frameBufferTV.Init(initArg, g3ddemo::AllocMem2(bufferSize), bufferSize);
        frameBufferTV.Setup();
        frameBufferTV.Alloc(g3ddemo::AllocMem1);
    }

    g3ddemo::FrameBuffer frameBufferWater;
    {
        g3ddemo::FrameBuffer::InitArg initArg(640, 360);
        size_t bufferSize = g3ddemo::FrameBuffer::CalcSize(initArg);
        frameBufferWater.Init(initArg, g3ddemo::AllocMem2(bufferSize), bufferSize);
        frameBufferWater.Setup();
        frameBufferWater.GetDepthBufferTexture()->Alloc(g3ddemo::AllocMem1);
    }

    g3ddemo::FrameBuffer frameBufferShadow;
    {
        g3ddemo::FrameBuffer::InitArg initArg(640, 640);
        size_t bufferSize = g3ddemo::FrameBuffer::CalcSize(initArg);
        frameBufferShadow.Init(initArg, g3ddemo::AllocMem2(bufferSize), bufferSize);
        nw::g3d::fnd::GfxTexture& depthTex = frameBufferShadow.GetDepthBufferTexture()->texture;
        depthTex.GetGX2Texture()->surface.dim = GX2_SURFACE_DIM_2D_ARRAY;
        depthTex.GetGX2Texture()->surface.depth = s_nCascade;
        frameBufferShadow.Setup();
    }

    g3ddemo::FrameBuffer frameBufferGeometry;
    {
        g3ddemo::FrameBuffer::InitArg initArg(640, 360);
        size_t bufferSize = g3ddemo::FrameBuffer::CalcSize(initArg);
        frameBufferGeometry.Init(initArg, g3ddemo::AllocMem2(bufferSize), bufferSize);
        frameBufferGeometry.Setup();
    }

    g3ddemo::FrameBuffer frameBufferLight;
    {
        g3ddemo::FrameBuffer::InitArg initArg(640, 360);
        initArg.colorBufferFormat = GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM;
        initArg.useDepthBuffer = false;
        size_t bufferSize = g3ddemo::FrameBuffer::CalcSize(initArg);
        frameBufferLight.Init(initArg, g3ddemo::AllocMem2(bufferSize), bufferSize);
        frameBufferLight.Setup();
    }

    nw::g3d::GfxColorBuffer& colorBuffer =
        frameBufferTV.GetColorBufferTexture(GX2_RENDER_TARGET_0)->renderBuffer;
    nw::g3d::GfxDepthBuffer& depthBuffer = frameBufferTV.GetDepthBufferTexture()->renderBuffer;

    nw::g3d::GfxColorBuffer& colorBufferWater =
        frameBufferWater.GetColorBufferTexture(GX2_RENDER_TARGET_0)->renderBuffer;
    nw::g3d::GfxDepthBuffer& depthBufferWater = frameBufferWater.GetDepthBufferTexture()->renderBuffer;

    nw::g3d::GfxDepthBuffer& depthBufferShadow = frameBufferShadow.GetDepthBufferTexture()->renderBuffer;

    nw::g3d::GfxColorBuffer& colorBufferNormal =
        frameBufferGeometry.GetColorBufferTexture(GX2_RENDER_TARGET_0)->renderBuffer;
    nw::g3d::GfxDepthBuffer& depthBufferGeometry = frameBufferGeometry.GetDepthBufferTexture()->renderBuffer;

    nw::g3d::GfxColorBuffer& colorBufferLight =
        frameBufferLight.GetColorBufferTexture(GX2_RENDER_TARGET_0)->renderBuffer;

    GX2Surface& surfaceWater = s_WaterTexture.GetGX2Texture()->surface;
    surfaceWater.width = frameBufferWater.GetWidth();
    surfaceWater.height = frameBufferWater.GetHeight();
    surfaceWater.format = GX2_SURFACE_FORMAT_TC_R11_G11_B10_FLOAT;
    s_WaterTexture.CalcSize();
    s_WaterTexture.SetImagePtrs(g3ddemo::AllocMem1(s_WaterTexture.GetBaseSize(), s_WaterTexture.GetAlignment()), NULL);
    s_WaterTexture.Setup();

    GX2Surface& surfaceShadow = s_ShadowTexture.GetGX2Texture()->surface;
    surfaceShadow.width = frameBufferShadow.GetWidth();
    surfaceShadow.height = frameBufferShadow.GetHeight();
    surfaceShadow.dim = GX2_SURFACE_DIM_2D_ARRAY;
    surfaceShadow.depth = s_nCascade;
    surfaceShadow.format = GX2_SURFACE_FORMAT_TCD_R32_FLOAT;
    surfaceShadow.use = GX2_SURFACE_USE_DEPTH_BUFFER_TEXTURE;
    s_ShadowTexture.GetGX2Texture()->viewNumSlices = s_nCascade;
    s_ShadowTexture.CalcSize();
    s_ShadowTexture.SetImagePtrs(g3ddemo::AllocMem1(s_ShadowTexture.GetBaseSize(), s_ShadowTexture.GetAlignment()), NULL);
    s_ShadowTexture.Setup();

    GX2Surface& surfaceColor = s_ColorTexture.GetGX2Texture()->surface;
    surfaceColor.width = frameBufferTV.GetWidth();
    surfaceColor.height = frameBufferTV.GetHeight();
    surfaceColor.format = GX2_SURFACE_FORMAT_TC_R11_G11_B10_FLOAT;
    s_ColorTexture.CalcSize();
    s_ColorTexture.SetImagePtrs(g3ddemo::AllocMem1(s_ColorTexture.GetBaseSize(), s_ColorTexture.GetAlignment()), NULL);
    s_ColorTexture.Setup();

    GX2Surface& surfaceNormal = s_NormalTexture.GetGX2Texture()->surface;
    surfaceNormal.width = frameBufferGeometry.GetWidth();
    surfaceNormal.height = frameBufferGeometry.GetHeight();
    surfaceNormal.format = GX2_SURFACE_FORMAT_TC_R11_G11_B10_FLOAT;
    s_NormalTexture.CalcSize();
    s_NormalTexture.SetImagePtrs(g3ddemo::AllocMem1(s_NormalTexture.GetBaseSize(), s_NormalTexture.GetAlignment()), NULL);
    s_NormalTexture.Setup();

    GX2Surface& surfaceLight = s_LightTexture.GetGX2Texture()->surface;
    surfaceLight.width = frameBufferLight.GetWidth();
    surfaceLight.height = frameBufferLight.GetHeight();
    surfaceLight.format = GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM;
    s_LightTexture.CalcSize();
    s_LightTexture.SetImagePtrs(g3ddemo::AllocMem1(s_LightTexture.GetBaseSize(), s_LightTexture.GetAlignment()), NULL);
    s_LightTexture.Setup();

    GX2Surface& surfaceDepth = s_DepthTexture.GetGX2Texture()->surface;
    surfaceDepth.width = frameBufferGeometry.GetWidth();
    surfaceDepth.height = frameBufferGeometry.GetHeight();
    surfaceDepth.format = GX2_SURFACE_FORMAT_TCD_R32_FLOAT;
    surfaceDepth.use = GX2_SURFACE_USE_DEPTH_BUFFER_TEXTURE;
    s_DepthTexture.CalcSize();
    s_DepthTexture.SetImagePtrs(g3ddemo::AllocMem1(s_DepthTexture.GetBaseSize(), s_DepthTexture.GetAlignment()), NULL);
    s_DepthTexture.Setup();

    s_Screen.Setup(1024);
    s_Screen.SetShapeColor(0x7F, 0x7F, 0x7F, 0x7F);
    s_Screen.SetFontColor(0xFF, 0xFF, 0xFF);
    s_Screen.SetFontSize(2.5f);

    s_Meter.Setup();

#if NW_G3D_CONFIG_USE_HOSTIO
    // edit library
    {
        nw::g3d::edit::EditServer::CreateArg arg;
        arg.allocator = &s_Allocator;
        arg.editCallback = &s_Editor;

        bool createResult = nw::g3d::edit::EditServer::CreateInstance(arg);
        NW_G3D_ASSERT(createResult);

        bool openConnectionResult = nw::g3d::edit::EditServer::Instance().Open();
        NW_G3D_ASSERT(openConnectionResult);
    }
#endif

    s_Context.Setup();

    // ビューの Uniform Block
    s_View.Setup();

    s_Holder.Init();

    // シェーダの初期化。
    u32 maxInputRingItem = 0;
    u32 maxOutputRingItem = 0;
    for (const char** ppPath = BFSHA_PATH; *ppPath != NULL; ++ppPath)
    {
        g3ddemo::FileHandle handle = 0;
        if (g3ddemo::OpenFile(&handle, ppPath[0], "r"))
        {
            continue;
        }
        g3ddemo::CloseFile(handle);
        void* pFile = g3ddemo::LoadFile(ppPath[0], NULL, GX2_SHADER_ALIGNMENT);
        if (!nw::g3d::ResShaderArchive::IsValid(pFile))
        {
            continue;
        }
        nw::g3d::ResShaderArchive* pShaderArchive = nw::g3d::ResShaderArchive::ResCast(pFile);
        pShaderArchive->Setup();

        for (int idxShadingModel = 0, numShadingModel = pShaderArchive->GetShadingModelCount();
            idxShadingModel < numShadingModel; ++idxShadingModel)
        {
            nw::g3d::ResShadingModel* pShadingModel =
                pShaderArchive->GetShadingModel(idxShadingModel);
            maxInputRingItem = std::max(maxInputRingItem, pShadingModel->GetMaxInputRingItem());
            maxOutputRingItem = std::max(maxOutputRingItem, pShadingModel->GetMaxOutputRingItem());
        }

        s_Holder.shaders.PushBack(pShaderArchive);

        if (nw::g3d::ResShadingModel* pShadingModel = pShaderArchive->GetShadingModel("tex_average"))
        {
            s_TexAverageShader.Setup(pShadingModel);
            s_TexAverageShader.SetTileResolution(64, 64);
        }
    }

    // リソースファイルの初期化。
    for (const char** ppPath = BFRES_PATH; *ppPath != NULL; ++ppPath)
    {
        // ResFile のアライメントは内部に持っているテクスチャの最大アライメントに一致します。
        size_t size = 0;
        void* pFile = g3ddemo::LoadFile(ppPath[0], &size, TEXTURE_ALIGNMENT);
        NW_G3D_ASSERT(nw::g3d::ResFile::IsValid(pFile));
        nw::g3d::ResFile* pResFile = nw::g3d::ResFile::ResCast(pFile);
        pResFile->Setup();
        // ResFile 内のテクスチャや頂点を CPU で書き込んだ場合は
        // CPU のキャッシュを吐き出す必要があります。
        s_Holder.files.PushBack(pResFile);
    }

    // アニメーションリソースの登録
    g3ddemo::RegistAnim(&s_Holder);

    // サブメッシュ LOD 計算コールバック
    float thresholdsTV[] = { 1.0f, 0.4f, 0.2f, 0.1f };
    float thresholdsWater[] = { 1.0f, 0.8f, 0.4f, 0.2f };
    CalcLod calcLodTV(&s_View.GetViewMtx(VIEW_TV), 1.0f / nw::g3d::math::Math::Tan(
        nw::g3d::Math::DegToRad(45.0f) * 0.5f), thresholdsTV, sizeof(thresholdsTV) / sizeof(*thresholdsTV));
    CalcLod calcLodWater(&s_View.GetViewMtx(VIEW_WATER), 1.0f / nw::g3d::math::Math::Tan(
        nw::g3d::Math::DegToRad(45.0f) * 0.5f), thresholdsWater, sizeof(thresholdsWater) / sizeof(*thresholdsWater));

    // モデルリソースとシェーダの関連付け。
    for (int idxFile = 0, numFile = s_Holder.files.Size(); idxFile < numFile; ++idxFile)
    {
        nw::g3d::ResFile* pResFile = s_Holder.files[idxFile];

        for (int idxModel = 0, numModel = pResFile->GetModelCount();
            idxModel < numModel; ++idxModel)
        {
            nw::g3d::ResModel* pResModel = pResFile->GetModel(idxModel);

            // モデル固有シェーダのセットアップ
            nw::g3d::ResShaderArchive* pModelShader = NULL;
            nw::g3d::ResExternalFile* pResExt = NULL;
            if ((pResExt = pResFile->GetExternalFile(MODEL_SHADER)) != NULL)
            {
                void* pExtFile = pResExt->GetData();
                NW_G3D_ASSERT(nw::g3d::ResShaderArchive::IsValid(pExtFile));
                pModelShader = nw::g3d::ResShaderArchive::ResCast(pExtFile);
                pModelShader->Setup();

                for (int idxResShadingModel = 0, numResShadingModel =
                    pModelShader->GetShadingModelCount();
                    idxResShadingModel < numResShadingModel; ++idxResShadingModel)
                {
                    nw::g3d::ResShadingModel* pResShadingModel =
                        pModelShader->GetShadingModel(idxResShadingModel);
                    maxInputRingItem = (std::max)(maxInputRingItem,
                        pResShadingModel->GetMaxInputRingItem());
                    maxOutputRingItem = (std::max)(maxOutputRingItem,
                        pResShadingModel->GetMaxOutputRingItem());
                }
            }

            // シェーダ割り当ての構築。
            g3ddemo::ModelAssign* pModelAssign =
                SetupModelAssign(pResModel, NULL, 0, pModelShader);
            s_Holder.assigns.PushBack(pModelAssign);

            if (strcmp(pResFile->GetName(), "Light") == 0)
            {
                continue;
            }

            // モデルインスタンスの構築。
            g3ddemo::CreateModelObjArg arg(pResModel);
            arg.initArg.ViewCount(NUM_VIEW);
            arg.initArg.EnableBounding();
            nw::g3d::ModelObj* pModelObj = g3ddemo::CreateModelObj(arg);
            g3ddemo::UserModel* pUserModel = SetupUserModel(pModelObj);
#if NW_G3D_CONFIG_USE_HOSTIO
            pUserModel->EnableEditCalcSkeletalAnimations();
#endif
            pUserModel->EnableFrustumCulling();
            s_Holder.models.PushBack(pUserModel);

            SetupWaterVariation(pModelObj);

            if (strcmp(pResModel->GetName(), "env") == 0)
            {
                s_Holder.pEnvModel = pUserModel;
                nw::g3d::ResTexture* pResWater = pResFile->GetTexture("water");
                pResWater->Cleanup();
                pResWater->ref().gfxTexture = s_WaterTexture;
                colorBufferWater.SetImagePtrs(pResWater->GetGfxTexture());
                nw::g3d::ResTexture* pResShadow = pResFile->GetTexture("shadow");
                pResShadow->Cleanup();
                pResShadow->ref().gfxTexture = s_ShadowTexture;
                depthBufferShadow.SetImagePtrs(pResShadow->GetGfxTexture());
                nw::g3d::ResTexture* pResColor = pResFile->GetTexture("color");
                pResColor->Cleanup();
                pResColor->ref().gfxTexture = s_ColorTexture;
                nw::g3d::ResTexture* pResNormal = pResFile->GetTexture("normal");
                pResNormal->Cleanup();
                pResNormal->ref().gfxTexture = s_NormalTexture;
                colorBufferNormal.SetImagePtrs(pResNormal->GetGfxTexture());
                nw::g3d::ResTexture* pResLight = pResFile->GetTexture("light");
                pResLight->Cleanup();
                pResLight->ref().gfxTexture = s_LightTexture;
                colorBufferLight.SetImagePtrs(pResLight->GetGfxTexture());
                nw::g3d::ResTexture* pResDepth = pResFile->GetTexture("depth");
                pResDepth->Cleanup();
                pResDepth->ref().gfxTexture = s_DepthTexture;
                depthBufferGeometry.SetImagePtrs(pResDepth->GetGfxTexture());

                nw::g3d::ResMaterial* pResMat = pResModel->GetMaterial(0);
                nw::g3d::GfxSampler* pGfxSampler = pResMat->GetSampler("_d0")->GetGfxSampler();
                pGfxSampler->SetClampX(GX2_TEX_CLAMP_CLAMP_BORDER);
                pGfxSampler->SetClampY(GX2_TEX_CLAMP_CLAMP_BORDER);
                pGfxSampler->SetBorderType(GX2_TEX_BORDER_SOLID_WHITE);
                pGfxSampler->SetCompareEnable(GX2_TRUE);
                pGfxSampler->SetCompareFunc(GX2_COMPARE_LESS);
                pGfxSampler->UpdateRegs();
            }
            else if (strcmp(pResModel->GetName(), "bg_WhiteTown") == 0)
            {
                pUserModel->SetCalcLodLevel(VIEW_TV, &calcLodTV);
                pUserModel->SetCalcLodLevel(VIEW_WATER, &calcLodWater);
                pUserModel->SetCalcLodLevel(VIEW_LIGHT, &calcLodTV);
            }
            else if (strcmp(pResModel->GetName(), "FemaleA") == 0)
            {
                pUserModel->GetLayoutMtx().SetR(nw::g3d::math::Vec3::Make(0.0f, nw::g3d::Math::DegToRad(60.0f), 0.0f));
                pUserModel->GetLayoutMtx().SetT(nw::g3d::math::Vec3::Make(-300.0f, 12.0f, 1550.0f));
            }
            else if (strcmp(pResModel->GetName(), "MaleA") == 0)
            {
                pUserModel->GetLayoutMtx().SetT(nw::g3d::math::Vec3::Make(0.0f, 15.0f, 170.0f));
            }
        }
    }

    // UserShadingModel の構築
    for (int idxShader = 0, numShader = s_Holder.shaders.Size(); idxShader < numShader; ++idxShader)
    {
        SetupUserShadingModel(s_Holder.shaders[idxShader]);
    }
    for (int idxAssign = 0, numAssign = s_Holder.assigns.Size(); idxAssign < numAssign; ++idxAssign)
    {
        SetupUserShadingModel(s_Holder.assigns[idxAssign]->GetModelShader());
    }

    // アニメーションの登録
    AddAnim("bg_WhiteTown", "bg_WhiteTown", 1.0f);
    AddAnim("bg_WhiteTown_cloth", "bg_WhiteTown_cloth", 1.0f);
    AddAnim("Duck", "Duck", 1.0f);
    AddAnim("FemaleA", "FemaleA_MoAnime", 1.0f);
    AddAnim("Fish", "Fish", 1.0f);
    AddAnim("MaleA", "MaleA_MoAnime", 1.0f);
    AddAnim("WhiteTown_Water_Caustics", "WhiteTown_Water_Caustics", 0.2f);
    AddAnim("bg_WhiteTown_Water_Flat", "bg_WhiteTown_Water_Flat", 0.5f);

    // ライトインスタンスの構築
    const int max_lights = 256;
    size_t bufferSize = sizeof(g3ddemo::UserModel*) * max_lights;
    s_Lights.SetBuffer(g3ddemo::AllocMem2(bufferSize), bufferSize);
    nw::g3d::ResFile* pLightFile = NULL;
    for (int idxFile = 0, nFiles = s_Holder.files.Size(); idxFile < nFiles; ++idxFile)
    {
        if (strcmp(s_Holder.files[idxFile]->GetName(), "Light") == 0)
        {
            pLightFile = s_Holder.files[idxFile];
        }
    }
    if (pLightFile)
    {
        nw::g3d::ResModel* pResPoint = pLightFile->GetModel("Point");
        g3ddemo::UserModel* pTownModel = NULL;
        for (int idxModel = 0, nModels = s_Holder.models.Size(); idxModel < nModels; ++idxModel)
        {
            g3ddemo::UserModel* pUserModel = s_Holder.models[idxModel];
            if (strcmp(pUserModel->GetModelObj()->GetResource()->GetName(), "bg_WhiteTown") == 0)
            {
                pTownModel = pUserModel;
            }
        }
        if (pTownModel)
        {
            pTownModel->CalcWorld();
            pTownModel->CalcAnim();

            const char* pointLightPrefix = "PointLight_";
            const nw::g3d::Vec3 pointLightScale = nw::g3d::Vec3::Make(
                s_PointLightMaxRadius, s_PointLightMaxRadius, s_PointLightMaxRadius);
            const nw::g3d::Vec3 pointLightColor = nw::g3d::Vec3::Make(1.0f, 0.8f, 0.4f);
            nw::g3d::SkeletonObj* pSkeleton = pTownModel->GetModelObj()->GetSkeleton();
            for (int idxBone = 0, nBones = pSkeleton->GetBoneCount(); idxBone < nBones; ++idxBone)
            {
                nw::g3d::ResBone* pBone = pSkeleton->GetBone(idxBone);
                if (strncmp(pBone->GetName(), pointLightPrefix, strlen(pointLightPrefix)) == 0)
                {
                    g3ddemo::CreateModelObjArg arg(pResPoint);
                    arg.initArg.ViewCount(NUM_VIEW);
                    nw::g3d::ModelObj* pModelObj = g3ddemo::CreateModelObj(arg);
                    g3ddemo::UserModel* pPointLight = SetupUserModel(pModelObj);
                    pPointLight->EnableFrustumCulling();
                    const nw::g3d::Mtx34& mtx = pSkeleton->GetWorldMtxArray()[idxBone];
                    nw::g3d::Vec3 center = nw::g3d::Vec3::Make(mtx.m03, mtx.m13, mtx.m23);

                    pPointLight->GetLayoutMtx().SetS(pointLightScale);
                    pPointLight->GetLayoutMtx().SetT(center);

                    nw::g3d::MaterialObj* pMatObj = pModelObj->GetMaterial(0);
                    int locCenter = pMatObj->GetShaderParamIndex("center");
                    int locRadius = pMatObj->GetShaderParamIndex("radius");
                    int locColor = pMatObj->GetShaderParamIndex("color");
                    *pMatObj->EditShaderParam<nw::g3d::Vec3>(locCenter) = center;
                    *pMatObj->EditShaderParam<float>(locRadius) = s_PointLightMaxRadius;
                    *pMatObj->EditShaderParam<nw::g3d::Vec3>(locColor) = pointLightColor;

                    s_Lights.PushBack(pPointLight);
                }
            }
        }
    }

    // ジオメトリシェーダ用にリングバッファの設定を行います。
    nw::g3d::GfxRingBuffer ringBuffer;
    {
        size_t inputSize = nw::g3d::GfxRingBuffer::CalcInputBufferSize(maxInputRingItem);
        if (inputSize > 0)
        {
            ringBuffer.SetInputBuffer(g3ddemo::AllocMem1(inputSize, GX2_SHADER_ALIGNMENT), inputSize);
        }
        size_t outputSize = nw::g3d::GfxRingBuffer::CalcOutputBufferSize(maxOutputRingItem);
        if (outputSize > 0)
        {
            ringBuffer.SetOutputBuffer(g3ddemo::AllocMem1(outputSize, GX2_SHADER_ALIGNMENT), outputSize);
        }
        ringBuffer.Load();
    }

    // 演算シェーダ用に SSBO とディスパッチパラメータの設定を行います。
    int expBufSize = s_TileDimX * s_TileDimY * sizeof(nw::g3d::Vec4);
    void* expGPUBuf = g3ddemo::AllocMem2(expBufSize, GX2_EXPORT_BUFFER_ALIGNMENT);
    g3ddemo::ExportBuffer expBuf;
    expBuf.Setup(expGPUBuf, expBufSize);
    nw::g3d::GfxDispatchParams* pDispatchParams = static_cast<nw::g3d::GfxDispatchParams*>(
        g3ddemo::AllocMem2(sizeof(nw::g3d::GfxDispatchParams), nw::g3d::GfxDispatchParams::CLASS_ALIGNMENT));
    pDispatchParams->SetParams(s_TileDimX, s_TileDimY);
    pDispatchParams->CPUFlush();

    // カメラ初期化。
    {
        s_CameraPosition.Set(380.0f, 140.0f, 2400.0f);
        s_CameraUp.Set(0.0f, 1.0f, 0.0f);
        s_CameraTarget.Set(-400.0f, 0.0f, 0.0f);
        s_LightPosition.Set(2000.0f, 2200.0f, 2000.0f);

        s_View.GetViewMtx(VIEW_TV).LookAt(
            s_CameraPosition,
            s_CameraUp,
            s_CameraTarget);
    }

    s_FrameDL.Setup(2);

    int locRadius = -1;
    if (s_Lights.Size() > 0)
    {
        locRadius = s_Lights[0]->GetModelObj()->GetMaterial(0)->GetShaderParamIndex("radius");
    }
    float pointLightRadius = 0.0f;
    float vRadius = s_PointLightMaxRadius * 0.1f;

    nw::g3d::ut::PerfManager::Init();
    s64 beforeTick = 0;
    bool restart = false;
    bool nextFrame = true;
#ifdef _WIN32
    nextFrame = false;
#endif
    float diffIntensity = 1.0f;

    // メインループ
    for (int frame = 0; g3ddemo::ProcessMsg(); ++frame)
    {
        pCtx->TempPrepare();
        int idxFrame = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_FRAME);

        s64 tick = nw::g3d::ut::PerfManager::GetTicks();
        float second = nw::g3d::ut::PerfManager::ToSeconds(tick - beforeTick);
        beforeTick = tick;

#if NW_G3D_CONFIG_USE_HOSTIO
        if (s_AssignChanged)
        {
            s_FrameDL.Reset();
        }
        else
#endif
        {
            s_FrameDL.DirectCall();
        }

        if (s_SelectedModel < 0 || s_SelectedModel >= s_Holder.models.Size())
        {
            s_SelectedModel = 0;
        }

        if (!pad.Read() || pad.IsTriggered(g3ddemo::Pad::BUTTON_START))
        {
            if (pad.IsHold(g3ddemo::Pad::TRIGGER_R))
            {
                restart = true;
            }
            g3ddemo::PostQuitMsg();
        }

        if (pad.IsTriggered(g3ddemo::Pad::TRIGGER_L))
        {
            s_IsNormalMode = !s_IsNormalMode;
        }

        if (s_IsNormalMode)
        {
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_DOWN))
            {
                if (++s_SelectedModel >= s_Holder.models.Size())
                {
                    s_SelectedModel = 0;
                }
            }
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_UP))
            {
                if (--s_SelectedModel < 0)
                {
                    s_SelectedModel = s_Holder.models.Size() - 1;
                }
            }
            if (pad.IsHold(g3ddemo::Pad::BUTTON_LEFT))
            {
                ChangeAnimStep(s_Holder.models[s_SelectedModel], -0.01f);
            }
            if (pad.IsHold(g3ddemo::Pad::BUTTON_RIGHT))
            {
                ChangeAnimStep(s_Holder.models[s_SelectedModel], 0.01f);
            }
            if (pad.IsHold(g3ddemo::Pad::TRIGGER_Z))
            {
                g3ddemo::LookAtModel(&s_CameraPosition, &s_CameraTarget,
                    s_Holder.models[s_SelectedModel]->GetModelObj());
            }
        }
        else
        {
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_DOWN))
            {
                s_EnableFlag = nw::g3d::ut::InvertFlag(s_EnableFlag, ENABLE_MODEL_ANIMATION);
            }
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_UP))
            {
                s_EnableFlag = nw::g3d::ut::InvertFlag(s_EnableFlag, ENABLE_LIGHT_ANIMATION);
            }
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_A))
            {
                s_EnableFlag = nw::g3d::ut::InvertFlag(s_EnableFlag, ENABLE_SHADOW_RENDERING);
            }
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_B))
            {
                s_EnableFlag = nw::g3d::ut::InvertFlag(s_EnableFlag, ENABLE_WATER_RENDERING);
            }
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_X))
            {
                s_EnableFlag = nw::g3d::ut::InvertFlag(s_EnableFlag, ENABLE_GEOMETRY_RENDERING);
            }
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_Y))
            {
                s_EnableFlag = nw::g3d::ut::InvertFlag(s_EnableFlag, ENABLE_LIGHT_RENDERING);
            }
            if (pad.IsTriggered(g3ddemo::Pad::TRIGGER_R))
            {
                s_EnableFlag = nw::g3d::ut::InvertFlag(s_EnableFlag, ENABLE_TEXTURE_COMPUTE);
            }
            if (pad.IsTriggered(g3ddemo::Pad::TRIGGER_Z))
            {
#ifndef _WIN32
                s_EnableFlag = nw::g3d::ut::InvertFlag(s_EnableFlag, ENABLE_DRC_RENDERING);
#endif
            }
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_LEFT))
            {
                s_EnableFlag = nw::g3d::ut::InvertFlag(s_EnableFlag, ENABLE_WIREFRAME);
                GX2PolygonMode mode = nw::g3d::CheckFlag(s_EnableFlag, ENABLE_WIREFRAME)
                    ? GX2_POLYGON_MODE_LINE : GX2_POLYGON_MODE_TRIANGLE;
                for (int idxModel = 0, nModels = s_Holder.models.Size(); idxModel < nModels; ++idxModel)
                {
                    nw::g3d::ModelObj* pModel = s_Holder.models[idxModel]->GetModelObj();
                    for (int idxMaterial = 0, nMaterials = pModel->GetMaterialCount();
                        idxMaterial < nMaterials; ++idxMaterial)
                    {
                        nw::g3d::MaterialObj* pMaterial = pModel->GetMaterial(idxMaterial);
                        nw::g3d::ResRenderState* pRenderState = pMaterial->GetResRenderState();
                        nw::g3d::GfxPolygonCtrl& polygonCtrl = pRenderState->GetPolygonCtrl();
                        polygonCtrl.SetPolygonModeEnable(GX2_TRUE);
                        polygonCtrl.SetPolygonModeBack(mode);
                        polygonCtrl.SetPolygonModeFront(mode);
                    }
                }
            }
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_RIGHT))
            {
                s_EnableFlag = nw::g3d::ut::InvertFlag(s_EnableFlag, ENABLE_ICE);
                const char* choice = nw::g3d::CheckFlag(s_EnableFlag, ENABLE_ICE) ? "1" : "0";
                for (int idxModel = 0, nModels = s_Holder.models.Size(); idxModel < nModels; ++idxModel)
                {
                    nw::g3d::ModelObj* pModel = s_Holder.models[idxModel]->GetModelObj();
                    for (int idxShape = 0, nShapes = pModel->GetShapeCount();
                        idxShape < nShapes; ++idxShape)
                    {
                        nw::g3d::ShapeObj* pShape = pModel->GetShape(idxShape);
                        nw::g3d::ResRenderInfo* pIceInfo = pModel->GetMaterial(pShape->
                            GetMaterialIndex())->GetResource()->GetRenderInfo("ice");
                        if (pIceInfo && *pIceInfo->GetInt())
                        {
                            g3ddemo::UserShape* pUserShape = pShape->GetUserPtr<g3ddemo::UserShape>();
                            for (int idxPass = 0; idxPass < g3ddemo::NUM_PASS; ++idxPass)
                            {
                                nw::g3d::ShaderSelector* pSelector = NULL;
                                int idxIce = -1;
                                if (( pSelector = pUserShape->GetShaderSelector(
                                    static_cast<g3ddemo::PassType>(idxPass)) ) != NULL
                                    && ( idxIce = pSelector->GetDynamicOptionIndex("ice_world") ) >= 0)
                                {
                                    nw::g3d::ShadingModelObj* pShadingModel = pSelector->GetShadingModel();
                                    nw::g3d::ResShaderOption* pOption = pSelector->GetDynamicOption(idxIce);
                                    const nw::g3d::ShaderRange& range = pShadingModel->GetShaderRange();
                                    int before = pSelector->ReadDynamicKey(idxIce);
                                    pSelector->WriteDynamicKey(idxIce, pOption->GetChoiceIndex(choice));
                                    if (pShadingModel->GetResource()->FindProgramIndex(
                                        range, pSelector->GetDynamicKey()) == nw::g3d::SHADER_PROGRAM_NONE)
                                    {
                                        pSelector->WriteDynamicKey(idxIce, before);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

#if NW_G3D_CONFIG_USE_HOSTIO
        if (s_IsNormalMode)
        {
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_A))
            {
                g3ddemo::UserModel* pUserModel = s_Holder.models[s_SelectedModel];
                pUserModel->AttachToEdit();
            }
            else if (pad.IsTriggered(g3ddemo::Pad::BUTTON_B))
            {
                AttachShader("tex_average");
            }
            else if (pad.IsTriggered(g3ddemo::Pad::BUTTON_Y))
            {
                AttachShader("gbuffer");
            }
            else if (pad.IsTriggered(g3ddemo::Pad::BUTTON_X))
            {
                AttachShader("shadow");
            }
        }
#endif

        // 計算
        {
            int idxCalcCPU = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_CPU);
            pCtx->TempPrepare();

            // カメラコントロール
            UpdateCamera(pad);
            s_View.GetViewMtx(VIEW_WATER).LookAt(nw::g3d::math::Vec3::Make(
                s_CameraPosition.x, -s_CameraPosition.y, s_CameraPosition.z), s_CameraUp,
                nw::g3d::math::Vec3::Make(s_CameraTarget.x, -s_CameraTarget.y, s_CameraTarget.z));
            s_View.GetViewMtx(VIEW_LIGHT).LookAt(s_LightPosition,
                nw::g3d::math::Vec3::Make(0.0f, 1.0f, 0.0f),
                nw::g3d::math::Vec3::Make(0.0f, 0.0f, 500.0f));

            // 投影行列。
            s_View.GetProjMtx(VIEW_TV).Perspective(
                nw::g3d::Math::DegToRad(45.0f), 16.0f / 9.0f, 10.0f, 40000.0f);
            s_View.GetProjMtx(VIEW_WATER).Perspective(nw::g3d::Math::DegToRad(45.0f),
                static_cast<float>(frameBufferWater.GetWidth()) / static_cast<float>(frameBufferWater.GetHeight()),
                10.0f, 40000.0f);
            s_View.GetProjMtx(VIEW_LIGHT).Perspective(nw::g3d::Math::DegToRad(45.0f),
                static_cast<float>(frameBufferShadow.GetWidth()) / static_cast<float>(frameBufferShadow.GetHeight()),
                1800.0f, 10000.0f);

            // 環境マップ用行列。
            s_TexMtxEnv.LookAt(s_CameraPosition, s_CameraUp, s_CameraTarget);
            s_TexMtxEnv.SetT(nw::g3d::Vec3::Make(0.0f, 0.0f, 0.0f));

            // 投影マップ用行列。
            s_TexMtxProj.TexProjPerspective(nw::g3d::Math::DegToRad(45.0f), 1.0f);
            s_TexMtxProj.Mul(s_TexMtxProj, s_View.GetViewMtx(0));

            for (int idxModel = 0, nModels = s_Holder.models.Size(); idxModel < nModels; ++idxModel)
            {
                g3ddemo::UserModel* pUserModel = s_Holder.models[idxModel];
                pUserModel->CalcWorld();
                if (nw::g3d::CheckFlag(s_EnableFlag, ENABLE_MODEL_ANIMATION))
                {
                    pUserModel->CalcAnim();
                    pUserModel->UpdateFrame();
                }
                pUserModel->UpdateShader();
            }
#if NW_G3D_CONFIG_USE_HOSTIO
            nw::g3d::edit::EditServer::Instance().CalcAnimations();
#endif
            if (s_Holder.pEnvModel)
            {
                s_Holder.pEnvModel->UpdateShader();
            }
            pointLightRadius += vRadius;
            if (pointLightRadius < 0.0f)
            {
                vRadius = fabs(vRadius);
                pointLightRadius += vRadius;
            }
            else if(pointLightRadius > s_PointLightMaxRadius)
            {
                vRadius = -fabs(vRadius);
                pointLightRadius += vRadius;
            }
            for (int idxLight = 0, nLights = s_Lights.Size(); idxLight < nLights; ++idxLight)
            {
                g3ddemo::UserModel* pUserModel = s_Lights[idxLight];
                if (nw::g3d::CheckFlag(s_EnableFlag, ENABLE_LIGHT_ANIMATION))
                {
                    nw::g3d::MaterialObj* pMaterial = pUserModel->GetModelObj()->GetMaterial(0);
                    *pMaterial->EditShaderParam<float>(locRadius) = pointLightRadius;
                }
                pUserModel->CalcWorld();
                pUserModel->UpdateShader();
            }
            s_Meter.EndTimeSpan(idxCalcCPU);
        }

        s_TexAverageShader.Update();

        // GPU 演算結果
        nw::g3d::Vec4 texAverage = nw::g3d::Vec4::Make(0.0f, 0.0f, 0.0f, 0.0f);
        {
            nw::g3d::Vec4 *results = static_cast<nw::g3d::Vec4*>(expBuf.GetCPUBuf());
            int numTile = s_TileDimX * s_TileDimY;
            for (int idxResult = 0; idxResult < numTile; ++idxResult)
            {
                texAverage.Add(texAverage, results[idxResult]);
            }
            texAverage.Div(texAverage, static_cast<float>(numTile));
        }

        {
            float ofsX = 1.5f;
            float ofsY = 2.0f;
            float line = s_Screen.GetFontHeight();
            s_Screen.Reset();
            s_Screen.SetShapeColor(0x7F, 0x7F, 0x7F, 0x7F);
            if (s_IsNormalMode)
            {
#if NW_G3D_CONFIG_USE_HOSTIO
                s_Screen.PutRect(ofsX, ofsY, 19.0f, line * 16);
#else
                s_Screen.PutRect(ofsX, ofsY, 19.0f, line * 12);
#endif
                s_Screen.PutString(ofsX, ofsY, "Edit Mode");
                ofsY += line;
                s_Screen.PutString(ofsX, ofsY, "L : Mode Change");
                ofsY += line;
                ofsY += line;
                if (s_SelectedModel < s_Holder.models.Size())
                {
                    s_Screen.PutStringFmt(ofsX, ofsY, "Selected Model : %s",
                        s_Holder.models[s_SelectedModel]->GetModelObj()->GetResource()->GetName());
                }
                else
                {
                    s_Screen.PutString(ofsX, ofsY, "Selected Model : ?");
                }
                ofsY += line;
                s_Screen.PutString(ofsX, ofsY, "LStick : Rotate");
                ofsY += line;
                s_Screen.PutString(ofsX, ofsY, "RStick : Translate");
                ofsY += line;
                s_Screen.PutString(ofsX, ofsY, "v ^ : Model Select");
                ofsY += line;
                s_Screen.PutString(ofsX, ofsY, "> < : Anim Speed");
                ofsY += line;
#if NW_G3D_CONFIG_USE_HOSTIO
                s_Screen.PutString(ofsX, ofsY, "A : Attach Model");
                ofsY += line;
                s_Screen.PutString(ofsX, ofsY, "B : Attach Average Shader");
                ofsY += line;
                s_Screen.PutString(ofsX, ofsY, "X : Attach Shadow Shader");
                ofsY += line;
                s_Screen.PutString(ofsX, ofsY, "Y : Attach Geometry Shader");
                ofsY += line;
#endif
                s_Screen.PutString(ofsX, ofsY, "ZR : Follow Model");
                ofsY += line;
                ofsY += line;
                s_Screen.PutStringFmt(ofsX, ofsY, "%d models are registered.", s_Holder.models.Size());
                ofsY += line;
                s_Screen.PutStringFmt(ofsX, ofsY, "FPS : %d", static_cast<int>(1.0f / second));
                ofsY += line;
            }
            else
            {
#ifdef _WIN32
                s_Screen.PutRect(ofsX, ofsY, 22.0f, line * 26);
#else
                s_Screen.PutRect(ofsX, ofsY, 22.0f, line * 27);
#endif
                s_Screen.PutString(ofsX, ofsY, "Debug Mode");
                ofsY += line;
                s_Screen.PutString(ofsX, ofsY, "L : Mode Change");
                ofsY += line;
                ofsY += line;
                s_Screen.PutString(ofsX, ofsY, "LStick : Rotate");
                ofsY += line;
                s_Screen.PutString(ofsX, ofsY, "RStick : Translate");
                ofsY += line;
                s_Screen.PutStringFmt(ofsX, ofsY, "v  : Model Animation  %s",
                    nw::g3d::CheckFlag(s_EnableFlag, ENABLE_MODEL_ANIMATION) ? "[Enable]" : "[Disable]");
                ofsY += line;
                s_Screen.PutStringFmt(ofsX, ofsY, "^  : Light Animation  %s",
                    nw::g3d::CheckFlag(s_EnableFlag, ENABLE_LIGHT_ANIMATION) ? "[Enable]" : "[Disable]");
                ofsY += line;
                s_Screen.PutStringFmt(ofsX, ofsY, "<  : Wireframe        %s",
                    nw::g3d::CheckFlag(s_EnableFlag, ENABLE_WIREFRAME) ? "[Enable]" : "[Disable]");
                ofsY += line;
                s_Screen.PutStringFmt(ofsX, ofsY, ">  : Ice              %s",
                    nw::g3d::CheckFlag(s_EnableFlag, ENABLE_ICE) ? "[Enable]" : "[Disable]");
                ofsY += line;
                s_Screen.PutStringFmt(ofsX, ofsY, "A  : Shadow (Red)     %s",
                    nw::g3d::CheckFlag(s_EnableFlag, ENABLE_SHADOW_RENDERING) ? "[Enable]" : "[Disable]");
                ofsY += line;
                s_Screen.PutStringFmt(ofsX, ofsY, "B  : Water (Blue)     %s",
                    nw::g3d::CheckFlag(s_EnableFlag, ENABLE_WATER_RENDERING) ? "[Enable]" : "[Disable]");
                ofsY += line;
                s_Screen.PutStringFmt(ofsX, ofsY, "X  : Geometry (Green) %s",
                    nw::g3d::CheckFlag(s_EnableFlag, ENABLE_GEOMETRY_RENDERING) ? "[Enable]" : "[Disable]");
                ofsY += line;
                s_Screen.PutStringFmt(ofsX, ofsY, "Y  : Light (Yellow)   %s",
                    nw::g3d::CheckFlag(s_EnableFlag, ENABLE_LIGHT_RENDERING) ? "[Enable]" : "[Disable]");
                ofsY += line;
                s_Screen.PutStringFmt(ofsX, ofsY, "     Opaque (Pink)    [Enable]");
                ofsY += line;
                s_Screen.PutStringFmt(ofsX, ofsY, "     Copy (White)     [Enable]");
                ofsY += line;
                s_Screen.PutStringFmt(ofsX, ofsY, "     Trans (Cyan)     [Enable]");
                ofsY += line;
                s_Screen.PutStringFmt(ofsX, ofsY, "R  : Average (Red)    %s",
                    nw::g3d::CheckFlag(s_EnableFlag, ENABLE_TEXTURE_COMPUTE) ? "[Enable]" : "[Disable]");
                ofsY += line;
                s_Screen.PutStringFmt(ofsX, ofsY, "     Info (Gray)      [Enable]");
                ofsY += line;
                s_Screen.PutStringFmt(ofsX, ofsY, "     Clear (Black)    [Enable]");
                ofsY += line;
#ifndef _WIN32
                s_Screen.PutStringFmt(ofsX, ofsY, "ZR : DRC Rendering    %s",
                    nw::g3d::CheckFlag(s_EnableFlag, ENABLE_DRC_RENDERING) ? "[Enable]" : "[Disable]");
                ofsY += line;
#endif
                ofsY += line;
                s_Screen.PutStringFmt(ofsX, ofsY, "%d models are registered.", s_Holder.models.Size());
                ofsY += line;
                s_Screen.PutStringFmt(ofsX, ofsY, "FPS : %d", static_cast<int>(1.0f / second));
                ofsY += line;
                if (nw::g3d::CheckFlag(s_EnableFlag, ENABLE_TEXTURE_COMPUTE))
                {
                    const float pixelAverageW = 22.0f;
                    s_Screen.PutStringFmt(ofsX, ofsY, "Pixel Average:",
                        texAverage.x, texAverage.y, texAverage.z, texAverage.w);
                    ofsY += line;
                    s_Screen.SetShapeColor(0xFF, 0, 0, 0x7F);
                    s_Screen.PutRect(ofsX, ofsY, pixelAverageW * texAverage.x, line);
                    ofsY += line;
                    s_Screen.SetShapeColor(0, 0xFF, 0, 0x7F);
                    s_Screen.PutRect(ofsX, ofsY, pixelAverageW * texAverage.y, line);
                    ofsY += line;
                    s_Screen.SetShapeColor(0, 0, 0xFF, 0x7F);
                    s_Screen.PutRect(ofsX, ofsY, pixelAverageW * texAverage.z, line);
                    ofsY += line;
                }
            }
        }

        // GPU のリードキャッシュはここで一括して Invalidate します。
        nw::g3d::GPUCache::InvalidateAll();

        s_FrameDL.Begin();

        // 影の描画
        if (nw::g3d::CheckFlag(s_EnableFlag, ENABLE_SHADOW_RENDERING))
        {
            depthBufferShadow.GetGX2DepthBuffer()->viewNumSlices = 1;
            for (int idxSlice = 0; idxSlice < s_nCascade; ++idxSlice)
            {
                int idxCPU = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_CPU);
                int idxShadow = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_GPU,
                    static_cast<u8>(0xFF - 0x20 * idxSlice), 0, 0, 0xFF, nextFrame);

                GX2ShaderMode shaderMode = GX2_SHADER_MODE_UNIFORM_BLOCK;
                nw::g3d::SetShaderMode(shaderMode);

                depthBufferShadow.GetGX2DepthBuffer()->viewFirstSlice = idxSlice;
                depthBufferShadow.UpdateRegs();

                frameBufferShadow.Load();

                for (int idxModel = 0, nModels = s_Holder.models.Size(); idxModel < nModels; ++idxModel)
                {
                    s_Holder.models[idxModel]->DebugDraw(g3ddemo::PASS_SHADOW,
                        VIEW_LIGHT + idxSlice, shaderMode);
                }

                s_Meter.EndTimeSpan(idxShadow);
                s_Meter.EndTimeSpan(idxCPU);
            }
        }

        // 水面の描画
        if (nw::g3d::CheckFlag(s_EnableFlag, ENABLE_WATER_RENDERING))
        {
            int idxCPU = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_CPU);
            int idxWater = s_Meter.BeginTimeSpan(
                g3ddemo::ProcessMeter::GROUP_GPU, 0, 0, 0xFF, 0xFF, nextFrame);

            frameBufferWater.Load();

            GX2ShaderMode shaderMode = GX2_SHADER_MODE_UNIFORM_BLOCK;
            nw::g3d::SetShaderMode(shaderMode);

            for (int idxModel = 0, nModels = s_Holder.models.Size(); idxModel < nModels; ++idxModel)
            {
                s_Holder.models[idxModel]->DebugDraw(g3ddemo::PASS_WATER, VIEW_WATER, shaderMode);
            }

            s_Meter.EndTimeSpan(idxWater);
            s_Meter.EndTimeSpan(idxCPU);
        }

        // 幾何学の描画
        if (nw::g3d::CheckFlag(s_EnableFlag, ENABLE_GEOMETRY_RENDERING))
        {
            int idxCPU = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_CPU);
            int idxGeometry = s_Meter.BeginTimeSpan(
                g3ddemo::ProcessMeter::GROUP_GPU, 0, 0xFF, 0, 0xFF, nextFrame);

            frameBufferGeometry.Load();

            GX2ShaderMode shaderMode = GX2_SHADER_MODE_UNIFORM_BLOCK;
            nw::g3d::SetShaderMode(shaderMode);

            for (int idxModel = 0, nModels = s_Holder.models.Size(); idxModel < nModels; ++idxModel)
            {
                s_Holder.models[idxModel]->DebugDraw(g3ddemo::PASS_GEOMETRY, VIEW_TV, shaderMode);
            }

            s_Meter.EndTimeSpan(idxGeometry);
            s_Meter.EndTimeSpan(idxCPU);
        }

        // 光の描画
        if (nw::g3d::CheckFlag(s_EnableFlag, ENABLE_LIGHT_RENDERING))
        {
            int timeSpanCPU = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_CPU);
            int timeSpanLight = s_Meter.BeginTimeSpan(
                g3ddemo::ProcessMeter::GROUP_GPU, 0xFF, 0xFF, 0, 0xFF, nextFrame);

            frameBufferLight.Load();

            GX2ShaderMode shaderMode = GX2_SHADER_MODE_UNIFORM_BLOCK;
            nw::g3d::SetShaderMode(shaderMode);

            for (int idxLight = 0, nLights = s_Lights.Size(); idxLight < nLights; ++idxLight)
            {
                s_Lights[idxLight]->DebugDraw(g3ddemo::PASS_DEFAULT, VIEW_TV, shaderMode);
            }

            s_Meter.EndTimeSpan(timeSpanLight);
            s_Meter.EndTimeSpan(timeSpanCPU);
        }

        // TV 描画
        {
            int idxCPU = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_CPU);
            int idxOpaque = s_Meter.BeginTimeSpan(
                g3ddemo::ProcessMeter::GROUP_GPU, 0xFF, 0x00, 0xFF, 0xFF, nextFrame);

            frameBufferTV.Load();

            GX2ShaderMode shaderMode = GX2_SHADER_MODE_UNIFORM_BLOCK;
            nw::g3d::SetShaderMode(shaderMode);

            for (int idxModel = 0, nModels = s_Holder.models.Size(); idxModel < nModels; ++idxModel)
            {
                s_Holder.models[idxModel]->DebugDraw(g3ddemo::PASS_OPAQUE, VIEW_TV, shaderMode);
            }

            s_Meter.EndTimeSpan(idxOpaque);
            s_Meter.EndTimeSpan(idxCPU);

            int idxCopy = s_Meter.BeginTimeSpan(
                g3ddemo::ProcessMeter::GROUP_SYS_GPU, 0xFF, 0xFF, 0xFF, 0xFF, nextFrame);
#if NW_G3D_IS_GX2
            GX2Invalidate(GX2_INVALIDATE_COLOR_BUFFER,
                colorBuffer.GetGX2ColorBuffer()->surface.imagePtr,
                colorBuffer.GetGX2ColorBuffer()->surface.imageSize);
            GX2Invalidate(GX2_INVALIDATE_TEXTURE,
                s_ColorTexture.GetGX2Texture()->surface.imagePtr,
                s_ColorTexture.GetGX2Texture()->surface.imageSize);
            GX2CopySurface(&colorBuffer.GetGX2ColorBuffer()->surface, 0, 0,
                &s_ColorTexture.GetGX2Texture()->surface, 0, 0);
#else
            glBindTexture(GL_TEXTURE_2D, s_ColorTexture.handle);
            glCopyTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, 0, 0,
                s_ColorTexture.GetWidth(), s_ColorTexture.GetHeight());
#endif
            pCtx->Activate();
            s_Meter.EndTimeSpan(idxCopy);

            idxCPU = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_CPU);
            int idxTranslucent = s_Meter.BeginTimeSpan(
                g3ddemo::ProcessMeter::GROUP_GPU, 0, 0xFF, 0xFF, 0xFF, nextFrame);

            for (int idxModel = 0, nModels = s_Holder.models.Size(); idxModel < nModels; ++idxModel)
            {
                s_Holder.models[idxModel]->DebugDraw(g3ddemo::PASS_TRANSLUCENT, VIEW_TV, shaderMode);
            }

            s_Meter.EndTimeSpan(idxTranslucent);
            s_Meter.EndTimeSpan(idxCPU);
        }

        // GPU 演算
        if (nw::g3d::CheckFlag(s_EnableFlag, ENABLE_TEXTURE_COMPUTE))
        {
            int idxCPU = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_CPU);
            int idxCompute = s_Meter.BeginTimeSpan(
                g3ddemo::ProcessMeter::GROUP_GPU, 0xFF, 0x00, 0x00, 0xFF, nextFrame);

            frameBufferWater.Load(); // TV のバッファを使うので他のバッファをロード
#if NW_G3D_IS_GX2
            GX2Invalidate(GX2_INVALIDATE_COLOR_BUFFER,
                colorBuffer.GetGX2ColorBuffer()->surface.imagePtr,
                colorBuffer.GetGX2ColorBuffer()->surface.imageSize);
            GX2Invalidate(GX2_INVALIDATE_TEXTURE,
                frameBufferTV.GetColorBufferTexture(GX2_RENDER_TARGET_0)->texture.GetBasePtr(),
                frameBufferTV.GetColorBufferTexture(GX2_RENDER_TARGET_0)->texture.GetBaseSize());
#endif
#if NW_G3D_COMPUTE_SHADER_ENABLE
            nw::g3d::SetShaderMode(GX2_SHADER_MODE_COMPUTE_SHADER);
#endif
            s_TexAverageShader.LoadTextureSampler(
                &frameBufferTV.GetColorBufferTexture(GX2_RENDER_TARGET_0)->texture);
            expBuf.Load();
            s_TexAverageShader.Load();
            pDispatchParams->Dispatch();

            s_Meter.EndTimeSpan(idxCompute);
            s_Meter.EndTimeSpan(idxCPU);
        }

        // 情報
        {
            int idxInfo = s_Meter.BeginTimeSpan(
                g3ddemo::ProcessMeter::GROUP_GPU, 0x80, 0x80, 0x80, 0xFF, nextFrame);

            frameBufferTV.Load();

            nw::g3d::SetShaderMode(GX2_SHADER_MODE_UNIFORM_BLOCK);
            g3ddemo::ScreenInfo::LoadState();
            s_Screen.Draw();
            if (!s_IsNormalMode)
            {
                s_Meter.Draw();
            }

            s_Meter.EndTimeSpan(idxInfo);
        }

        expBuf.GPUFlush();

        s_FrameDL.End();

        nw::g3d::GfxManage::DrawDone();

        // 画面のコピー
        {
            int idxCopy = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_SYS_GPU, 0xFF, 0xFF, 0xFF);
            g3ddemo::CopyOut(&colorBuffer, GX2_SCAN_TARGET_TV);
#ifndef _WIN32
            if (nw::g3d::CheckFlag(s_EnableFlag, ENABLE_DRC_RENDERING))
            {
                int idxCopyDRC = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_SYS_GPU, 0xFF, 0xFF, 0xFF);
                g3ddemo::CopyOut(&colorBuffer, GX2_SCAN_TARGET_DRC_FIRST);
                s_Meter.EndTimeSpan(idxCopyDRC);
            }
#endif
            pCtx->TempPrepare();
            s_Meter.EndTimeSpan(idxCopy);
        }

        nw::g3d::fnd::GfxManage::FlushCommands();

        // 画面のクリア
        {
            pCtx->Activate();
            int idxClear = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_SYS_GPU, 0, 0, 0);

            depthBufferShadow.GetGX2DepthBuffer()->viewFirstSlice = 0;
            depthBufferShadow.GetGX2DepthBuffer()->viewNumSlices = s_nCascade;
            depthBufferShadow.UpdateRegs();
            depthBufferShadow.Clear(GX2_CLEAR_DEPTH);
            nw::g3d::ClearBuffers(&colorBufferWater, &depthBufferWater,
                    0.3f, 0.3f, 0.3f, 1.0f, GX2_CLEAR_DEPTH);
            nw::g3d::ClearBuffers(&colorBufferNormal, &depthBufferGeometry,
                    0.3f, 0.3f, 0.3f, 1.0f, GX2_CLEAR_DEPTH);
            colorBufferLight.Clear(0.0f, 0.0f, 0.0f, 0.0f);
            nw::g3d::ClearBuffers(&colorBuffer, &depthBuffer,
                0.3f, 0.3f, 0.3f, 1.0f, GX2_CLEAR_DEPTH);

            pCtx->Activate();
            s_Meter.EndTimeSpan(idxClear);
        }

        nw::g3d::fnd::GfxManage::FlushCommands();

        // GPU 待ちが必要な計算
        {
            int idxCalcGPU = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_CPU);
            pCtx->TempPrepare(); // 更新に GL コンテクストが必要。

            for (int idxModel = 0, nModels = s_Holder.models.Size(); idxModel < nModels; ++idxModel)
            {
                s_Holder.models[idxModel]->CalcGPU();
            }
            for (int idxLight = 0, nLights = s_Lights.Size(); idxLight < nLights; ++idxLight)
            {
                s_Lights[idxLight]->CalcGPU();
            }

            // コンテキストの更新
            for (int idxSlice = 0; idxSlice < s_nCascade; ++idxSlice)
            {
                nw::g3d::Mtx44 proj;
                proj.Perspective(nw::g3d::Math::DegToRad(45.0f), 16.0f / 9.0f,
                    s_CascadeDepth[idxSlice], s_CascadeDepth[idxSlice+1]);
                s_View.GetViewMtx(VIEW_LIGHT + idxSlice).LookAt(s_LightPosition,
                    nw::g3d::Vec3::Make(0.0f, 1.0f, 0.0f), nw::g3d::Vec3::Make(0.0f, 0.0f, 0.0f));
                CalcLightProj(&s_View.GetProjMtx(VIEW_LIGHT + idxSlice),
                    &s_View.GetViewMtx(VIEW_LIGHT + idxSlice), &s_View.GetViewMtx(VIEW_TV), &proj);
                s_Context.GetContext().lightViewProj[idxSlice].Mul(
                    s_View.GetProjMtx(VIEW_LIGHT + idxSlice), s_View.GetViewMtx(VIEW_LIGHT + idxSlice));
            }
            s_Context.GetContext().cascadeDepth.Set(
                s_CascadeDepth[1], s_CascadeDepth[2], s_CascadeDepth[3], s_CascadeDepth[4]);
            s_Context.GetContext().nearFar.Set(10.0f, 40000.0f);
            float tanFovY = nw::g3d::Math::Tan(nw::g3d::Math::DegToRad(45.0f * 0.5f));
            s_Context.GetContext().tanFov.Set(tanFovY * ( 16.0f / 9.0f ), tanFovY);
            s_Context.GetContext().diffIntensity =
                nw::g3d::Math::Abs(( ( frame + 27000 ) % 36000 - 18000 ) / 18000.0f) * 2.0f;
            if (nw::g3d::CheckFlag(s_EnableFlag, ENABLE_TEXTURE_COMPUTE) && frame > 1)
            {
                float diff = texAverage.x + texAverage.y + texAverage.z - 0.65f;
                diffIntensity -= diff * diff * diff * 0.04f;
                if (diffIntensity < 0.3f) diffIntensity = 0.3f;
                if (diffIntensity > 1.7f) diffIntensity = 1.7f;
#if NW_G3D_COMPUTE_SHADER_ENABLE || NW_G3D_IS_GL
                s_Context.GetContext().diffIntensity = diffIntensity;
#endif
            }
            s_Context.Calc();

            // ビュー毎の更新。
            for (int idxView = 0; idxView < NUM_VIEW; ++idxView)
            {
                s_View.Calc(idxView);
            }
            s_View.GetViewVolume(VIEW_TV).SetPerspective(nw::g3d::Math::DegToRad(45.0f), 16.0f / 9.0f,
                10.0f, 40000.0f, s_View.GetInvViewMtx(VIEW_TV));
            s_View.GetViewVolume(VIEW_WATER).SetPerspective(nw::g3d::Math::DegToRad(45.0f), 16.0f / 9.0f,
                10.0f, 40000.0f, s_View.GetInvViewMtx(VIEW_WATER));

            for (int idxModel = 0, nModels = s_Holder.models.Size(); idxModel < nModels; ++idxModel)
            {
                for (int idxView = 0; idxView < NUM_VIEW; ++idxView)
                {
                    s_Holder.models[idxModel]->CalcViewGPU(idxView);
                }
                s_Holder.models[idxModel]->SetContext(&s_Context);
                s_Holder.models[idxModel]->SetView(&s_View);
            }
            for (int idxLight = 0, nLights = s_Lights.Size(); idxLight < nLights; ++idxLight)
            {
                s_Lights[idxLight]->CalcViewGPU(VIEW_TV);
                s_Lights[idxLight]->SetContext(&s_Context);
                s_Lights[idxLight]->SetView(&s_View);
            }

            // 環境
            if (s_Holder.pEnvModel)
            {
                s_Holder.pEnvModel->CalcGPU();
            }

            expBuf.CPUInvalidate();
            expBuf.CopyToCPU();

            // ディスパッチパラメータ
            pDispatchParams->SetParams(s_TileDimX, s_TileDimY);
            pDispatchParams->CPUFlush();

            s_Meter.EndTimeSpan(idxCalcGPU);
        }

        nw::g3d::CPUCache::Sync(); // CPU キャッシュ操作の完了を待ちます。

        nw::g3d::GfxManage::DrawDone();

        pCtx->TempPrepare();
        s_Meter.EndTimeSpan(idxFrame);
        s_Meter.Calc();

#if NW_G3D_CONFIG_USE_HOSTIO
        if (s_AssignChanged)
        {
            s_AssignChanged = false;
        }
        else
#endif
        {
            g3ddemo::SwapScanBuffers();
        }

#if NW_G3D_CONFIG_USE_HOSTIO
        // 描画中のリソースを破棄してしまわないよう、描画完了後に呼び出します。
        // UnloadFile 内で描画リソースを破棄しないような遅延処理による対策も可。
        pCtx->TempPrepare();
        nw::g3d::edit::EditServer::Instance().Poll();
#endif
    }

    // グラフィックスリソースの終了処理。
    pCtx->TempPrepare();

    s_FrameDL.Cleanup();

#if NW_G3D_CONFIG_USE_HOSTIO
    {
        nw::g3d::edit::EditServer::Instance().Clear(true);
        nw::g3d::edit::EditServer::Instance().Close();
        nw::g3d::edit::EditServer::DeleteInstance();
    }
#endif

    if (void* ptr = ringBuffer.GetInputBuffer())
    {
        g3ddemo::FreeMem1(ptr);
    }
    if (void* ptr = ringBuffer.GetOutputBuffer())
    {
        g3ddemo::FreeMem1(ptr);
    }

    g3ddemo::FreeMem1(s_DepthTexture.GetBasePtr());
    s_DepthTexture.Cleanup();

    g3ddemo::FreeMem1(s_LightTexture.GetBasePtr());
    s_LightTexture.Cleanup();

    g3ddemo::FreeMem1(s_NormalTexture.GetBasePtr());
    s_NormalTexture.Cleanup();

    g3ddemo::FreeMem1(s_ColorTexture.GetBasePtr());
    s_ColorTexture.Cleanup();

    g3ddemo::FreeMem1(s_ShadowTexture.GetBasePtr());
    s_ShadowTexture.Cleanup();

    g3ddemo::FreeMem1(s_WaterTexture.GetBasePtr());
    s_WaterTexture.Cleanup();

    for (int idxAssign = 0, numAssign = s_Holder.assigns.Size(); idxAssign < numAssign; ++idxAssign)
    {
        if (nw::g3d::ResShaderArchive* pShader = s_Holder.assigns[idxAssign]->GetModelShader())
        {
            for (int idxShadingModel = 0, numShadingModel = pShader->GetShadingModelCount();
                idxShadingModel < numShadingModel; ++idxShadingModel)
            {
                g3ddemo::DestroyUserShadingModel(pShader->GetShadingModel(idxShadingModel));
            }
        }
    }

    // エディットライブラリが所有しているモデルを破棄しないように、
    // エディットの終了処理後に残りのデータの破棄を行います。
    g3ddemo::DestroyAll(&s_Holder);

    for (int idxLight = 0, nLights = s_Lights.Size(); idxLight < nLights; ++idxLight)
    {
        nw::g3d::ModelObj* pModelObj = s_Lights[idxLight]->GetModelObj();
        s_Lights[idxLight]->DestroyAnimObj();
        g3ddemo::DestroyUserModel(pModelObj);
        g3ddemo::DestroyModelObj(pModelObj);
    }
    if (void *pBuffer = s_Lights.GetBuffer())
    {
        g3ddemo::FreeMem2(pBuffer);
    }

    s_Context.Cleanup();
    s_View.Cleanup();

    s_Screen.Cleanup();
    s_Meter.Cleanup();

    g3ddemo::FreeMem2(expGPUBuf);
    expBuf.Cleanup();
    g3ddemo::FreeMem2(pDispatchParams);

    s_TexAverageShader.Cleanup();

    frameBufferLight.Cleanup();
    g3ddemo::FreeMem2(frameBufferLight.GetBufferPtr());

    frameBufferGeometry.Cleanup();
    g3ddemo::FreeMem2(frameBufferGeometry.GetBufferPtr());

    frameBufferShadow.Cleanup();
    g3ddemo::FreeMem2(frameBufferShadow.GetBufferPtr());

    frameBufferWater.Free(g3ddemo::FreeMem1);
    frameBufferWater.Cleanup();
    g3ddemo::FreeMem2(frameBufferWater.GetBufferPtr());

    frameBufferTV.Free(g3ddemo::FreeMem1);
    frameBufferTV.Cleanup();
    g3ddemo::FreeMem2(frameBufferTV.GetBufferPtr());

    pCtx->Cleanup();
    g3ddemo::FreeMem2(pCtx);
    pCtx = NULL;

    // 終了処理。
    g3ddemo::ShutdownDisplay();
    g3ddemo::Shutdown();

    return restart ? -1 : EXIT_SUCCESS;
}

namespace {

g3ddemo::ModelAssign* SetupModelAssign(nw::g3d::ResModel* pResModel,
    nw::g3d::ResShaderArchive** pShaderArchiveArray, int ShaderArchiveCount,
    nw::g3d::ResShaderArchive* pModelShader)
{
    g3ddemo::CreateShaderAssign(pResModel);
    g3ddemo::ModelAssign* pModelAssign = pResModel->GetUserPtr<g3ddemo::ModelAssign>();
    pModelAssign->SetModelShader(pModelShader);

    nw::g3d::BindResult result = nw::g3d::BindResult::NotBound();
    // 3DEditor 経由
    if (pShaderArchiveArray)
    {
        result = pModelAssign->BindShader(g3ddemo::PASS_DEFAULT, pShaderArchiveArray, ShaderArchiveCount);
        pModelAssign->BindShader(g3ddemo::PASS_OPAQUE, pShaderArchiveArray, ShaderArchiveCount);
        pModelAssign->BindShader(g3ddemo::PASS_TRANSLUCENT, pShaderArchiveArray, ShaderArchiveCount);
        pModelAssign->BindShader(g3ddemo::PASS_WATER, pShaderArchiveArray, ShaderArchiveCount);
    }

    // モデルシェーダ
    if (!result.IsComplete() && pModelShader)
    {
        result = pModelAssign->BindShader(g3ddemo::PASS_DEFAULT, pModelShader);
        pModelAssign->BindShader(g3ddemo::PASS_OPAQUE, pModelShader);
        pModelAssign->BindShader(g3ddemo::PASS_TRANSLUCENT, pModelShader);
        pModelAssign->BindShader(g3ddemo::PASS_WATER, pModelShader);
    }

    // グローバルシェーダ
    for (int idxShader = 0, numShader = s_Holder.shaders.Size();
        !result.IsComplete() && idxShader < numShader; ++idxShader)
    {
        nw::g3d::ResShaderArchive* pShaderArchive = s_Holder.shaders[idxShader];
        result = pModelAssign->BindShader(g3ddemo::PASS_DEFAULT, pShaderArchive);
        pModelAssign->BindShader(g3ddemo::PASS_OPAQUE, pShaderArchive);
        pModelAssign->BindShader(g3ddemo::PASS_TRANSLUCENT, pShaderArchive);
        pModelAssign->BindShader(g3ddemo::PASS_WATER, pShaderArchive);
    }
    //NW_G3D_ASSERT(result.IsComplete());

    for (int idxShader = 0, numShader = s_Holder.shaders.Size();
        idxShader < numShader; ++idxShader)
    {
        if (strcmp(s_Holder.shaders[idxShader]->GetName(), "shadow") == 0)
        {
            nw::g3d::ResShadingModel* pShadingModel =
                s_Holder.shaders[idxShader]->GetShadingModel("shadow");
            pModelAssign->BindShader(g3ddemo::PASS_SHADOW, pShadingModel);
        }

        if (strcmp(s_Holder.shaders[idxShader]->GetName(), "gbuffer") == 0)
        {
            nw::g3d::ResShadingModel* pShadingModel =
                s_Holder.shaders[idxShader]->GetShadingModel("gbuffer");
            pModelAssign->BindShader(g3ddemo::PASS_GEOMETRY, pShadingModel);
        }
    }

    pModelAssign->Setup();
    return pModelAssign;
}

g3ddemo::UserModel* SetupUserModel(nw::g3d::ModelObj* pModelObj)
{
    g3ddemo::CreateUserModel(pModelObj);
    g3ddemo::UserModel* pUserModel = pModelObj->GetUserPtr<g3ddemo::UserModel>();
    pUserModel->SetView(&s_View);
    if (s_Holder.pEnvModel)
    {
        pUserModel->SetEnv(s_Holder.pEnvModel->GetModelObj());
    }

    g3ddemo::SetupMaterialsArg argSetup(pModelObj);
    g3ddemo::SetupMaterials(argSetup);

    // アニメーションインスタンスの構築。
    g3ddemo::UserModel::CreateAnimObjArg createAnimObjArg;
    if (!s_Holder.modelAnims.Empty())
    {
        createAnimObjArg.Reserve(&s_Holder.modelAnims[0], s_Holder.modelAnims.Size());
    }
    pUserModel->CreateAnimObj(createAnimObjArg);

    return pUserModel;
}

void SetupUserShadingModel(nw::g3d::ResShaderArchive* pShader)
{
    if (pShader)
    {
        for (int idxShadingModel = 0, numShadingModel = pShader->GetShadingModelCount();
            idxShadingModel < numShadingModel; ++idxShadingModel)
        {
            nw::g3d::ResShadingModel* pShadingModel = pShader->GetShadingModel(idxShadingModel);
            if (pShadingModel->GetUserPtr() == NULL )
            {
                g3ddemo::CreateUserShadingModel(pShadingModel, s_Holder.pEnvModel->GetModelObj());
            }
        }
    }
}

void SetupWaterVariation(nw::g3d::ModelObj* pModelObj)
{
    for (int idxShape = 0, nShapes = pModelObj->GetShapeCount(); idxShape < nShapes; ++idxShape)
    {
        g3ddemo::UserShape* pUserShape =
            pModelObj->GetShape(idxShape)->GetUserPtr<g3ddemo::UserShape>();
        nw::g3d::ShaderSelector* pShaderSelector =
            pUserShape->GetShaderSelector(g3ddemo::PASS_WATER);
        int idxOption = pShaderSelector->GetDynamicOptionIndex("water_world");
        if (idxOption > 0)
        {
            const nw::g3d::ResShaderOption* pShaderOption =
                pShaderSelector->GetDynamicOption(idxOption);
            int choice = pShaderOption->GetChoiceIndex("1");
            pShaderSelector->WriteDynamicKey(idxOption, choice);
        }
    }
}

void AddAnim(const char* pModelName, const char* pAnimName, float step)
{
    int idxModel;
    int nModels = s_Holder.models.Size();
    for (idxModel = 0; idxModel < nModels; ++idxModel)
    {
        if (strcmp(pModelName,
            s_Holder.models[idxModel]->GetModelObj()->GetResource()->GetName()) == 0)
        {
            break;
        }
    }

    if (idxModel < nModels)
    {
        for (int idxAnim = 0, nAnims = s_Holder.modelAnims.Size(); idxAnim < nAnims; ++idxAnim)
        {
            const char* pName = NULL;
            void* pAnim = s_Holder.modelAnims[idxAnim];
            switch (static_cast<nw::g3d::BinaryBlockHeader*>(pAnim)->sigWord)
            {
            case nw::g3d::ResSkeletalAnim::SIGNATURE:
                pName = static_cast<nw::g3d::ResSkeletalAnim*>(pAnim)->GetName();
                break;
            case nw::g3d::ResVisibilityAnim::SIGNATURE:
                pName = static_cast<nw::g3d::ResVisibilityAnim*>(pAnim)->GetName();
                break;
            case nw::g3d::ResShaderParamAnim::SIGNATURE:
                pName = static_cast<nw::g3d::ResShaderParamAnim*>(pAnim)->GetName();
                break;
            case nw::g3d::ResTexPatternAnim::SIGNATURE:
                pName = static_cast<nw::g3d::ResTexPatternAnim*>(pAnim)->GetName();
                break;
            case nw::g3d::ResShapeAnim::SIGNATURE:
                pName = static_cast<nw::g3d::ResShapeAnim*>(pAnim)->GetName();
                break;
            }
            if (pName && strcmp(pName, pAnimName) == 0)
            {
                s_Holder.models[idxModel]->AddAnim(pAnim);
                ChangeAnimStep(s_Holder.models[idxModel], step - 1.0f);
            }
        }
    }
}

void ChangeAnimStep(g3ddemo::UserModel* pUserModel, float step)
{
    nw::g3d::ModelAnimObj* pAnimObj;
    pAnimObj = pUserModel->GetAnimObj(g3ddemo::UserModel::ANIM_TYPE_SKELETAL);
    pAnimObj->GetFrameCtrl().SetStep(pAnimObj->GetFrameCtrl().GetStep() + step);
    pAnimObj = pUserModel->GetAnimObj(g3ddemo::UserModel::ANIM_TYPE_SHADERPARAM);
    pAnimObj->GetFrameCtrl().SetStep(pAnimObj->GetFrameCtrl().GetStep() + step);
}

void UpdateCamera(const g3ddemo::Pad& pad)
{
    nw::g3d::math::Vec3& pos = s_CameraPosition;
    nw::g3d::math::Vec3& target = s_CameraTarget;
    nw::g3d::math::Vec3& up = s_CameraUp;
    const nw::g3d::math::Mtx34& viewMtx = s_View.GetViewMtx(VIEW_TV);

    const f32 RotSpeed     = 0.02f;
    const f32 AtMoveSpeed  = 20.0f;

    nw::g3d::Vec3 right, camUp, look;
    right.Set(viewMtx.m00, viewMtx.m01, viewMtx.m02);
    camUp.Set(viewMtx.m10, viewMtx.m11, viewMtx.m12);
    look.Set(viewMtx.m20, viewMtx.m21, viewMtx.m22);

    nw::g3d::Vec3 diff;
    diff.Sub(target, pos);

    float dist = diff.Normalize(diff);

    nw::g3d::Vec3 dirX, dirY, dirZ;

    dirZ.Cross(right, camUp);
    dirZ.Normalize(dirZ);

    dirY.Cross(dirZ, right);
    dirY.Normalize(dirY);

    const g3ddemo::Pad::AnalogStick& analogStick = pad.GetAnalogStick();

    pos.Add(dirX.Mul(right, AtMoveSpeed * ( analogStick.rightX) ), pos);
    pos.Add(dirZ.Mul(dirZ, AtMoveSpeed * ( -analogStick.rightY)), pos);

    nw::g3d::math::Quat qt, qtX, qtY;
    {
        float ang = analogStick.leftY * RotSpeed * 0.5f;
        float c = nw::g3d::math::Math::Cos(ang);
        float s = nw::g3d::math::Math::Sin(ang);
        qtX.Set(s * right.x, s * right.y, s *right.z, c);
    }
    {
        float ang = analogStick.leftX * -RotSpeed * 0.5f;
        float c = nw::g3d::math::Math::Cos(ang);
        float s = nw::g3d::math::Math::Sin(ang);
        qtY.Set(0.0f, s, 0.0f, c);
    }

    qt.Mul(qtY, qtX);

    nw::g3d::Mtx34 qtMtx;
    qtMtx.SetR(qt);

    diff.Rotate(qtMtx, diff);
    diff.Normalize(diff);

    target.Mul(diff, dist);
    target.Add(target, pos);

    up.Cross(right, diff);
    up.Set(0.0f, (up.y >= 0) ? 1.0f : 0.0f, 0.0f);

    s_View.GetViewMtx(VIEW_TV).LookAt(s_CameraPosition, s_CameraUp, s_CameraTarget);
}

void CalcLightProj(nw::g3d::Mtx44* pOutLightProj, const nw::g3d::Mtx34* pLightView,
                   const nw::g3d::Mtx34* pCameraView, const nw::g3d::Mtx44* pCameraProj)
{
    float det;
    nw::g3d::Mtx44 cameraViewProj;
    nw::g3d::Mtx44 invCameraViewProj;
    nw::g3d::Mtx44 screenToLightView;
    cameraViewProj.Mul(*pCameraProj, *pCameraView);
    invCameraViewProj.Inverse(&det, cameraViewProj);
    screenToLightView.Mul(*pLightView, invCameraViewProj);

    nw::g3d::AABB aabb;
    nw::g3d::AABB aabbObj;
    for (int dim = 0; dim < nw::g3d::Vec3::DIM; ++dim)
    {
        aabb.min.a[dim] = aabbObj.min.a[dim] = FLT_MAX;
        aabb.max.a[dim] = aabbObj.max.a[dim] = -FLT_MAX;
    }

    nw::g3d::Vec3 posLightView;
    for (int x = 0; x < 2; ++x)
    {
        for (int y = 0; y < 2; ++y )
        {
            for (int z = 0; z < 2; ++z)
            {
                posLightView.Project(screenToLightView, nw::g3d::Vec3::Make(
                    x == 0 ? -1.0f : 1.0f, y == 0 ? -1.0f : 1.0f, z == 0 ? -1.0f : 1.0f));
                for (int axis = 0; axis < nw::g3d::Vec3::DIM; ++axis)
                {
                    aabb.min.a[axis] = (std::min)(aabb.min.a[axis], posLightView.a[axis]);
                    aabb.max.a[axis] = (std::max)(aabb.max.a[axis], posLightView.a[axis]);
                }
            }
        }
    }

    nw::g3d::ViewVolume viewVolume;
    nw::g3d::Mtx34 invLightView;
    invLightView.Inverse(&det, *pLightView);
    viewVolume.SetOrtho(aabb.max.y, aabb.min.y, aabb.min.x, aabb.max.x, 0.0f, -aabb.min.z, invLightView);
    for (int idxModel = 0, numModel = s_Holder.models.Size(); idxModel < numModel; ++idxModel)
    {
        nw::g3d::ModelObj* pModelObj = s_Holder.models[idxModel]->GetModelObj();
        for (int idxShape = 0, numShape = pModelObj->GetShapeCount(); idxShape < numShape; ++idxShape)
        {
            const nw::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(idxShape);
            const nw::g3d::MaterialObj* pMatObj = pModelObj->GetMaterial(pShapeObj->GetMaterialIndex());
            const nw::g3d::Sphere* pSphere = pShapeObj->GetBounding();

            if (pMatObj->GetUserPtr<g3ddemo::UserMaterial>()->GetRenderShadow() &&
                pSphere && viewVolume.TestIntersection(*pSphere))
            {
                posLightView.Mul(*pLightView, pSphere->center);
                for (int axis = 0; axis < nw::g3d::Vec3::DIM; ++axis)
                {
                    aabbObj.min.a[axis] = (std::min)(aabbObj.min.a[axis], posLightView.a[axis] - pSphere->radius);
                    aabbObj.max.a[axis] = (std::max)(aabbObj.max.a[axis], posLightView.a[axis] + pSphere->radius);
                }
            }
        }
    }

    pOutLightProj->Ortho(aabbObj.min.x, aabbObj.max.x, aabbObj.min.y,
        aabbObj.max.y, -aabbObj.max.z, -aabb.min.z);
}

#if NW_G3D_CONFIG_USE_HOSTIO
void AttachShader(const char* pShaderName)
{
    nw::g3d::ResShaderArchive* pShader = NULL;
    for (int idxShader = 0, numShader = s_Holder.shaders.Size();
        idxShader < numShader; ++idxShader)
    {
        if (strcmp(s_Holder.shaders[idxShader]->GetName(), pShaderName) == 0)
        {
            pShader = s_Holder.shaders[idxShader];
            break;
        }
    }
    if (pShader && !nw::g3d::edit::EditServer::Instance().HasShaderArchive(pShader))
    {
        nw::g3d::edit::EditServer::AttachShaderArchiveArg arg;
        arg.resShaderArchive = pShader;
        nw::g3d::edit::EditServer::Instance().AttachShaderArchive(arg);
    }
}
#endif

} // anonymous namespace
