﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d.h>
#include <nw/g3d/g3d_edit.h>
#include <nw/g3d/edit/g3d_InputCapture.h>

#include <g3ddemo_MaterialPick.h>
#include <g3ddemo_DemoUtility.h>
#include <g3ddemo_GfxUtility.h>
#include <g3ddemo_UserModel.h>
#include <g3ddemo_ModelUtility.h>

namespace g3ddemo = nw::g3d::demo;

namespace {

#if NW_G3D_CONFIG_USE_HOSTIO
#ifndef _WIN32
#define USE_INPUTCAPTURE
#endif
#endif

enum
{
    // モデルに使用するテクスチャのアライメントの最大値を設定します。
    // リソースファイルごとのアライメントを事前に取得しておくことで、
    // 必要なメモリのアライメントを最小限に抑えてメモリを節約することも可能です。
    TEXTURE_ALIGNMENT   = 8192
};

#define SHADER_PATH "shader/"
#define MODEL_PATH  ""

const char* BFSHA_PATH[] = {
    SHADER_PATH "demo.bfsha",
    SHADER_PATH "material_pick.bfsha",
    NULL
};

const char* BFRES_PATH[] = {
    MODEL_PATH "env.bfres",
    NULL
};

nw::g3d::Vec3 s_CameraPosition;
nw::g3d::Vec3 s_CameraUp;
nw::g3d::Vec3 s_CameraTarget;

const int NUM_VIEW = 1;
const f32 CURSOR_DATA[3][2] = { { 0.014f, 0.0122f }, { 0.f, 0.f }, { 0.00625f, 0.025f } };

g3ddemo::ResourceHolder s_Holder;
g3ddemo::UserView s_View;

g3ddemo::MaterialPicker s_MaterialPicker;
nw::g3d::math::Vec2 s_MouseTriggeredPos;
bool s_MouseTriggered = false;

g3ddemo::ModelAssign* SetupModelAssign(nw::g3d::ResModel* pResModel,
    nw::g3d::ResShaderArchive** pShaderArchiveArray = NULL, int shaderArchiveCount = 0);
g3ddemo::UserModel* SetupUserModel(nw::g3d::ModelObj* pModelObj);

#if NW_G3D_CONFIG_USE_HOSTIO

g3ddemo::EditAllocator s_Allocator;

class Editor : public nw::g3d::edit::EditCallback
{
public:
    virtual nw::g3d::ResFile* LoadFile(const nw::g3d::edit::LoadFileArg& arg)
    {
        // 3DEditor が開いたモデルがリソースとして送られてきます。

        // リソースファイルはアプリケーションが管理する必要あるため、コピーします。
        void* pFile = g3ddemo::AllocMem2(arg.fileSize, arg.align);
        memcpy(pFile, arg.resFile, arg.fileSize);
        // CPU がコピーしたので、GPU が参照できるようキャッシュをフラッシュします。
        nw::g3d::CPUCache::Flush(pFile, arg.fileSize);

        NW_G3D_ASSERT(nw::g3d::ResFile::IsValid(pFile));
        nw::g3d::ResFile* pResFile = nw::g3d::ResFile::ResCast(pFile);
        pResFile->Setup();

        s_Holder.files.PushBack(pResFile);

        for (int idxModel = 0, numModel = pResFile->GetModelCount(); idxModel < numModel; ++idxModel)
        {
            nw::g3d::ResModel* pResModel = pResFile->GetModel(idxModel);

            // シェーダ割り当ての構築。
            g3ddemo::ModelAssign* pModelAssign = SetupModelAssign(pResModel);
            s_Holder.assigns.PushBack(pModelAssign);

            // モデルインスタンスの構築。
            g3ddemo::CreateModelObjArg createModelObjArg(pResModel);
            createModelObjArg.initArg.ViewCount(NUM_VIEW);
            nw::g3d::ModelObj* pModelObj = g3ddemo::CreateModelObj(createModelObjArg);
            g3ddemo::UserModel* pUserModel = SetupUserModel(pModelObj);
            pUserModel->EnableEditCalcSkeletalAnimations();
            s_Holder.models.PushBack(pUserModel);

            // エディットへの登録。
            nw::g3d::edit::EditServer::AttachModelArg attachArg;
            attachArg.modelObj = pModelObj;
            nw::g3d::edit::EditServer::Instance().AttachModel(attachArg);

            // マテリアルピッカーへの登録。
            s_MaterialPicker.Entry(pUserModel);
        }

        return pResFile;
    }

    virtual void UnloadFile(const nw::g3d::edit::UnloadFileArg& arg)
    {
        // 3DEditor でファイルが閉じられたので、関連するデータを破棄します。

        nw::g3d::ResFile* pResFile = arg.resFile;

        for (int idxResMdl = 0, numResMdl = pResFile->GetModelCount();
            idxResMdl < numResMdl; ++idxResMdl)
        {
            nw::g3d::ResModel* pResModel = pResFile->GetModel(idxResMdl);

            // モデルインスタンスの破棄。
            for (int idxModel = 0, numModel = s_Holder.models.Size();
                 idxModel < numModel; ++idxModel)
            {
                g3ddemo::UserModel* pUserModel = s_Holder.models[idxModel];
                nw::g3d::ModelObj* pModelObj = pUserModel->GetModelObj();
                if (pModelObj->GetResource() == pResModel)
                {
                    pUserModel->DestroyAnimObj();
                    g3ddemo::DestroyUserModel(pModelObj);
                    g3ddemo::DestroyModelObj(pModelObj);
                    s_Holder.models.Erase(idxModel);

                    // マテリアルピッカーから削除。
                     s_MaterialPicker.Erase(pUserModel);

                    break;
                }
            }

            // シェーダ割り当ての破棄。
            for (int idxAssign = 0, numAssign = s_Holder.assigns.Size();
                 idxAssign < numAssign; ++idxAssign)
            {
                g3ddemo::ModelAssign* pModelAssign = s_Holder.assigns[idxAssign];
                if (pModelAssign->GetResModel() == pResModel)
                {
                    pModelAssign->Cleanup();
                    g3ddemo::DestroyShaderAssign(pResModel);
                    s_Holder.assigns.Erase(idxAssign);
                    break;
                }
            }

            // リソースファイルの破棄。
            for (int idxFile = 0, numFile = s_Holder.files.Size();
                 idxFile < numFile; ++idxFile)
            {
                if (s_Holder.files[idxFile] == pResFile)
                {
                    pResFile->Cleanup();
                    g3ddemo::FreeMem2(pResFile);
                    s_Holder.files.Erase(idxFile);
                    break;
                }
            }
        }
    }

    virtual void UpdateMaterials(const nw::g3d::edit::UpdateMaterialsArg& arg)
    {
        // UpdateShaderAssign で行われた変更に伴って ModelObj に変更が必要な場合は
        // このコールバックで処理を行います。
        // 編集のために使用するヒープを区別したい場合は arg.state で状態を判定します。

        nw::g3d::ModelObj* pModelObj = arg.modelObj;

        // シェーダが変更された場合のためにシェーダインスタンスを再構築します。
        for (int idxShape = 0, numShape = pModelObj->GetShapeCount();
            idxShape < numShape; ++idxShape)
        {
            nw::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(idxShape);
            g3ddemo::UserShape* pUserShape = pShapeObj->GetUserPtr<g3ddemo::UserShape>();
            pUserShape->Cleanup();
            pUserShape->Setup();
        }

        // マテリアルを更新します。
        g3ddemo::SetupMaterialsArg argSetup(pModelObj);
        g3ddemo::SetupMaterials(argSetup);

        // マテリアルピッカーの情報を更新します。
        s_MaterialPicker.UpdateBlock();
    }

    virtual void UpdateShaderAssign(const nw::g3d::edit::UpdateShaderAssignArg& arg)
    {
        // モデルリソースの構造変更を伴う編集が 3DEditor で行われたため、
        // 古いモデルリソースの破棄と新しいモデルリソースの構築を行います。
        // モデルインスタンスが新しいリソースを参照するように、
        // エディットライブラリが参照関係を変更します。
        switch(arg.state)
        {
        case nw::g3d::edit::UpdateShaderAssignArg::STATE_MODEL_UPDATE:
        case nw::g3d::edit::UpdateShaderAssignArg::STATE_MODEL_END:
            {
                // モデルリソースにユーザ領域を関連付けて使用していた場合は破棄します。
                g3ddemo::ModelAssign* pModelAssign = arg.oldResModel->GetUserPtr<g3ddemo::ModelAssign>();
                NW_G3D_ASSERT_NOT_NULL(pModelAssign);
                pModelAssign->Cleanup();
                g3ddemo::DestroyShaderAssign(arg.oldResModel);
            }
            break;
        default:
            break;
        }
        switch(arg.state)
        {
        case nw::g3d::edit::UpdateShaderAssignArg::STATE_MODEL_BEGIN:
        case nw::g3d::edit::UpdateShaderAssignArg::STATE_MODEL_UPDATE:
            {
                // モデルリソースにユーザ領域を関連付けて管理します。
                // アプリケーションで使用している構築処理をそのまま使用することができます。
                SetupModelAssign(arg.newResModel, arg.shaderArchivePtrs, arg.numShaderArchive);
            }
            break;
        default:
            break;
        }
    }

    virtual void UpdateBoneBind(const nw::g3d::edit::UpdateBoneBindArg& arg)
    {
        // 3DEditor でプレビュー配置のバインド設定が行われた際にモデルに通知を行います。
        // 配置自体はここで行わずにアプリケーションのバインド機能によってバインドします。
        g3ddemo::UserModel* pUserModel = arg.modelObj->GetUserPtr<g3ddemo::UserModel>();
        pUserModel->BindTo(arg.parentModelObj, arg.parentBoneIndex);
    }

    virtual void UpdateModelLayout(const nw::g3d::edit::UpdateModelLayoutArg& arg)
    {
        g3ddemo::UserModel* pUserModel = arg.modelObj->GetUserPtr<g3ddemo::UserModel>();
        pUserModel->GetScale().Set(*nw::g3d::Vec3::Cast(arg.scale.a));
        pUserModel->GetLayoutMtx().SetR(*nw::g3d::Vec3::Cast(arg.rotate.a));
        pUserModel->GetLayoutMtx().SetT(*nw::g3d::Vec3::Cast(arg.translate.a));
    }

    virtual bool SendModelLayout(nw::g3d::edit::SendModelLayoutData* outputData,
        const nw::g3d::edit::SendModelLayoutArg& arg)
    {
        const g3ddemo::UserModel* pUserModel = arg.modelObj->GetUserPtr<g3ddemo::UserModel>();
        *nw::g3d::Vec3::Cast(outputData->scale.a) = pUserModel->GetScale();
        const nw::g3d::Mtx34& layoutMtx = pUserModel->GetLayoutMtx();
        float sy = -layoutMtx.m20;
        // Y 軸周りの回転は ±90°の範囲として計算します。
        outputData->rotate.y = nw::g3d::Math::Asin(sy);
        if (nw::g3d::Math::Abs(sy) > 0.9999f)
        {
            // Y 軸周りの回転が ±90°の場合は角度が一意に決まらないため、
            // X 軸周りの回転は 0°として扱います。
            outputData->rotate.x = 0.0f;
            // Z 軸周りの回転は ±180°の範囲として計算します。
            outputData->rotate.z = nw::g3d::Math::Atan2(layoutMtx.m12, layoutMtx.m02);
        }
        else
        {
            // X 軸および Z 軸周りの回転は ±180°の範囲として計算します。
            outputData->rotate.x = nw::g3d::Math::Atan2(layoutMtx.m21, layoutMtx.m22);
            outputData->rotate.z = nw::g3d::Math::Atan2(layoutMtx.m10, layoutMtx.m00);
        }
        outputData->translate.x = layoutMtx.m03;
        outputData->translate.y = layoutMtx.m13;
        outputData->translate.z = layoutMtx.m23;
        return true;
    }
} s_Editor;

#endif

} // anonymous namespace

int MaterialPickMain(int argc, const char* argv[])
{
    (void)argc;
    (void)argv;

    // 初期化処理。
    g3ddemo::InitDisplayArg initDisplayArg;
    initDisplayArg.modeDRC = GX2_DRC_NONE;
    g3ddemo::Init();
    g3ddemo::InitDisplay(initDisplayArg);

    g3ddemo::Pad pad;
    if (!pad.Reset())
    {
        g3ddemo::PostQuitMsg();
    }

    // グラフィックスリソースの初期化処理。
    nw::g3d::GfxContext::Prepare(); // 構築に GL コンテクストが必要。
    nw::g3d::GfxContext* pCtx = g3ddemo::AllocMem2<nw::g3d::GfxContext>(
        sizeof(*pCtx), GX2_CONTEXT_STATE_ALIGNMENT);
    pCtx->Setup();
    pCtx->TempPrepare();

    g3ddemo::FrameBuffer frameBufferTV;
    {
        g3ddemo::FrameBuffer::InitArg initArg(1280, 720);
        initArg.colorBufferFTV = true;
        size_t bufferSize = g3ddemo::FrameBuffer::CalcSize(initArg);
        frameBufferTV.Init(initArg, g3ddemo::AllocMem2(bufferSize), bufferSize);
        frameBufferTV.Setup();
        frameBufferTV.Alloc(g3ddemo::AllocMem1);
    }

    nw::g3d::GfxColorBuffer& colorBufferTV =
        frameBufferTV.GetColorBufferTexture(GX2_RENDER_TARGET_0)->renderBuffer;
    nw::g3d::GfxDepthBuffer& depthBufferTV = frameBufferTV.GetDepthBufferTexture()->renderBuffer;

    g3ddemo::ScreenInfo debugPrint;
    debugPrint.Setup(1024);
    debugPrint.SetShapeColor(0x7F, 0x7F, 0x7F, 0x7F);
    debugPrint.SetFontColor(0xFF, 0xFF, 0xFF);
    debugPrint.SetFontSize(2.0f);
    g3ddemo::ScreenInfo cursor;
    cursor.Setup(1024);
    cursor.SetShapeColor(0xFF, 0x0, 0x0, 0xFF);
    g3ddemo::ScreenInfo selectedRect;
    selectedRect.Setup(1024);
    selectedRect.SetShapeColor(0x8F, 0x8F, 0x5F, 0x7F);

#if NW_G3D_CONFIG_USE_HOSTIO
    // edit library
    {
        nw::g3d::edit::EditServer::CreateArg arg;
        arg.allocator = &s_Allocator;
        arg.editCallback = &s_Editor;

        bool createResult = nw::g3d::edit::EditServer::CreateInstance( arg );
        NW_G3D_ASSERT(createResult);

        bool openConnectionResult = nw::g3d::edit::EditServer::Instance().Open();
        NW_G3D_ASSERT(openConnectionResult);
    }
#endif

#ifdef USE_INPUTCAPTURE
    // InputCapture を開始します。
    nw::g3d::edit::InputCapture inputCapture;
    inputCapture.Open();
#endif

    // ビューの Uniform Block
    s_View.Setup();

    s_Holder.Init();

    // シェーダの初期化。
    for (const char** path = BFSHA_PATH; *path != NULL; ++path)
    {
        void* pFile = g3ddemo::LoadFile(path[0], NULL, GX2_SHADER_ALIGNMENT);
        NW_G3D_ASSERT(nw::g3d::ResShaderArchive::IsValid(pFile));
        nw::g3d::ResShaderArchive* pShaderArchive = nw::g3d::ResShaderArchive::ResCast(pFile);
        pShaderArchive->Setup();

        s_Holder.shaders.PushBack(pShaderArchive);
    }

    // マテリアルピッカーの初期化。
    {
        // マテリアルピック用シェーディングモデルを取得。
        nw::g3d::ResShadingModel* pShadingModel = NULL;
        for (int idxShader = 0, numShader = s_Holder.shaders.Size();
             idxShader < numShader; ++idxShader)
        {
            nw::g3d::ResShaderArchive* pShaderArchive = s_Holder.shaders[idxShader];
            pShadingModel = pShaderArchive->GetShadingModel("material_pick");
            if (pShadingModel)
            {
                break;
            }
        }

        g3ddemo::MaterialPicker::SetupArg arg(pShadingModel);
        s_MaterialPicker.Setup(arg);
    }

    // リソースファイルの初期化。
    for (const char** path = BFRES_PATH; *path != NULL; ++path)
    {
        // ResFile のアライメントは内部に持っているテクスチャの最大アライメントに一致します。
        size_t size = 0;
        void* pFile = g3ddemo::LoadFile(path[0], &size, TEXTURE_ALIGNMENT);
        NW_G3D_ASSERT(nw::g3d::ResFile::IsValid(pFile));
        nw::g3d::ResFile* pResFile = nw::g3d::ResFile::ResCast(pFile);
        pResFile->Setup();
        // ResFile 内のテクスチャや頂点を CPU で書き込んだ場合は
        // CPU のキャッシュを吐き出す必要があります。
        s_Holder.files.PushBack(pResFile);
    }

    // モデルリソースとシェーダの関連付け。
    for (int idxFile = 0, numFile = s_Holder.files.Size(); idxFile < numFile; ++idxFile)
    {
        nw::g3d::ResFile* pResFile = s_Holder.files[idxFile];
        for (int idxModel = 0, numModel = pResFile->GetModelCount();
            idxModel < numModel; ++idxModel)
        {
            nw::g3d::ResModel* pResModel = pResFile->GetModel(idxModel);

            // シェーダ割り当ての構築。
            g3ddemo::ModelAssign* pModelAssign = SetupModelAssign(pResModel);
            s_Holder.assigns.PushBack(pModelAssign);

            // モデルインスタンスの構築。
            g3ddemo::CreateModelObjArg arg(pResModel);
            arg.initArg.ViewCount(NUM_VIEW);
            nw::g3d::ModelObj* pModelObj = g3ddemo::CreateModelObj(arg);
            g3ddemo::UserModel* pUserModel = SetupUserModel(pModelObj);
#if NW_G3D_CONFIG_USE_HOSTIO
            pUserModel->EnableEditCalcSkeletalAnimations();
#endif
            if (0 == strcmp("env", pResModel->GetName()))
            {
                s_Holder.pEnvModel = pUserModel;
            }
            else
            {
                s_Holder.models.PushBack(pUserModel);
            }
        }
    }

    // カメラ初期化。
    {
        s_CameraPosition.Set(0.0f, 2.5f, 7.5f);
        s_CameraUp.Set(0.0f, 1.0f, 0.0f);
        s_CameraTarget.Set(0.0f, 2.5f, 0.0f);

        s_View.GetViewMtx(0).LookAt(
            s_CameraPosition,
            s_CameraUp,
            s_CameraTarget);
    }

    // メインループ
    while (g3ddemo::ProcessMsg())
    {
        if (!pad.Read() || pad.IsTriggered(g3ddemo::Pad::BUTTON_START))
        {
            g3ddemo::PostQuitMsg();
        }

        // 計算
        {
            pCtx->TempPrepare();

            // カメラコントロール。
            ControllCamera(&s_CameraPosition, &s_CameraUp, &s_CameraTarget, &s_View.GetViewMtx(0), &pad);
            s_View.GetViewMtx(0).LookAt(s_CameraPosition, s_CameraUp, s_CameraTarget);

            // 投影行列。
            s_View.GetProjMtx(0).Perspective(
                nw::g3d::Math::DegToRad(45.0f), 16.0f / 9.0f, 0.01f, 1000.0f);

            if (!s_Holder.models.Empty())
            {
            #if NW_G3D_CONFIG_USE_HOSTIO
                nw::g3d::edit::EditServer::Instance().CalcAnimations();
            #endif
            }

            for (int idxModel = 0, numModel = s_Holder.models.Size();
                 idxModel < numModel; ++idxModel)
            {
                g3ddemo::UserModel* pUserModel = s_Holder.models[idxModel];
                pUserModel->CalcAnim();
                pUserModel->CalcWorld();
                pUserModel->UpdateFrame();

                pUserModel->UpdateShader(); // 更新に GL コンテクストが必要。
            }

            if (s_Holder.pEnvModel)
            {
                s_Holder.pEnvModel->UpdateShader();
            }

        #ifdef USE_INPUTCAPTURE
            inputCapture.Read();
        #endif
        }

        // 残っているコマンドの完了を待ちます。
        //nw::g3d::GfxManage::DrawDone(); // 直列実行では不要

        // GPU 待ちが必要な計算
        {
            pCtx->TempPrepare(); // 更新に GL コンテクストが必要。

            for (int idxModel = 0, numModel = s_Holder.models.Size();
                 idxModel < numModel; ++idxModel)
            {
                g3ddemo::UserModel* pUserModel = s_Holder.models[idxModel];
                pUserModel->CalcGPU();
            }

            // ビュー毎の更新。
            for (int idxView = 0; idxView < NUM_VIEW; ++idxView)
            {
                s_View.Calc(idxView);

                for (int idxModel = 0, numModel = s_Holder.models.Size();
                     idxModel < numModel; ++idxModel)
                {
                    g3ddemo::UserModel* pUserModel = s_Holder.models[idxModel];
                    pUserModel->CalcViewGPU(idxView);
                }
            }

            // 環境
            if (s_Holder.pEnvModel)
            {
                s_Holder.pEnvModel->CalcGPU();
            }

            {
                float ofsX = 1.5f;
                float ofsY = 2.0f;
                float line = debugPrint.GetFontHeight();
                debugPrint.Reset();
                debugPrint.PutRect(ofsX, ofsY, 15.0f, line * 3);
                debugPrint.PutStringFmt(ofsX, ofsY, "%d models are registered.", s_Holder.models.Size());
                ofsY += line;
            #ifndef USE_INPUTCAPTURE
                debugPrint.PutString(ofsX, ofsY, "- MaterialPick demo");
                ofsY += line;
                debugPrint.PutString(ofsX, ofsY, "  does not work for PC.");
                ofsY += line;
            #endif

                // マウス位置を取得してポインタを描画します。
                f32 normalPosX = 0.f;
                f32 normalPosY = 0.f;
            #ifdef USE_INPUTCAPTURE
                normalPosX = inputCapture.GetMousePosX() / 1280.f;
                normalPosY = inputCapture.GetMousePosY() / 720.f;
            #endif

                cursor.Reset();
                cursor.PutTriangle(
                    ( CURSOR_DATA[0][0] + normalPosX ) * 100.f,
                    ( CURSOR_DATA[0][1] + normalPosY ) * 100.f,
                    ( CURSOR_DATA[1][0] + normalPosX ) * 100.f,
                    ( CURSOR_DATA[1][1] + normalPosY ) * 100.f,
                    ( CURSOR_DATA[2][0] + normalPosX ) * 100.f,
                    ( CURSOR_DATA[2][1] + normalPosY ) * 100.f
                );

                selectedRect.Reset();
                if (s_MouseTriggered)
                {
                    nw::g3d::math::Vec2 pos;
                    pos.x = s_MouseTriggeredPos.x / 1280.f;
                    pos.y = s_MouseTriggeredPos.y / 720.f;
                    float width = normalPosX - pos.x;
                    float height = normalPosY - pos.y;

                    selectedRect.PutRect(
                        pos.x * 100.f,
                        pos.y * 100.f,
                        width * 100.f,
                        height * 100.f
                    );
                }
            }
        }

        nw::g3d::CPUCache::Sync(); // CPU キャッシュ操作の完了を待ちます。
        // GPU のリードキャッシュはここで一括して Invalidate します。
        nw::g3d::GPUCache::InvalidateAll();

    #ifdef USE_INPUTCAPTURE
        // ポインタ位置をピックします。
        if ( inputCapture.IsMouseButtonTriggered( nw::g3d::edit::MOUSE_BUTTON_LEFT ) )
        {
            s_MouseTriggered = true;
            s_MouseTriggeredPos.x = static_cast<float>( inputCapture.GetMousePosX() );
            s_MouseTriggeredPos.y = static_cast<float>( inputCapture.GetMousePosY() );
        }

        if ( inputCapture.IsMouseButtonReleased( nw::g3d::edit::MOUSE_BUTTON_LEFT ) )
        {
            nw::g3d::math::Vec2 releasedPos;
            releasedPos.x = static_cast<float>( inputCapture.GetMousePosX() );
            releasedPos.y = static_cast<float>( inputCapture.GetMousePosY() );

            static const int MAX_MATERIAL_PICK = 64;
            g3ddemo::MaterialPicker::PickInfo pickInfo[MAX_MATERIAL_PICK];

            // マテリアルピックを行います。
            // この中では、マテリアルピックのための情報を描画した画像を生成するために、
            // 独自にフレームバッファの設定や Draw および DrawDone を行っています。
            int pickedNum = s_MaterialPicker.Pick(
                pickInfo,
                MAX_MATERIAL_PICK,
                s_MouseTriggeredPos,
                releasedPos,
                pCtx
            );

            nw::g3d::DebugPrint("================================\n");
            nw::g3d::DebugPrint("MaterialPicker: %d materials picked.\n",pickedNum);

            if (pickedNum > 0)
            {
                for (int i =0; i < pickedNum; ++i)
                {
                    nw::g3d::DebugPrint("   ModelObj: 0x%X / MatId: %d\n", pickInfo[i].userModelPtr->GetModelObj(), pickInfo[i].matId);

                    // ピックしたマテリアルの選択情報を 3DEditor に送ります。
                    nw::g3d::edit::EditServer::Instance().PushPickupMaterial(pickInfo[i].userModelPtr->GetModelObj(), pickInfo[i].matId);
                }
            }
            else
            {
                nw::g3d::edit::EditServer::Instance().ClearPickupMaterial();
            }

            nw::g3d::DebugPrint("================================\n");

            s_MouseTriggered = false;
        }

        if (s_MouseTriggered && !inputCapture.GetIsCapturing())
        {
            s_MouseTriggered = false;
        }
    #endif

        // TV 描画。
        {
            nw::g3d::ClearBuffers(&colorBufferTV, &depthBufferTV,
                0.3f, 0.3f, 0.3f, 1.0f, GX2_CLEAR_BOTH);

            pCtx->Activate(); // Clear に変更されたコンテクストの復帰。
            frameBufferTV.Load();

            GX2ShaderMode shaderMode = GX2_SHADER_MODE_UNIFORM_BLOCK;
            nw::g3d::SetShaderMode(shaderMode);

            for (int idxModel = 0, numModel = s_Holder.models.Size();
                 idxModel < numModel; ++idxModel)
            {
                g3ddemo::UserModel* pUserModel = s_Holder.models[idxModel];
                pUserModel->DebugDraw(g3ddemo::PASS_DEFAULT, 0, shaderMode);
            }

            // 情報表示のためにシェーダモードを切り替えます。
            nw::g3d::SetShaderMode(GX2_SHADER_MODE_UNIFORM_BLOCK);
            g3ddemo::ScreenInfo::LoadState();
            debugPrint.Draw();
            selectedRect.Draw();
            cursor.Draw();

            g3ddemo::CopyOut(&colorBufferTV, GX2_SCAN_TARGET_TV);
        }

        nw::g3d::GfxManage::DrawDone();

        g3ddemo::SwapScanBuffers();

#if NW_G3D_CONFIG_USE_HOSTIO
        // 描画中のリソースを破棄してしまわないよう、描画完了後に呼び出します。
        // UnloadFile 内で描画リソースを破棄しないような遅延処理による対策も可。
        pCtx->TempPrepare();
        nw::g3d::edit::EditServer::Instance().Poll();
#endif
    }

    // グラフィックスリソースの終了処理。
    pCtx->TempPrepare(); // 更新に GL コンテクストが必要。

    s_MaterialPicker.Cleanup();

#ifdef USE_INPUTCAPTURE
    inputCapture.Close();
#endif

#if NW_G3D_CONFIG_USE_HOSTIO
    {
        nw::g3d::edit::EditServer::Instance().Clear(true);
        nw::g3d::edit::EditServer::Instance().Close();
        nw::g3d::edit::EditServer::DeleteInstance();
    }
#endif

    // エディットライブラリが所有しているモデルを破棄しないように、
    // エディットの終了処理後に残りのデータの破棄を行います。
    g3ddemo::DestroyAll(&s_Holder);

    s_View.Cleanup();

    debugPrint.Cleanup();
    cursor.Cleanup();
    selectedRect.Cleanup();

    frameBufferTV.Free(g3ddemo::FreeMem1);
    frameBufferTV.Cleanup();
    g3ddemo::FreeMem2(frameBufferTV.GetBufferPtr());

    pCtx->Cleanup();
    g3ddemo::FreeMem2(pCtx);
    pCtx = NULL;

    // 終了処理。
    g3ddemo::ShutdownDisplay();
    g3ddemo::Shutdown();

    return 0;
}

//---------------------------------------------------------------------------

namespace
{
g3ddemo::ModelAssign* SetupModelAssign(nw::g3d::ResModel* pResModel,
    nw::g3d::ResShaderArchive** pShaderArchiveArray, int shaderArchiveCount)
{
    g3ddemo::CreateShaderAssign(pResModel);
    g3ddemo::ModelAssign* pModelAssign = pResModel->GetUserPtr<g3ddemo::ModelAssign>();

    // モデルに記録されたデフォルトシェーダの関連付け。
    nw::g3d::BindResult result = nw::g3d::BindResult::NotBound();
    if (pShaderArchiveArray)
    {
        result = pModelAssign->BindShader(
            g3ddemo::PASS_DEFAULT, pShaderArchiveArray, shaderArchiveCount);
    }
    for (int idxShader = 0, numShader = s_Holder.shaders.Size();
         !result.IsComplete() && idxShader < numShader; ++idxShader)
    {
        nw::g3d::ResShaderArchive* pShaderArchive = s_Holder.shaders[idxShader];
        result = pModelAssign->BindShader(g3ddemo::PASS_DEFAULT, pShaderArchive);
    }
    //NW_G3D_ASSERT(result.IsComplete());

    // マテリアルピック用シェーダの関連付け。
    pModelAssign->BindShader(g3ddemo::PASS_MATERIAL_PICK, s_MaterialPicker.GetResShadingModel());

    pModelAssign->Setup();
    return pModelAssign;
}

g3ddemo::UserModel* SetupUserModel(nw::g3d::ModelObj* pModelObj)
{
    g3ddemo::CreateUserModel(pModelObj);
    g3ddemo::UserModel* pUserModel = pModelObj->GetUserPtr<g3ddemo::UserModel>();
    pUserModel->SetView(&s_View);
    if (s_Holder.pEnvModel)
    {
        pUserModel->SetEnv(s_Holder.pEnvModel->GetModelObj());
    }

    g3ddemo::SetupMaterialsArg argSetup(pModelObj);
    g3ddemo::SetupMaterials(argSetup);

    // アニメーションインスタンスの構築。
    g3ddemo::UserModel::CreateAnimObjArg createAnimObjArg;
    if (!s_Holder.modelAnims.Empty())
    {
        createAnimObjArg.Reserve(&s_Holder.modelAnims[0], s_Holder.modelAnims.Size());
    }
    pUserModel->CreateAnimObj(createAnimObjArg);

    return pUserModel;
}

} // anonymous namespace
