﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_DEMO_GFXUTILITY_H_
#define NW_G3D_DEMO_GFXUTILITY_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/fnd/g3d_GfxObject.h>
#include <nw/g3d/fnd/g3d_GfxState.h>
#include <nw/g3d/fnd/g3d_GfxShader.h>
#include <nw/g3d/fnd/g3d_GfxManage.h>
#include <g3ddemo_DemoUtility.h>

namespace nw { namespace g3d { namespace res {

class ResFile;
class ResShaderProgram;

}}} // namespace nw::g3d

namespace nw { namespace g3d { namespace demo {

//--------------------------------------------------------------------------------------------------
// Graphics

enum BufferType
{
    COLOR_BUFFER        = 0x1 << 0,
    AUX_BUFFER          = 0x1 << 1,
    DEPTH_BUFFER        = 0x1 << 2,
    HIZ_BUFFER          = 0x1 << 3,
    COLOR_AUX_BUFFER    = COLOR_BUFFER | AUX_BUFFER,
    DEPTH_HIZ_BUFFER    = DEPTH_BUFFER | HIZ_BUFFER,
    ALL_BUFFER          = COLOR_AUX_BUFFER | DEPTH_HIZ_BUFFER
};

struct ColorBufferTexture
{
    nw::g3d::GfxTexture texture;
    nw::g3d::GfxColorBuffer renderBuffer;
    bool isFTV;

    void Setup();
    void Cleanup();

    void Alloc(FuncAlloc funcAlloc, bit32 bufferType = ALL_BUFFER);
    void Free(FuncFree funcFree, bit32 bufferType = ALL_BUFFER);
};

struct DepthBufferTexture
{
    nw::g3d::GfxTexture texture;
    nw::g3d::GfxDepthBuffer renderBuffer;

    void Setup();
    void Cleanup();

    void Alloc(FuncAlloc funcAlloc, bit32 bufferType = ALL_BUFFER);
    void Free(FuncFree funcFree, bit32 bufferType = ALL_BUFFER);
};

class FrameBuffer
{
public:
    enum
    {
        MAX_WIDTH = 8192,
        MAX_HEIGHT = 8192,
        MAX_RENDER_TARGET = 8
    };

    FrameBuffer() {}

    struct InitArg;
    static size_t CalcSize(const InitArg& arg);
    bool Init(const InitArg& arg, void* pBuffer, size_t bufferSize);

    void Setup();
    void Cleanup();

    void Alloc(FuncAlloc funcAlloc, bit32 bufferType = ALL_BUFFER);
    void Free(FuncFree funcFree, bit32 bufferType = ALL_BUFFER);

    void Load();

    u32 GetWidth() const { return m_Width; }
    u32 GetHeight() const { return m_Height; }

    void SetMipLevel(u32 mipLevel);

    ColorBufferTexture* GetColorBufferTexture(GX2RenderTarget target)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(target, m_NumRenderTarget);
        return &m_pColorBuffer[target];
    }

    const ColorBufferTexture* GetColorBufferTexture(GX2RenderTarget target) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(target, m_NumRenderTarget);
        return &m_pColorBuffer[target];
    }

    DepthBufferTexture* GetDepthBufferTexture() { return m_pDepthBuffer; }
    const DepthBufferTexture* GetDepthBufferTexture() const { return m_pDepthBuffer; }

    nw::g3d::GfxViewport& GetViewport() { return m_Viewport; }
    const nw::g3d::GfxViewport& GetViewport() const { return m_Viewport; }

    nw::g3d::GfxScissor& GetScissor() { return m_Scissor; }
    const nw::g3d::GfxScissor& GetScissor() const { return m_Scissor; }

    void* GetBufferPtr()
    {
        return m_pColorBuffer ?
            static_cast<void*>(m_pColorBuffer) :
            static_cast<void*>(m_pDepthBuffer);
    }

private:
    u32 m_Handle;
    u32 m_Width;
    u32 m_Height;
    int m_NumRenderTarget;
    ColorBufferTexture* m_pColorBuffer;
    DepthBufferTexture* m_pDepthBuffer;
    nw::g3d::GfxViewport m_Viewport;
    nw::g3d::GfxScissor m_Scissor;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(FrameBuffer);
};

struct FrameBuffer::InitArg
{
    InitArg(u32 width, u32 height)
        : width(width)
        , height(height)
        , colorBufferCount(1)
        , colorBufferFormat(GX2_SURFACE_FORMAT_TC_R11_G11_B10_FLOAT)
        , colorBufferFTV(false)
        , useDepthBuffer(true)
    {
    }

    u32 width;
    u32 height;
    int colorBufferCount;
    GX2SurfaceFormat colorBufferFormat;

    bool colorBufferFTV;
    bool useDepthBuffer;
};

//--------------------------------------------------------------------------------------------------

class SimpleShader : public nw::g3d::GfxShader
{
public:
    struct Path
    {
        const char* gsh;
        const char* vertexShader;
        const char* geometryShader;
        const char* fragmentShader;
        const char* computeShader;
        const char* streamOut;
    };

    void Setup(const Path& path);
    void Cleanup();

#ifndef _WIN32
    void* pGsh;
#endif
};

//--------------------------------------------------------------------------------------------------

class ScreenInfo
{
public:
    void Setup(u32 quadCount);
    void Cleanup();

    void SetFontSize(float pctH, float screenAspectRatio = 16.0f / 9.0f);
    float GetFontWidth() const { return m_FontSize[0]; }
    float GetFontHeight() const { return m_FontSize[1]; }

    void SetShapeColor(u8 r, u8 g, u8 b, u8 a = 255);
    void SetFontColor(u8 r, u8 g, u8 b);

    void Reset() { m_Count = 0; }

    void PutRect(float pctX, float pctY, float pctW, float pctH);
    void PutLineH(float pctX, float pctY, float pctW);
    void PutLineV(float pctX, float pctY, float pctH);
    void PutTriangle(float pctX0, float pctY0, float pctX1, float pctY1, float pctX2, float pctY2);

    void PutString(float pctX, float pctY, const char* str);
    void PutStringFmt(float pctX, float pctY, const char* format, ...);

    void DumpTexture();

    void DCFlush() const;

    void Draw() const;

    static void LoadState();

protected:
    enum
    {
        PERCENT         = 100,
        BIT_X           = 11,
        BIT_Y           = 10,
        BIT_U           = 1,
        BIT_V           = 7,
        BIT_COLOR       = 3,
        SHIFT_X         = 0,
        SHIFT_Y         = SHIFT_X + BIT_X,
        SHIFT_U         = SHIFT_Y + BIT_Y,
        SHIFT_V         = SHIFT_U + BIT_U,
        SHIFT_COLOR     = SHIFT_V + BIT_V,
        MASK_X          = (1 << BIT_X) - 1,
        MASK_Y          = (1 << BIT_Y) - 1,
        MASK_U          = (1 << BIT_U) - 1,
        MASK_V          = (1 << BIT_V) - 1,
        MASK_COLOR      = (1 << BIT_COLOR) - 1
    };

    static u32 QuantizeX(float pctX);
    static u32 QuantizeY(float pctY);

    void Pack(u32 x, u32 y, u32 u, u32 v, u32 c);
    void PutChar(u32 left, u32 top, u32 right, u32 bottom, u32 charCode);

private:
    nw::g3d::GfxBuffer m_Buffer;
    u32 m_Count;
    float m_FontSize[2];
    u8 m_FontColor[4]; // 左上 右上 左下 右下
    u8 m_ShapeColor[4]; // U V Alpha padding

    static int s_RefCount;
    static SimpleShader s_Shader;
    static nw::g3d::ResFile* s_pTexture;
    static nw::g3d::GfxSampler s_Sampler;
    static nw::g3d::GfxPolygonCtrl s_PolygonCtrl;
    static nw::g3d::GfxDepthCtrl s_DepthCtrl;
    static nw::g3d::GfxAlphaTest s_AlphaTest;
    static nw::g3d::GfxColorCtrl s_ColorCtrl;
    static nw::g3d::GfxBlendCtrl s_BlendCtrl;
    static nw::g3d::GfxChannelMasks s_ChannelMask;
};

//--------------------------------------------------------------------------------------------------

class ProcessMeter
{
public:
    enum
    {
        MAX_GPU_TIMESPAN    = 16,
        MAX_CPU_TIMESPAN    = 64,
        MAX_TIMESPAN        = MAX_GPU_TIMESPAN + MAX_CPU_TIMESPAN,
        MAX_QUAD            = 256,
        LOG_BIT_SHIFT       = 16,
        LOG_BIT_MASK        = 0xFFFF0000
    };

    enum Group
    {
        GROUP_GPU,
        GROUP_CPU,
        GROUP_CPU0,
        GROUP_CPU1,
        GROUP_CPU2,
        GROUP_FRAME,
        GROUP_SYS_GPU,
        GROUP_SYS_CPU,
        GROUP_SYS_CPU0,
        GROUP_SYS_CPU1,
        GROUP_SYS_CPU2,
        NUM_GROUP,
        NUM_GROUP_METER = GROUP_FRAME,
        NUM_GROUP_USER  = GROUP_FRAME + 1
    };

    void Setup();
    void Cleanup();

    void Calc();
    void Draw() const;

    int BeginTimeSpan(Group group, u8 r, u8 g, u8 b, u8 a = 0xFF, bool addToNextFrame = false);
    int BeginTimeSpan(Group group, bool addToNextFrame = false);
    void EndTimeSpan(int index);

protected:
    void PutFrameFG(int numFrame, u8 color[4]);
    void PutFrameBG(float pctFrame, u8 color[4]);
    void PutTimeSpan(float pctBegin, float pctTimeSpan, Group group, u8 color[4]);

private:
    struct Log
    {
#ifdef _WIN32
        volatile long numGPUTimeSpan;
        volatile long numCPUTimeSpan;
        s32 GetNumGPUTimeSpanNonAtomic() { return numGPUTimeSpan; }
        s32 GetNumCPUTimeSpanNonAtomic() { return numCPUTimeSpan; }
#else
        OSAtomicVar numGPUTimeSpan;
        OSAtomicVar numCPUTimeSpan;
        s32 GetNumGPUTimeSpanNonAtomic() { return numGPUTimeSpan.u.s32; }
        s32 GetNumCPUTimeSpanNonAtomic() { return numCPUTimeSpan.u.s32; }
#endif
        GPUClock::Tick beginGPU;
        GPUClock::Tick endGPU;
        struct GPUTimeSpan
        {
            Group group;
            u8 color[4];
            GPUClock* begin;
            GPUClock* end;
        } gpu[MAX_GPU_TIMESPAN];
        struct CPUTimeSpan
        {
            Group group;
            u8 color[4];
            CPUClock::Tick begin;
            CPUClock::Tick end;
        } cpu[MAX_CPU_TIMESPAN];
    } m_Log[3];
    GPUClock* m_pClockBuffer; // Invalidate するために分離する
    int m_CurrentLog;
    float m_ScaleX;
    float m_FrameScaleX;
    float m_OffsetX;
    float m_OffsetY;
    float m_RowHeight;
    u8 m_DefaultColor[NUM_GROUP][4];
    u8 m_FontColor[3];
    u8 m_SysAlpha;
    ScreenInfo m_ScreenInfo;
};

//--------------------------------------------------------------------------------------------------

class TextureDataConverter
{
public:
    enum TGAFormat
    {
        TGAFormat_R8,
        TGAFormat_R8_G8_B8,
        TGAFormat_R8_G8_B8_A8
    };

    static size_t CalcSize(const nw::g3d::GfxTexture* gfxTexture, TGAFormat tgaFormat);

    // GX2_SURFACE_FORMAT_TC_R8_UNORM と GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM のみ変換可能
    // 他のフォーマットの場合は一旦 GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM などに変換してください。
    static void Convert(void* pBuffer, size_t size, const nw::g3d::GfxTexture* gfxTexture, TGAFormat tgaFormat, FuncAlloc funcDebugAlloc, FuncFree funcDebugFree);


private:
    struct TgaHeader
    {
        u8  identSize;     // ヘッダに続く拡張領域のサイズ
        u8  colorMapType;
        u8  imageType;
        u8  colormapp_info[5];
        u16 xOrigin;
        u16 yOrigin;
        u16 imageWidth;
        u16 imageHeight;
        u8  pixelDepth;
        u8  imageDescriptor;
    };

    struct TgaFooter
    {
        u32 ext;           // 0
        u32 dev;           // 0
        u8  signeture[18]; // "TRUEVISION-XFILE.\0"
    };
};

//--------------------------------------------------------------------------------------------------
class DisplayList
{
public:
    enum
    {
        // ディスプレイリストのサイズを指定します。
        // 4MB が指定可能な最大サイズです。
        MAX_DISPLAY_LIST_SIZE   = 0x3FFF00,
        MAX_BUFFER_COUNT        = 2
    };

    void Setup(int bufferCount, int bufferSize = MAX_DISPLAY_LIST_SIZE);
    void Cleanup();
    void Reset();

    void Begin();
    void End();

    void Call();
    void DirectCall();

private:
    void* m_pBuffer[MAX_BUFFER_COUNT];
    u32 m_Size[MAX_BUFFER_COUNT];

    int m_CurrentBuffer;
    int m_BufferCount;
    int m_BufferSize;
};

//--------------------------------------------------------------------------------------------------

class ExportBuffer
{
public:
    ExportBuffer() : m_pExpBlock(NULL), m_pBufCPU(NULL) {}

    void Setup(void* pGPUBuffer, size_t size, int numBuffering = 1);
    void Cleanup();

    size_t GetSize() { return m_Size; }
    void* GetCPUBuf(int buferIndex = 0);
    const void* GetCPUBuf(int bufferIndex = 0) const;

    void Load(int bufferIndex = 0) { m_Buffer.LoadExportBuffer(bufferIndex); }

    void CPUFlush(int bufferIndex = 0);
    void CPUInvalidate(int bufferIndex = 0);
    void GPUFlush(int bufferIndex = 0) { m_Buffer.FlushExportBuffer(bufferIndex); }

    void CopyToCPU(int bufferIndex = 0);
    void CopyToGPU(int bufferIndex = 0);

private:
    size_t m_Size;
    GfxBuffer m_Buffer;
    void* m_pExpBlock;
    void* m_pBufCPU;
};

//--------------------------------------------------------------------------------------------------

void LoadUniformBlock(const nw::g3d::GfxBuffer* pBlock,
     const nw::g3d::ResShaderProgram* pShaderProgram, int blockIndex, int bufferIndex = 0);
void LoadTextureSampler(const nw::g3d::GfxTexture* pTexture, const nw::g3d::GfxSampler* pSampler,
    const nw::g3d::ResShaderProgram* pShaderProgram, int samplerIndex);

}}} // namespace nw::g3d::demo

#endif // NW_G3D_DEMO_GFXUTILITY_H_
