﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/types.h>
#include <nw/demo.h>
#include <nw/dev.h>
#include <nw/gfnd.h>
#include <nw/math.h>
#include <nw/ut.h>
#include <nw/font.h>

//------------------------------------------------------------------------------
//
// 欧文ワードラッピングのデモです。
//
// nw::font::WordWrapping クラスを用いて、行の表示幅に応じて文字列の適切な
// 位置に改行コードを挿入します。
//

//------------------------------------------------------------------------------
//   変数の宣言
//------------------------------------------------------------------------------

// 定数
const u32 c_HeapSize    = 64 * 1024 * 1024;
const u32 c_ScreenW     = 1280;
const u32 c_ScreenH     = 720;
const u32 c_CharNum     = 4096;
const u32 c_Margin      = 25;

enum Tag
{
    TAG_BEGIN = 0x0001 // 強調表示の開始
    , TAG_END = 0x0002 // 強調表示の終了
};

// ワードラップ処理の対象となるソース文字列です。
const char16 s_Text[] =
    L"\001nw::font::WordWrapping Demo\002\n"
    L"\n"
    L"\001Creating a Layout\002\n"
    L"\n"
    L"As a first step, here we provide a description of the basic work "
    L"flow, which involves:\n"
    L"\n"
    L"* Creating a new layout\n"
    L"* Positioning the pane\n"
    L"* Previewing in the Viewer\n"
    L"* Saving the data\n"
    L"\n"
    L"\001Creating a New Layout\002\n"
    L"\n"
    L"Start LayoutEditor and create a new layout file by selecting "
    L"File > Create New > Layout."
    L"A window with the title \"Untitled\" will appear on the screen. "
    L"This is called the Layout Window."
    L"With LayoutEditor, you create game screen layouts by "
    L"positioning images in the Layout Window.\n"
    L"\n"
    L"\001Description of the Layout Window Screen\002\n"
    L"\n"
    L"Look at the Layout Window and notice that a black square drawn "
    L"on the screen."
    L"This square is called the layout screen region (A)."
    L"The layout screen region corresponds to the screen shown on the "
    L"actual hardware."
    L"The red line drawn in the center of the screen is the coordinate "
    L"origin of the layout screen region (B). "
    L"(The layout coordinate system is a three-dimensional coordinate "
    L"system consisting of a two-dimensional X-Y axis plus a "
    L"one-dimensional Z axis.)\n"
    L"\n"
    L"\001Moving the Display Area (View)\002\n"
    L"\n"
    L"Pressing the space bar on the Layout window changes the figure "
    L"of the mouse cursor to a hand symbol (A)."
    L"At this point, the editor is in View Change mode ."
    L"While in this View Change mode, the display region in the window "
    L"can be changed by dragging the mouse (B).\n"
    L"\n"
    L"\001Changing Scale of the Display Area (View)\002\n"
    L"\n"
    L"Confirm that there is a combo box displaying at 100.0% and a "
    L"slider with + and - icons at the lower right of the Layout "
    L"window (A)."
    L"You can operate these to zoom in and zoom out.\n"
    L"\n"
    L"\001Create a New Null Pane\002\n"
    L"\n"
    L"Right-click in the window and select Create Pane > Null Pane.\n"
    L"\n"
    L"A green square appears in the Layout window. This is a Null pane."
    L"A pane is the smallest component of the layout data, representing "
    L"an individual image on the screen."
    L"Use the editor to position panes and create layouts.\n"
    L"\n"
    L"In addition to Null panes, there are also four other types of panes."
    L"A Null pane serves only an auxiliary purpose and is not displayed "
    L"on the actual game screen.";

// ワードラップ処理された文字列を格納するバッファです。
// ソース文字列＋挿入される改行コード分の容量が必要です。
char16*                         s_WordWrappedText;

// s_WordWrappedTextに格納可能な文字数(ヌル終端を含む)です。
u32                             s_WordWrappedTextSize;

// 表示する行数です。
u32                             s_NumLines;

// アロケーター
nw::ut::MemoryAllocator         s_Allocator;
// デモシステム
nw::demo::DemoSystem*           s_pDemo = NULL;
// グラフィックスコンテキスト
nw::gfnd::GraphicsContext       s_GraphicsContext;
void*                           s_VtxBuffer = NULL;
// デモで使用するフォント
nw::font::ResFont*              s_pFont = NULL;
u8*                             s_pFontBinary = NULL;
// フォント描画クラス
nw::font::RectDrawer            s_Drawer;
// フォント描画文字列バッファクラスの内部メモリ
void*                           s_pFontGraphicsBuf = NULL;
// フォント描画文字列バッファクラス
nw::font::DispStringBuffer*     s_pDrawStringBuf = NULL;
// 投影行列
nw::math::MTX44                 s_projection;

#if defined(NW_PLATFORM_CAFE)
static const char* const DEMO_FONT_FILENAME = "nintendo_NTLG-DB_002.bffnt";
#else
static const char* const DEMO_FONT_FILENAME = "nintendo_NTLG-DB_002_Nw4f.bffnt";
#endif

//---------------------------------------------------------------------------
// 表示する文字列の最大文字数を指定して、表示文字列用バッファを確保します。
// 確保した文字列バッファクラスへのポインタを返します。
//---------------------------------------------------------------------------
nw::font::DispStringBuffer*
InitializeFont_AllocDispStringBuffer_(int charMax)
{
    // 指定した文字数分バッファを確保
    const u32 drawBufSize   = nw::font::CharWriter::GetDispStringBufferSize(charMax);
    const u32 gfxBufSize    = nw::font::CharWriter::GetGraphicsBufferSize(charMax);

    void* pDispStringBuf    = s_Allocator.Alloc(drawBufSize);
    s_pFontGraphicsBuf      = s_Allocator.Alloc(gfxBufSize, nw::font::CharWriter::GRAPHICS_BUFFER_ALIGNMENT);

    // 表示文字列バッファを初期化
    return nw::font::CharWriter::InitDispStringBuffer(pDispStringBuf, s_pFontGraphicsBuf, charMax);
}

//------------------------------------------------------------------------------
// 描画クラスを初期化します。
//------------------------------------------------------------------------------
#if defined(NW_PLATFORM_CAFE)
void
InitRectDrawer_(nw::font::RectDrawer& rectDrawer, int charMax, nw::ut::MemoryRange& fontShaderBinary)
{
    // Cafeではシェーダも同時に渡す必要がある。
    const u32 vtxBufferSize = nw::font::RectDrawer::GetWorkBufferSize(charMax, fontShaderBinary);
    nw::ut::SafeFree(s_VtxBuffer, &s_Allocator);
    s_VtxBuffer = s_Allocator.Alloc(vtxBufferSize);
    rectDrawer.Initialize(s_VtxBuffer, charMax, fontShaderBinary);
}
#else
void
InitRectDrawer_(nw::font::RectDrawer& rectDrawer, int charMax)
{
    const u32 vtxBufferSize = nw::font::RectDrawer::GetWorkBufferSize(charMax);
    nw::ut::SafeFree(s_VtxBuffer, &s_Allocator);
    s_VtxBuffer = s_Allocator.Alloc(vtxBufferSize);
    rectDrawer.Initialize(s_VtxBuffer, charMax);
}
#endif

//------------------------------------------------------------------------------
// フォントの処理
//------------------------------------------------------------------------------
void
InitializeFont_()
{
    nw::dev::FileDeviceManager* pFileSystem = nw::dev::FileDeviceManager::GetInstance();

    //-----------------------------------------
    // フォントの初期化
    {
        // フォントの読み込み
        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path        = DEMO_FONT_FILENAME;
        loadArg.allocator   = &s_Allocator;
        loadArg.alignment   = nw::font::RESOURCE_ALIGNMENT;

        s_pFontBinary = pFileSystem->Load( loadArg );

        // フォントリソースのセット。
        // 内部でテクスチャ初期化をするため、コンテキストをロックして実行する。
        nw::gfnd::Graphics::GetInstance()->LockDrawContext();
        {
            s_pFont = new( s_Allocator.Alloc( sizeof( nw::font::ResFont ) ) ) nw::font::ResFont();
            bool result = s_pFont->SetResource(s_pFontBinary);
            NW_ASSERT(result);
        }
        nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
    }

    //-----------------------------------------
    // RectDrawer の初期化
    #if defined(NW_PLATFORM_CAFE)
    {
        nw::ut::MemoryRange pFontShaderBinary;
        // シェーダバイナリの読み込み(Cafe プラットフォームだけの処理)
        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path        = "font_BuildinShader.gsh";
        loadArg.allocator   = &s_Allocator;

        u8* binary = pFileSystem->Load( loadArg );
        pFontShaderBinary = nw::ut::MakeMemoryRange(binary, loadArg.readSize);

        // 内部でシェーダ初期化をするため、コンテキストをロックして実行する。
        nw::gfnd::Graphics::GetInstance()->LockDrawContext();
        {
            InitRectDrawer_(s_Drawer, c_CharNum, pFontShaderBinary);
        }
        nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
        nw::ut::SafeFree(binary, &s_Allocator);
    }
    #else
        // 内部でシェーダ初期化をするため、コンテキストをロックして実行する。
        nw::gfnd::Graphics::GetInstance()->LockDrawContext();
        {
            InitRectDrawer_(s_Drawer, c_CharNum);
        }
        nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
    #endif

    //-----------------------------------------
    // StringBuffer の初期化
    s_pDrawStringBuf = InitializeFont_AllocDispStringBuffer_(c_CharNum);

    //-----------------------------------------
    // プロジェクション行列の計算
    nw::math::MTX44Ortho(&s_projection, 0, static_cast<f32>(c_ScreenW), static_cast<f32>(c_ScreenH), 0, 0.0f, 300.0f);

    //-----------------------------------------
    // ワードラップ処理のためのメモリの確保
    {
        // 表示する行数です。
        s_NumLines = (c_ScreenH - 2 * c_Margin) / s_pFont->GetHeight();

        // ワードラップ処理したテキストを格納する文字列バッファを確保
        s_WordWrappedTextSize = nw::ut::GetArrayLength(s_Text) + s_NumLines;
        s_WordWrappedText = static_cast<char16*>(s_Allocator.Alloc(s_WordWrappedTextSize * sizeof(char16)));
    }
}

//---------------------------------------------------------------------------
// TextWriterへの設定をカプセル化します。
//---------------------------------------------------------------------------
class TextWriterSetting
{
public:
    TextWriterSetting()
        : m_ScaleH(1.0f)
        , m_ScaleV(1.0f)
        , m_TextColor(nw::ut::Color4u8::WHITE)
    {}

    explicit TextWriterSetting(const nw::font::WideTextWriter& writer)
    {
        m_ScaleH = writer.GetScaleH();
        m_ScaleV = writer.GetScaleV();
        m_TextColor = writer.GetTextColor();
    }

    // 文字を強調します。
    TextWriterSetting& SetMagnify(const TextWriterSetting& base)
    {
        m_ScaleH = base.m_ScaleH * 2.f;
        m_ScaleV = base.m_ScaleV * 2.f;
        m_TextColor = nw::ut::Color4u8(nw::ut::Color4u8::GRAY);
        return *this;
    }

    // 初期値に戻します。
    TextWriterSetting& Set(const TextWriterSetting& setting)
    {
        *this = setting;
        return *this;
    }

    // TextWriterに設定を適用します。
    void Apply(nw::font::WideTextWriter& writer) const
    {
        writer.SetScale(m_ScaleH, m_ScaleV);
        writer.SetTextColor(m_TextColor);
    }

    f32 m_ScaleH;
    f32 m_ScaleV;
    nw::ut::Color4u8 m_TextColor;
};

//---------------------------------------------------------------------------
// タグで囲まれた文章を強調表示するタグプロセッサです。
//
// WideTagProcessor::Process() および　WideTagProcessor::CalcRect() に渡される
// pContext->writer はテキストライターのコピーです。
// このためテキストライターに設定した内容は処理の完了後は失われます。
//
// WordWrapping は行単位で処理を行うため、タグプロセッサの設定が行にまたがる
// 場合には、行の間で設定内容を伝える必要があります。
//---------------------------------------------------------------------------
class MyTagProcessor : public nw::font::WideTagProcessor
{
    typedef nw::font::WideTagProcessor Base;

public:
    MyTagProcessor()
    {}

    // TextWriterの状態を通常設定として記憶します。
    void InitializeSetting(const nw::font::WideTextWriter& writer)
    {
        m_NormalSetting = TextWriterSetting(writer);
        m_CurrentSetting = m_NormalSetting;
    }

    // TextWriterの状態を現在の設定内容として記憶します。
    void SetCurrentSetting(const nw::font::WideTextWriter& writer)
    {
        m_CurrentSetting = TextWriterSetting(writer);
    }

    // 現在の設定内容を取得します。
    const TextWriterSetting& GetCurrentSetting() const
    {
        return m_CurrentSetting;
    }

    // TextWriterを通常状態に戻します。
    void ApplyNormalSetting(nw::font::WideTextWriter& writer)
    {
        m_NormalSetting.Apply(writer);
    }

protected:
    virtual Operation CalcRect(
        nw::ut::Rect* pRect,
        u16 code,
        ContextType* pContext)
    {
        nw::font::WideTextWriter& writer = *pContext->writer;
        switch (code)
        {
        case TAG_BEGIN:
            m_CurrentSetting.SetMagnify(m_NormalSetting).Apply(writer);
            break;

        case TAG_END:
            m_CurrentSetting.Set(m_NormalSetting).Apply(writer);
            break;
        }

        return Base::CalcRect(pRect, code, pContext);
    }

    virtual Operation Process(
        u16 code,
        ContextType* pContext)
    {
        nw::font::WideTextWriter& writer = *pContext->writer;
        switch (code)
        {
        case TAG_BEGIN:
            m_CurrentSetting.SetMagnify(m_NormalSetting).Apply(writer);
            break;

        case TAG_END:
            m_CurrentSetting.Set(m_NormalSetting).Apply(writer);
            break;
        }

        return Base::Process(code, pContext);
    }

public:
    TextWriterSetting m_NormalSetting;
    TextWriterSetting m_CurrentSetting;
};

//---------------------------------------------------------------------------
// 一行に収まる文字列の範囲を求めます。
//---------------------------------------------------------------------------
class MyWordWrapCallback : public nw::font::WordWrapCallback
{
public:
    MyWordWrapCallback(const nw::font::WideTextWriter& textWriter, MyTagProcessor* pTagProcessor)
        : m_TextWriter(textWriter)
        , m_TextWriterSetting(textWriter)
        , m_pTagProcessor(pTagProcessor)
        , m_CurrentLine(0)
    {
    }

    // 与えられた文字列のうち一行の表示に収まる範囲を求めます。
    //
    // 表示される文字列の範囲は WideTextWriter::SetWidthLimit() で
    // 表示できる幅を指定したのち WideTextWriter::FindPosOfWidthLimit() を
    // 呼ぶことで求めることができます。
    //
    // タグプロセッサにより WideTextWriter の状態を変更している場合には注意が必要です。
    //
    // 一行の表示に収まる範囲と、実際に改行の行われる位置は異なるため、
    // 改行の位置が確定した時点であらためて改行位置における WideTextWriter の
    // 状態を取得しなければなりません。
    //
    // WideTextWriter::CalcStringWidth() はタグプロセッサを呼び出すとき、
    // コンテキストとしてテキストライタのコピーを渡すため、
    // このテキストライタに設定された内容は CalcStringWidth() の終了後は保持されません。
    // そこで本デモでは、タグプロセッサにテキストライタへの設定内容を保持しておき、
    // 行にまたがって設定を受け渡すようにしています。
    //
    virtual const char16* GetLineBreakLimit(const char16* str, const char16* end)
    {
        if (m_CurrentLine == NULL)
        {
            m_CurrentLine = str;
        }

        if (m_CurrentLine != str)
        {
            const char16* prevLine = m_CurrentLine;
            m_CurrentLine = str;

            // 改行が行われると str の値が変わります。
            // 前の行の末尾の状態にTextWriterを更新するため、
            // タグプロセッサの処理をやり直します。
            m_pTagProcessor->SetCurrentSetting(m_TextWriter);
            u32 prevLength = m_CurrentLine - prevLine;
            m_TextWriter.CalcStringWidth(prevLine, prevLength);
            m_pTagProcessor->GetCurrentSetting().Apply(m_TextWriter);
        }

        // 行頭でのタグプロセッサの状態を設定します。
        m_pTagProcessor->SetCurrentSetting(m_TextWriter);

        // 文字列の表示幅を求めます。
        const char16* lineBreak = m_TextWriter.FindPosOfWidthLimit(str, end - str);

        return lineBreak;
    }

    nw::font::WideTextWriter m_TextWriter;
    TextWriterSetting m_TextWriterSetting;
    MyTagProcessor* m_pTagProcessor;
    const char16* m_CurrentLine;
};

//---------------------------------------------------------------------------
// 更新処理
//---------------------------------------------------------------------------
void
Calc_(
    nw::font::RectDrawer*           pDrawer,
    nw::font::DispStringBuffer*     pDrawStringBuf,
    nw::font::ResFont*              pFont
)
{
    // 描画設定
    nw::font::WideTextWriter writer;
    writer.SetDispStringBuffer(pDrawStringBuf);
    writer.SetFont(pFont);
    writer.SetCursor(static_cast<f32>(c_Margin), static_cast<f32>(c_Margin));
    writer.SetScale(1.0f, 1.0f);

    MyTagProcessor tagProcessor;

    // 現在の設定を通常設定として記憶します。
    tagProcessor.InitializeSetting(writer);

    writer.SetTagProcessor(&tagProcessor);

    const u32 LOOP = 60 * 4;
    static u32 count = 0;

    // 一行に表示できる幅を WideTextWriter に設定します。
    // これにより WideTextWriter::GetLineBreak() により一行に表示できる
    // 文字列の範囲を求めることができます。
    count = (count + LOOP - 1) % LOOP;
    f32 widthLimit =  (static_cast<f32>(c_ScreenW) - c_Margin * 2) * (count + 1) / LOOP;
    writer.SetWidthLimit(widthLimit);

    // 一行の表示範囲に収まる文字列の範囲を判定するコールバックを作成します。
    MyWordWrapCallback func(writer, &tagProcessor);

    // 画面に収まる行数に制限します。
    //
    // Note: タグプロセッサにより文字の大きさを変更している場合、
    // 改行が挿入される位置によって行の高さが変わるため、
    // 事前計算による行数制限は正しく機能しません。
    // これは nw::font::WordWrapping の制限です。
    //
    nw::font::WordWrapConfig config;
    config.SetMaxLineNum(s_NumLines);

    // ワードラップ処理を行います。
    nw::font::WordWrapping::CalcWordWrapping(
        /* writeSize */ NULL,
        s_WordWrappedText,
        s_WordWrappedTextSize,
        s_Text,
        func,
        config);

    // 描画用に WideTextWriter を初期化します。
    tagProcessor.ApplyNormalSetting(writer);

    // 文字上端、下端カラーの設定はここで行います。
    // 上端、下端カラーは頂点カラーで実現されるため、
    // カラーを変更した場合、描画コマンドの生成を再度行わなければならない点に注意してください。
    // writer.SetTextColor(nw::ut::Color8(0, 255, 0, 255), nw::ut::Color8(255, 255, 255, 255));

    {
        writer.StartPrint();
        {
            (void)writer.Print(s_WordWrappedText);
        }
        writer.EndPrint();

        // 第 2 引数には、影付き文字列描画のパラメータを設定します。
    	//
    	// 注意：この処理は、GPUが描画中に参照するデータを書き換えます。
    	// そのため、DispStringBuffer は内部で頂点バッファをダブルバッファリングしており、
        // BuildVertexElements() 内でカレントバッファを切り替えています。
        pDrawer->BuildVertexElements(pDrawStringBuf, NULL);
    }
}

//------------------------------------------------------------------------------
// 描画ステートを設定します。
//
// フォントは基本的にデプステストなし、カリングなしで描画します。
// これらの設定はフォント内では行っていないため、Drawを呼び出す前に設定しておく必要があります。
// 逆にアルファテスト及びブレンドの設定はフォント内で書き換えるため注意が必要です。
//------------------------------------------------------------------------------
void
Draw_SetupDrawState_()
{
    //-----------------------------------------
    // 深度テストを有効にするかどうか設定します。
    // 深度書き込みを有効にするかどうか設定します。
    #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        glDisable( GL_DEPTH_TEST );
        glDepthMask( GL_FALSE );
    #else
        GX2SetDepthOnlyControl(
                GX2_FALSE,
                GX2_FALSE,
                GX2_COMPARE_ALWAYS);
    #endif

    //-----------------------------------------
    // カリングモード
    #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        glDisable( GL_CULL_FACE );
    #else
        GX2SetPolygonControl(
        GX2_FRONT_FACE_CCW, // frontFaceMode
        GX2_DISABLE,      // cullFront
        GX2_DISABLE,       // cullBack
        GX2_FALSE,
        GX2_POLYGON_MODE_TRIANGLE, // polygonModeFront
        GX2_POLYGON_MODE_TRIANGLE,  // polygonModeBack
        GX2_DISABLE,     // polyOffsetFrontEnable
        GX2_DISABLE,      // polyOffsetBackEnable
        GX2_DISABLE // pointLineOffsetEnable
        );
    #endif

    //-----------------------------------------
    // カラーブレンド
    #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        glDisable(GL_COLOR_LOGIC_OP); // 論理演算はしない
        glEnable(GL_BLEND); // ブレンド有効
        glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA); // ソースアルファでブレンド
        glBlendEquation(GL_FUNC_ADD);
        NW_GL_ASSERT();
    #elif defined(NW_PLATFORM_CAFE)
        GX2SetColorControl(
            GX2_LOGIC_OP_COPY,
            0x01,
            GX2_DISABLE,
            GX2_ENABLE);
        GX2SetBlendControl(
            GX2_RENDER_TARGET_0,
            GX2_BLEND_SRC_ALPHA,
            GX2_BLEND_ONE_MINUS_SRC_ALPHA,
            GX2_BLEND_COMBINE_ADD,
            GX2_FALSE,
            GX2_BLEND_SRC_ALPHA,
            GX2_BLEND_ONE_MINUS_SRC_ALPHA,
            GX2_BLEND_COMBINE_ADD);
    #endif

    //-----------------------------------------
    // アルファ設定
    #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        glDisable(GL_ALPHA_TEST);// アルファテスト無効
    #elif defined(NW_PLATFORM_CAFE)
        GX2SetAlphaTest(GX2_FALSE, GX2_COMPARE_NEVER, 0.0f);
    #endif
}

//------------------------------------------------------------------------------
// 描画処理
// TODO : drawContent.flags |= font::DrawContent::INVISIBLE_BORDER;
//------------------------------------------------------------------------------
void
Draw_(
    nw::font::RectDrawer*           pDrawer,
    nw::font::DispStringBuffer*     pDrawStringBuf)
{
    nw::font::DrawContent drawContent;
    drawContent.dispStringBuffer        = pDrawStringBuf;
    drawContent.projectionMatrix        = &s_projection;
    drawContent.viewMatrix              = &nw::math::MTX34::Identity();
    drawContent.localMatrix             = &nw::math::MTX34::Identity();

    // 描画ステートの設定
    Draw_SetupDrawState_();

    // 白黒カラー補完の設定
    // ここで色を変える場合は、描画コマンドの再生成が必要ないところに注意してください！
    // 頻繁にコマンド再生成をおこなうと処理負荷に影響があります。
    // 必要がなければ、白黒カラー補完で済ませるほうがパフォーマンス的に有利です。
    drawContent.interpolateBlack.Set(0.f, 0.f, 0.f, 1.f);
    drawContent.interpolateWhite.Set(1.f, 1.f, 1.f, 1.f);

    pDrawer->Draw(drawContent);
}

//------------------------------------------------------------------------------
//  フォント固有の 1 フレーム分の処理
//------------------------------------------------------------------------------
void
ProcFrame_Font_()
{
    // 更新
    Calc_(&s_Drawer, s_pDrawStringBuf, s_pFont);

    // 描画
    Draw_(&s_Drawer, s_pDrawStringBuf);
}

//------------------------------------------------------------------------------
// GPU状態 のクリア
void
ProcFrame_DemoClearGPUState_()
{
    #if defined(NW_PLATFORM_CAFE)
        // GPU の readキャッシュを全域クリアする
        GX2InvalidateType clearType =
                static_cast<GX2InvalidateType>(
                GX2_INVALIDATE_ATTRIB_BUFFER |
                GX2_INVALIDATE_TEXTURE |
                GX2_INVALIDATE_CONSTANT_BUFFER |
                GX2_INVALIDATE_SHADER);

        GX2Invalidate(clearType, 0x00000000, 0xffffffff);
    #endif
}

//------------------------------------------------------------------------------
//  1 フレーム分の処理
void
ProcFrame_()
{
    // フレームの開始処理 / デモパッドの更新
    nw::gfnd::Graphics* pGraphics = nw::gfnd::Graphics::GetInstance();
    s_pDemo->BeginFrame();
    s_pDemo->UpdatePad();

    pGraphics->LockDrawContext();
    {
        ProcFrame_DemoClearGPUState_();
        s_pDemo->ClearFrameBuffers();
        s_pDemo->SetViewport( 0.f, 0.f, static_cast<f32>( c_ScreenW ), static_cast<f32>( c_ScreenH ), 0.f, 1.f );
        s_pDemo->SetScissor( 0.f, 0.f, static_cast<f32>( c_ScreenW ), static_cast<f32>( c_ScreenH ) );

        // フォントの更新処理
        ProcFrame_Font_();
    }
    pGraphics->UnlockDrawContext();

    // バッファのスワップ / フレームの終了処理 / VBlank 待ち
    s_pDemo->SwapBuffer();
    s_pDemo->EndFrame();
    s_pDemo->WaitForVBlank();
}

//------------------------------------------------------------------------------
// ルートヒープを確保
void*
AllocDemoHeap_()
{
    #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        return malloc( c_HeapSize );
    #else
        return MEMAllocFromDefaultHeap( c_HeapSize );
    #endif
}

//------------------------------------------------------------------------------
// ルートヒープを開放
void
FreeDemoHeap_(void* addr)
{
    #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        free( addr );
    #else
        MEMFreeToDefaultHeap( addr );
    #endif
}

//------------------------------------------------------------------------------
// デモ共有の初期化
void
InitDemoCommon_()
{
    // デモファイルシステムの初期化
    nw::dev::FileDeviceManager*     pFileSystem = nw::dev::FileDeviceManager::GetInstance();
    NW_NULL_ASSERT(pFileSystem);
    nw::dev::FileDeviceManager::CreateArg fileDeviceArg;
    fileDeviceArg.allocator = &s_Allocator;
    pFileSystem->Initialize( fileDeviceArg );

    // デモシステムの初期化
    nw::demo::DemoSystem::CreateArg arg;
    arg.allocator   = &s_Allocator;
    arg.waitVBlank  = 1;
    arg.width       = c_ScreenW;
    arg.height      = c_ScreenH;
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    // primitiveRendererInitialize は 本来 false でOKです。
    // PC版の Demo 描画システムの都合から true を指定します。
    arg.primitiveRendererInitialize   = true;
#endif
    s_pDemo          = new( s_Allocator.Alloc( sizeof( nw::demo::DemoSystem ) ) ) nw::demo::DemoSystem( arg );
    NW_NULL_ASSERT(s_pDemo);
    s_pDemo->Initialize();

    // グラフィックシステムの初期化
    s_pDemo->InitializeGraphicsSystem();
}

//------------------------------------------------------------------------------
// デモ共有の終了処理
void
FinarizeDemoCommon_()
{
    // グラフィックシステムの終了処理
    s_pDemo->FinalizeGraphicsSystem();

    // デモシステムの終了処理
    s_pDemo->Finalize();
    nw::ut::SafeFreeWithDestruct( s_pDemo, &s_Allocator );

    // デモファイルシステムの終了処理
    nw::dev::FileDeviceManager*     pFileSystem = nw::dev::FileDeviceManager::GetInstance();
    pFileSystem->Finalize();

    nw::ut::SafeFree(s_VtxBuffer, &s_Allocator);
}

//------------------------------------------------------------------------------
// フォントの終了処理
//------------------------------------------------------------------------------
void
FinalizeFont_()
{
    nw::ut::SafeFreeWithDestruct( s_WordWrappedText, &s_Allocator );
    nw::ut::SafeFreeWithDestruct( s_pDrawStringBuf, &s_Allocator );
    nw::ut::SafeFree( s_pFontGraphicsBuf, &s_Allocator );

    s_Drawer.Finalize();

    // フォントの破棄
    nw::ut::SafeFreeWithDestruct( s_pFont, &s_Allocator );
    nw::ut::SafeFree(s_pFontBinary, &s_Allocator);
}

//------------------------------------------------------------------------------
// メインループを抜けるべきか？
bool
NeedsToBreakMainLoop_()
{
    return s_pDemo->GetPad()->IsHoldAll(
        nw::demo::Pad::MASK_L |
        nw::demo::Pad::MASK_R |
        nw::demo::Pad::MASK_X ) ||
        s_pDemo->IsExiting();
}

//------------------------------------------------------------------------------
//  メイン 関数
//------------------------------------------------------------------------------
int
NwDemoMain(int argc, char **argv)
{
    NW_UNUSED_VARIABLE(argc);
    NW_UNUSED_VARIABLE(argv);

    //----------------------------------------------
    // 初期化
    // ヒープの初期化
    void* pHeap = AllocDemoHeap_();
    // アロケータの初期化
    s_Allocator.Initialize( pHeap, c_HeapSize );

    // デモシステム共有の初期化
    InitDemoCommon_();
    // フォントの初期化処理
    InitializeFont_();

    //----------------------------------------------
    // メインループ
    {
        NW_LOG("Start demo.\n");
        // L,R,X 同時押しで抜ける
        while (!NeedsToBreakMainLoop_())
        {
            ProcFrame_();
        }
    }

    //----------------------------------------------
    // 終了処理
    // フォントの終了処理
    FinalizeFont_();
    // デモシステム共有の終了処理
    FinarizeDemoCommon_();

    // アロケータの終了処理
    s_Allocator.Finalize();
    // ヒープの解放
    FreeDemoHeap_(pHeap);

    NW_LOG("End demo.\n");

    return 0;
}
