﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Extensions;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.IO;
using System.Threading;
using System.Threading.Tasks;

namespace NintendoWare.Spy
{
    /// <summary>
    /// NwSnd 最終出力波形 Spy モデルです。
    /// </summary>
    public sealed class FinalOutSpyModel : SpyModel
    {
        /// <summary>
        /// バージョン 0.1.0.0
        /// </summary>
        /// <remarks>
        /// パケットフォーマット：
        /// <code>
        /// struct FinalOutPacketData {
        ///     u32 audioFrame;
        ///     u32 channelBitFlag;
        ///     u32 samplePerFrame;
        ///     s16 finalOutput[144 * (number of channels)];
        /// };
        /// </code>
        /// </remarks>
        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        private static readonly Version Version_0_1_0_0 = new Version(0, 1, 0, 0);

        /// <summary>
        /// 非サポートバージョン。
        /// 最新のサポートバージョンよりマイナーバージョンを１つ大きくします。
        /// </summary>
        private static readonly Version VersionUnexpected = new Version(0, 2, 0, 0);

        public const int SampleRate = 48000;
        public const int SamplePerFrame = 144;
        public const int BitsPerSample = 16;
        public const int ChannelCountMax = 8;
        public const int TvOutChannelCount = 6;
        public const int DrcOutChannelCount = 2;

        public const string TvOutName = "TV";
        public const string DrcOutName = "DRC";

        // FinalOutのキャッシュするインスタンスの最大数です。
        private const int MaxCacheCount = 100;

        //-----------------------------------------------------------------

        private readonly List<LoudnessInfo> _loudnessList = new List<LoudnessInfo>();

        private Frame _audioFrameBegin = Frame.InvalidValue;
        private Frame _audioFrameEnd = Frame.InvalidValue;
        private Frame _loudnessAudioFrameEnd = Frame.InvalidValue;

        private readonly LoudnessAccumulator _loudnessAccumulator = new LoudnessAccumulator();

        public delegate void UpdateEventHandler(object sender, EventArgs e);
        public event UpdateEventHandler UpdateEvent;

        /// <summary>
        /// バックグラウンドで新しいラウドネス情報が計算されたときに発生するイベントです。
        /// このイベントは非UIスレッドで実行されます。
        /// </summary>
        public event EventHandler LoudnessUpdateEvent;

        // AudioFrameIndex -> DataBlockID変換用のディクショナリです。
        private readonly Dictionary<int, long> _frameToIdDictionary = new Dictionary<int, long>();

        // FinalOutのインスタンスをキャッシュするディクショナリです。
        // キーは AudioFrameIndexになります。
        private readonly Dictionary<int, FinalOut> _finalOutDictionary = new Dictionary<int, FinalOut>();

        // FinalOutのインスタンスをキャッシュから古いものから外す為のリストです。
        private readonly List<int> _audioFrameIndexList = new List<int>();

        private readonly object _getFinalOutLockObject = new object();

        /// <summary>
        /// ラウドネスを計算するTaskです。
        /// </summary>
        private Task _calcLoudnessTask;

        /// <summary>
        /// ラウドネス計算タスクを終了させるときに使います。
        /// </summary>
        private readonly CancellationTokenSource _calcLoudnessCancellationTokenSource = new CancellationTokenSource();

        /// <summary>
        /// ラウドネス計算要求の待ち行列です。
        /// </summary>
        private readonly ConcurrentQueue<CalcLoudnessRequest> _calcLoudnessRequests = new ConcurrentQueue<CalcLoudnessRequest>();

        /// <summary>
        /// ラウドネス計算の結果です。UIスレッドでloudnessListに統合されるまで蓄積されます。
        /// </summary>
        private readonly ConcurrentQueue<CalcLoudnessResult> _calcLoudnessResults = new ConcurrentQueue<CalcLoudnessResult>();

        private readonly object _calcLoudnessLock = new object();

        //-----------------------------------------------------------------

        public enum ChannelIndex : int
        {
            MainFrontLeft,
            MainFrontRight,
            MainRearLeft,
            MainRearRight,
            MainFrontCenter,
            MainLfe,
            DrcLeft,
            DrcRight
        }

        public class SampleData
        {
            public short[][] Samples { get; } = new short[ChannelCountMax][].Populate(_ => new short[SamplePerFrame]);

            public SampleData()
            {
            }
        }

        public class ChannelInfo
        {
            public int MaxSampleValue { get; set; }
            public int MinSampleValue { get; set; }
            public float TotalMeanSquare { get; set; }

            public ChannelInfo()
            {
            }
        }

        public enum ChanneBitFlag : uint
        {
            MainFrontLeft = (1 << 0),
            MainFrontRight = (1 << 1),
            MainRearLeft = (1 << 2),
            MainRearRight = (1 << 3),
            MainFrontCenter = (1 << 4),
            MainLfe = (1 << 5),
            DrcLeft = (1 << 6),
            DrcRight = (1 << 7)
        }

        public class FinalOut
        {
            public SampleData SampleData { get; } = new SampleData();
            public ChannelInfo[] ChannelInfoList { get; } = new ChannelInfo[ChannelCountMax].Populate(_ => new ChannelInfo());

            public FinalOut()
            {
            }
        }

        public class LoudnessInfo
        {
            public float MomentaryLoudnessValue { get; set; }
            public float ShortTermLoudnessValue { get; set; }
            public float AbsGatedLoudnessValue { get; set; }
            public float RelGatedLoudnessValue { get; set; }
            public ChannelLoudnessInfo[] Channels { get; } = new ChannelLoudnessInfo[ChannelCountMax].Populate(_ => new ChannelLoudnessInfo());

            public LoudnessInfo()
            {
            }
        }

        public class ChannelLoudnessInfo
        {
            public float PeakValue { get; set; }
            public float RmsValue { get; set; }

            public ChannelLoudnessInfo()
            {
            }
        }

        /// <summary>
        /// ラウドネス情報をバックグラウンドで計算するタスクに与えるリクエストです。
        /// </summary>
        private class CalcLoudnessRequest
        {
            public uint AudioFrame { get; }
            public SpyDataBlock DataBlock { get; }
            public BinaryReader Reader { get; }

            public CalcLoudnessRequest(uint audioFrame, SpyDataBlock dataBlock, BinaryReader reader)
            {
                this.AudioFrame = audioFrame;
                this.DataBlock = dataBlock;
                this.Reader = reader;
            }
        }

        /// <summary>
        /// ラウドネス情報のバックグラウンドでの計算結果です。
        /// </summary>
        private class CalcLoudnessResult
        {
            public uint AudioFrame { get; }
            public LoudnessInfo LoudnessInfo { get; }

            public CalcLoudnessResult(uint audioFrame, LoudnessInfo loudnessInfo)
            {
                this.AudioFrame = audioFrame;
                this.LoudnessInfo = loudnessInfo;
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 最終出力波形情報のある最初のオーディオフレームです。
        /// 情報が無いときは Frame.InvalidValue となります。
        /// </summary>
        public Frame AudioFrameBegin
        {
            get { return _audioFrameBegin; }
            private set { this.SetPropertyValue(ref _audioFrameBegin, value); }
        }

        /// <summary>
        /// 最終出力波形情報のある最後のオーディオフレームです。
        /// 情報が無いときは Frame.InvalidValue となります。
        /// </summary>
        public Frame AudioFrameEnd
        {
            get { return _audioFrameEnd; }
            private set { this.SetPropertyValue(ref _audioFrameEnd, value); }
        }

        /// <summary>
        /// ラウドネス情報のある最後のオーディオフレームです。
        /// 情報が無いときはFrame.InvalidValueとなります。
        ///
        /// ラウドネス情報はバックグラウンドで計算されます。
        /// 計算結果をこのプロパティに反映するにはUpdateLoudness()を呼び出してください。
        /// </summary>
        public Frame LoudnessAudioFrameEnd
        {
            get { return _loudnessAudioFrameEnd; }
            private set { this.SetPropertyValue(ref _loudnessAudioFrameEnd, value); }
        }

        //-----------------------------------------------------------------

        public FinalOutSpyModel()
        {
        }

        protected override void Dispose(bool disposing)
        {
            _calcLoudnessCancellationTokenSource.Cancel();
            _calcLoudnessTask = null;
            base.Dispose(disposing);
        }

        /// <summary>
        /// 最終出力波形の情報を取得します。
        /// この関数はファイルからの読み込みを伴うため時間がかかります。
        /// 非同期処理を検討してください。
        /// </summary>
        /// <param name="audioFrame"></param>
        /// <returns>情報が無い場合は null を返します。</returns>
        public FinalOut GetFinalOut(Frame audioFrame)
        {
            lock (_getFinalOutLockObject)
            {
                if (audioFrame < AudioFrameBegin || AudioFrameEnd < audioFrame)
                {
                    return null;
                }

                int audioFrameIndex = (int)(audioFrame - AudioFrameBegin);

                if (_frameToIdDictionary.ContainsKey(audioFrameIndex) == false)
                {
                    return null;
                }

                if (_finalOutDictionary.ContainsKey(audioFrameIndex) == true)
                {
                    return _finalOutDictionary[audioFrameIndex];
                }

                FinalOut finalOut = GetFinalOut(_frameToIdDictionary[audioFrameIndex]);
                if (finalOut == null)
                {
                    return null;
                }

                _finalOutDictionary[audioFrameIndex] = finalOut;

                _audioFrameIndexList.Add(audioFrameIndex);
                if (_audioFrameIndexList.Count > FinalOutSpyModel.MaxCacheCount)
                {
                    _finalOutDictionary.Remove(_audioFrameIndexList[0]);
                    _audioFrameIndexList.RemoveAt(0);
                }

                return finalOut;
            }
        }

        private FinalOut GetFinalOut(long dataBlockID)
        {
            BinaryReader reader = CreateDataReader(ReadData(dataBlockID));
            if (reader == null)
            {
                return null;
            }

            uint audioFrame = reader.ReadUInt32();
            return ReadFinalOut(reader, audioFrame);
        }

        private FinalOut ReadFinalOut(BinaryReader reader, uint audioFrame)
        {
            uint channelBitFlag = reader.ReadUInt32();
            uint samplePerFrame = reader.ReadUInt32();

            if (samplePerFrame != SamplePerFrame)
            {
                return null;
            }

            FinalOut finalOut = new FinalOut();
            SampleData sampleData = finalOut.SampleData;

            for (int index = 0; index < SamplePerFrame; index++)
            {
                if ((channelBitFlag & (uint)ChanneBitFlag.MainFrontRight) != 0)
                {
                    sampleData.Samples[(int)ChannelIndex.MainFrontRight][index] = reader.ReadInt16();
                }
                if ((channelBitFlag & (uint)ChanneBitFlag.MainFrontLeft) != 0)
                {
                    sampleData.Samples[(int)ChannelIndex.MainFrontLeft][index] = reader.ReadInt16();
                }
                if ((channelBitFlag & (uint)ChanneBitFlag.MainFrontCenter) != 0)
                {
                    sampleData.Samples[(int)ChannelIndex.MainFrontCenter][index] = reader.ReadInt16();
                }
                if ((channelBitFlag & (uint)ChanneBitFlag.MainLfe) != 0)
                {
                    sampleData.Samples[(int)ChannelIndex.MainLfe][index] = reader.ReadInt16();
                }
                if ((channelBitFlag & (uint)ChanneBitFlag.MainRearRight) != 0)
                {
                    sampleData.Samples[(int)ChannelIndex.MainRearRight][index] = reader.ReadInt16();
                }
                if ((channelBitFlag & (uint)ChanneBitFlag.MainRearLeft) != 0)
                {
                    sampleData.Samples[(int)ChannelIndex.MainRearLeft][index] = reader.ReadInt16();
                }
            }
            for (int index = 0; index < SamplePerFrame; index++)
            {
                if ((channelBitFlag & (uint)ChanneBitFlag.DrcRight) != 0)
                {
                    sampleData.Samples[(int)ChannelIndex.DrcRight][index] = reader.ReadInt16();
                }
                if ((channelBitFlag & (uint)ChanneBitFlag.DrcLeft) != 0)
                {
                    sampleData.Samples[(int)ChannelIndex.DrcLeft][index] = reader.ReadInt16();
                }
            }

            for (int channelIndex = 0; channelIndex < ChannelCountMax; channelIndex++)
            {
                ChannelInfo channelInfo = finalOut.ChannelInfoList[channelIndex];

                channelInfo.MaxSampleValue = int.MinValue;
                channelInfo.MinSampleValue = int.MaxValue;
                channelInfo.TotalMeanSquare = 0;

                for (int index = 0; index < SamplePerFrame; index++)
                {
                    int sampleValue = (int)sampleData.Samples[channelIndex][index];
                    channelInfo.MaxSampleValue = Math.Max(channelInfo.MaxSampleValue, sampleValue);
                    channelInfo.MinSampleValue = Math.Min(channelInfo.MinSampleValue, sampleValue);
                    channelInfo.TotalMeanSquare += sampleValue * sampleValue;
                }
            }

            return finalOut;
        }

        /// <summary>
        /// バックグラウンドで計算したラウドネス情報を取得可能にします。
        /// </summary>
        public void UpdateLoudness()
        {
            CalcLoudnessResult result;
            while (_calcLoudnessResults.TryDequeue(out result))
            {
                int audioFrameIndex = (int)(result.AudioFrame - this.AudioFrameBegin);
                // データが欠損している場合はリストをnullで埋めます。
                while (_loudnessList.Count < audioFrameIndex)
                {
                    _loudnessList.Add(null);
                }

                _loudnessList.Add(result.LoudnessInfo);
            }

            if (_loudnessList.Count > 0)
            {
                this.LoudnessAudioFrameEnd = this.AudioFrameBegin + _loudnessList.Count - 1;
            }
        }

        /// <summary>
        /// ラウドネス情報を取得します。
        /// </summary>
        /// <param name="audioFrame"></param>
        /// <returns>情報が無い場合は null を返します。</returns>
        public LoudnessInfo GetLoudness(Frame audioFrame)
        {
            if (audioFrame < AudioFrameBegin || LoudnessAudioFrameEnd < audioFrame)
            {
                return null;
            }
            else
            {
                return _loudnessList[(int)(audioFrame - AudioFrameBegin)];
            }
        }

        protected override void OnPushData(SpyDataBlock dataBlock)
        {
            if (this.DataVersion >= VersionUnexpected)
            {
                return;
            }

            var reader = CreateDataReader(dataBlock);

            uint audioFrame = reader.ReadUInt32();

            if (this.AudioFrameBegin == Frame.InvalidValue)
            {
                this.AudioFrameBegin = new Frame(audioFrame);
            }

            if (this.AudioFrameEnd < audioFrame)
            {
                this.AudioFrameEnd = new Frame(audioFrame);
            }

            int audioFrameIndex = (int)(audioFrame - this.AudioFrameBegin);

            var request = new CalcLoudnessRequest(audioFrame, dataBlock, reader);
            lock (_calcLoudnessLock)
            {
                _calcLoudnessRequests.Enqueue(request);
                if (_calcLoudnessTask == null)
                {
                    var cancellationToken = _calcLoudnessCancellationTokenSource.Token;
                    _calcLoudnessTask = Task.Factory.StartNew(
                        CalcLoudnessFunc,
                        (object)cancellationToken,
                        cancellationToken);
                }
            }

            _frameToIdDictionary[audioFrameIndex] = dataBlock.ID;

            if (UpdateEvent != null)
            {
                UpdateEvent(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// バックグランドでラウドネス情報を計算するTaskです。
        /// </summary>
        /// <param name="state"></param>
        private void CalcLoudnessFunc(object state)
        {
            var cancellationToken = (CancellationToken)state;

            bool needUpdate = false;
            while (!cancellationToken.IsCancellationRequested)
            {
                CalcLoudnessRequest args;
                lock (_calcLoudnessLock)
                {
                    if (!_calcLoudnessRequests.TryDequeue(out args))
                    {
                        _calcLoudnessTask = null;
                        break;
                    }
                }

                uint audioFrame = args.AudioFrame;
                var finalOut = this.ReadFinalOut(args.Reader, audioFrame);
                var loudness = _loudnessAccumulator.Accumulate(finalOut);

                needUpdate = true;
                _calcLoudnessResults.Enqueue(new CalcLoudnessResult(audioFrame, loudness));
            }

            if (cancellationToken.IsCancellationRequested)
            {
                return;
            }

            if (needUpdate)
            {
                if (this.LoudnessUpdateEvent != null)
                {
                    this.LoudnessUpdateEvent(this, EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// ラウドネスの計算処理をまとめたクラスです。
        /// ラウドネスの計算に必要なサンプル数はフレーム毎のサンプル数よりも多いので
        /// 計算途中の状態を保持します。
        /// </summary>
        private class LoudnessAccumulator
        {
            private static readonly Loudness.Parameter Parameter = Loudness.Parameter48kHz;

            private readonly LoudnessCalculator _loudnessCalculator = new LoudnessCalculator();
            private readonly LoudnessCalculator _loudnessCalculatorDrc = new LoudnessCalculator();
            private readonly Loudness.LoudnessContext[] _loudnessContext = new Loudness.LoudnessContext[ChannelCountMax].Populate(_ => new Loudness.LoudnessContext());
            private readonly float[] _channelValue = new float[ChannelCountMax];
            private readonly float[] _sampleSquareSum = new float[ChannelCountMax];
            private readonly float[] _samplePeak = new float[ChannelCountMax];
            private LoudnessInfo _lastLoudnessInfo;
            private int _accumulatedSampleCount;

            public LoudnessAccumulator()
            {
                ResetValue();
                _loudnessCalculator.SetParameter(Parameter);
                _loudnessCalculatorDrc.SetParameter(Parameter);
            }

            public LoudnessInfo Accumulate(FinalOut finalOut)
            {
                int sampleCount = 0;

                // アルゴリズムの前提条件
                Assertion.Operation.True(Parameter.StepSampleCount >= SamplePerFrame);

                // このフレームのサンプルを追加するとラウドネスの計算に必要なサンプル数に到達する場合
                if (SamplePerFrame + _accumulatedSampleCount >= Parameter.StepSampleCount)
                {
                    // ラウドネスの計算に足りないサンプル数だけ累積
                    sampleCount = Math.Min(SamplePerFrame, Parameter.StepSampleCount - _accumulatedSampleCount);
                    this.AccumulateSample(finalOut, 0, sampleCount);

                    // TVのラウドネスを計算
                    {
                        float totalMeanSquare = 0.0f;
                        float[] sampleSquareSumArray = new float[Loudness.ChannelCount];
                        float[] samplePeakArray = new float[Loudness.ChannelCount];
                        for (int channelIndex = 0; channelIndex < Loudness.ChannelCount; channelIndex++)
                        {
                            totalMeanSquare += _channelValue[channelIndex] * Loudness.ChannelWeightArray[channelIndex];
                            sampleSquareSumArray[channelIndex] = _sampleSquareSum[channelIndex];
                            samplePeakArray[channelIndex] = _samplePeak[channelIndex];
                        }

                        _loudnessCalculator.AddStep(totalMeanSquare, sampleSquareSumArray, samplePeakArray);
                    }

                    // DRCのラウドネスを計算
                    {
                        float totalMeanSquare = 0.0f;
                        float[] sampleSquareSumArray = new float[Loudness.ChannelCount];
                        float[] samplePeakArray = new float[Loudness.ChannelCount];
                        for (int channelIndex = 0; channelIndex < 2; channelIndex++)
                        {
                            int channelIndexAbs = channelIndex + (int)ChannelIndex.DrcLeft;

                            totalMeanSquare += _channelValue[channelIndexAbs] * Loudness.ChannelWeightArray[channelIndex];
                            sampleSquareSumArray[channelIndex] = _sampleSquareSum[channelIndexAbs];
                            samplePeakArray[channelIndex] = _samplePeak[channelIndexAbs];
                        }

                        _loudnessCalculatorDrc.AddStep(totalMeanSquare, sampleSquareSumArray, samplePeakArray);
                    }

                    _lastLoudnessInfo = ReadLoudnessInfo();

                    ResetValue();
                }

                // 残りのサンプルを累積
                {
                    this.AccumulateSample(finalOut, sampleCount, SamplePerFrame);
                    _accumulatedSampleCount += SamplePerFrame - sampleCount;
                }

                return _lastLoudnessInfo;
            }

            private void ResetValue()
            {
                for (int i = 0; i < ChannelCountMax; ++i)
                {
                    _channelValue[i] = 0;
                    _sampleSquareSum[i] = 0;
                    _samplePeak[i] = 0;
                }

                _accumulatedSampleCount = 0;
            }

            private void AccumulateSample(FinalOut finalOut, int from, int to)
            {
                for (int channelIndex = 0; channelIndex < ChannelCountMax; channelIndex++)
                {
                    // LFEは計算不要
                    if (channelIndex == (int)ChannelIndex.MainLfe)
                    {
                        continue;
                    }

                    short[] sample = finalOut.SampleData.Samples[channelIndex];
                    for (int i = from; i < to; i++)
                    {
                        float sampleValue = sample[i] / 32768.0f;

                        _samplePeak[channelIndex] = Math.Max(_samplePeak[channelIndex], Math.Abs(sampleValue));
                        _sampleSquareSum[channelIndex] += sampleValue * sampleValue;
                        float kOut = Loudness.CalcKWeightingFilter(_loudnessContext[channelIndex], sampleValue, Parameter);
                        _channelValue[channelIndex] += kOut * kOut;
                    }
                }
            }

            private LoudnessInfo ReadLoudnessInfo()
            {
                var loudness = new LoudnessInfo();

                loudness.MomentaryLoudnessValue = _loudnessCalculator.MomentaryLoudnessValue;
                loudness.ShortTermLoudnessValue = _loudnessCalculator.ShortTermLoudnessValue;
                loudness.AbsGatedLoudnessValue = _loudnessCalculator.AbsGatedLoudnessValue;
                loudness.RelGatedLoudnessValue = _loudnessCalculator.RelGatedLoudnessValue;

                for (int channelIndex = 0; channelIndex < Loudness.ChannelCount; channelIndex++)
                {
                    var channelInfo = loudness.Channels[channelIndex];
                    channelInfo.PeakValue = _loudnessCalculator.PeakValue[channelIndex];
                    channelInfo.RmsValue = _loudnessCalculator.RmsValue[channelIndex];
                }
                for (int channelIndex = 0; channelIndex < 2; channelIndex++)
                {
                    var channelInfo = loudness.Channels[channelIndex + (int)ChannelIndex.DrcLeft];
                    channelInfo.PeakValue = _loudnessCalculatorDrc.PeakValue[channelIndex];
                    channelInfo.RmsValue = _loudnessCalculatorDrc.RmsValue[channelIndex];
                }

                return loudness;
            }
        }
    }
}
