﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.ToolFoundation.Windows.Documents;
using NintendoWare.Spy.Framework.Settings;
using NintendoWare.Spy.Framework.Windows;
using NintendoWare.Spy.Plugins;
using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Diagnostics;
using System.Linq;
using System.Windows;

namespace NintendoWare.Spy.Windows
{
    /// <summary>
    /// Spy パネルのプレゼンターです。
    /// </summary>
    public abstract class SpyPanelPresenter : PanelPresenter
    {
        private const string ResourcesUri = "pack://application:,,,/SpyFoundation;component/Windows/SpyPanelResources.xaml";

        private static readonly Lazy<ResourceDictionary> Resources = new Lazy<ResourceDictionary>(
            () => new ResourceDictionary() { Source = new Uri(ResourcesUri) });

        private readonly object _observerOwner = new object();

        //-----------------------------------------------------------------

        /// <summary>
        /// 依存するサービスの型列挙子を取得します。
        /// </summary>
        public override IEnumerable<Type> DependentServiceTypes
        {
            get
            {
                yield return typeof(SpyService);
                yield return typeof(SpyPlaybackService);
                yield return typeof(SpyTemporaryDataService);
                yield return typeof(SettingsService);
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 必要な Spy データ名のコレクションを取得します。
        /// </summary>
        public IEnumerable<string> RequiredSpyDataNames { get; internal set; }

        /// <summary>
        /// サポートするプラットフォームの列挙子を取得します。
        /// </summary>
        public IEnumerable<string> SupportedPlatforms { get; internal set; }

        //-----------------------------------------------------------------

        /// <summary>
        /// RequiredSpyDataNames に含まれる Spy モデルが更新されると実行されます。
        /// </summary>
        /// <param name="dataName">更新された Spy モデルのデータ名を指定します。</param>
        /// <param name="model">
        /// 更新された Spy モデルを指定します。
        /// Spy モデルがなくなった場合は null を指定します。
        /// </param>
        protected abstract void UpdateSpyModel(string dataName, SpyModel model);

        /// <summary>
        /// 設定が更新されると実行されます。
        /// </summary>
        /// <param name="service">SettingsService を指定します。</param>
        protected virtual void OnSettingsApplied(SettingsService service) { }

        protected override void OnInitialize()
        {
            Ensure.Operation.Null(this.Content);

            this.GetSettingsService().Applied += this.NotifySettingsApplied;

            PropertyChangedObservation.GetObserver(_observerOwner, this.GetSpyConnectionInfo())
                .AddHandler(
                    nameof(SpyConnectionInfo.SelectedConnectionInfo),
                    (sender, e) => this.ValidateTargetPlatform());

            this.GetSpyService().EndLoadFiles += this.OneEndLoadFiles;

            CollectionChangedObservation.GetObserver(_observerOwner, this.GetSpyService().GetCurrentData().Models)
                .AddHandler(this.BeginUpdateSpyModel);

            base.OnInitialize();

            this.ValidateTargetPlatform();

            // HACK : ★今は SettingsService プロパティを private にしているので、
            //        起動時に設定を反映させるために、ここでOnSettingsApplied() を呼ぶ
            // TODO : ★SettingsSevice.Applied を廃止する等して、個別の設定変更を観測できるように
            //        なったら、この呼び出しは削除する
            this.NotifySettingsApplied();

            if (!this.GetSpyService().IsLoading)
            {
                this.BeginUpdateAllSpyModel();
            }
        }

        protected override void OnUninitialize()
        {
            this.GetSpyService().EndLoadFiles -= this.OneEndLoadFiles;

            this.GetSettingsService().Applied -= this.NotifySettingsApplied;

            PropertyChangedObservation.RemoveObservers(_observerOwner);
            CollectionChangedObservation.RemoveObservers(_observerOwner);

            base.OnUninitialize();
        }

        protected virtual void SetAvailability(bool isAvailable)
        {
            if (!isAvailable)
            {
                this.Content.SetValue(ContentAdorner.ContentTemplateProperty, Resources.Value["NotSupportedSpyPanelTemplate"]);
            }
            else
            {
                this.Content.SetValue(ContentAdorner.ContentTemplateProperty, null);
            }
        }

        [DebuggerStepThrough]
        protected SpyService GetSpyService()
        {
            return this.ServiceProvider.GetService<SpyService>();
        }

        [DebuggerStepThrough]
        protected SpyPlaybackService GetPlaybackService()
        {
            return this.ServiceProvider.GetService<SpyPlaybackService>();
        }

        [DebuggerStepThrough]
        private SpyTemporaryDataService GetSpyTemporaryDataService()
        {
            return this.ServiceProvider.GetService<SpyTemporaryDataService>();
        }

        [DebuggerStepThrough]
        private SettingsService GetSettingsService()
        {
            return this.ServiceProvider.GetService<SettingsService>();
        }

        private SpyConnectionInfo GetSpyConnectionInfo()
        {
            return this.GetSpyTemporaryDataService().GetSpyConnectionInfo();
        }

        private void BeginUpdateSpyModel(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Reset:
                    this.BeginUpdateAllSpyModel();
                    break;

                case NotifyCollectionChangedAction.Move:
                case NotifyCollectionChangedAction.Replace:
                    // 想定していないので未実装
                    throw new NotImplementedException();

                case NotifyCollectionChangedAction.Add:
                    if (this.GetSpyService().IsLoading)
                    {
                        // データファイルの読み込みが完了してから、まとめて設定します。
                        break;
                    }
                    foreach (var item in e.NewItems.OfType<KeyValuePair<string, SpyModel>>())
                    {
                        if (this.RequiredSpyDataNames.Contains(item.Key))
                        {
                            this.BeginUpdateSpyModel(item.Key, item.Value);
                        }
                    }
                    break;

                case NotifyCollectionChangedAction.Remove:
                    foreach (var item in e.OldItems.OfType<KeyValuePair<string, SpyModel>>())
                    {
                        if (this.RequiredSpyDataNames.Contains(item.Key))
                        {
                            this.BeginUpdateSpyModel(item.Key, null);
                        }
                    }
                    break;
            }
        }

        private SpyModel TryGetSpyModel(string dataName)
        {
            SpyModel result = null;
            this.GetSpyService().GetCurrentData().Models.TryGetValue(dataName, out result);
            return result;
        }

        private void BeginUpdateAllSpyModel()
        {
            foreach (var dataName in this.RequiredSpyDataNames)
            {
                this.BeginUpdateSpyModel(dataName, this.TryGetSpyModel(dataName));
            }
        }

        private void BeginUpdateSpyModel(string dataName, SpyModel model)
        {
            this.Content.Dispatcher.BeginInvoke(new Action(
                () => this.UpdateSpyModel(dataName, model)));
        }

        private void OneEndLoadFiles(object sender, EventArgs args)
        {
            this.BeginUpdateAllSpyModel();
        }

        private void NotifySettingsApplied(object sender, EventArgs e)
        {
            this.NotifySettingsApplied();
        }

        private void NotifySettingsApplied()
        {
            this.OnSettingsApplied(this.GetSettingsService());
        }

        private void ValidateTargetPlatform(object sender, EventArgs e)
        {
            this.ValidateTargetPlatform();
        }

        private void ValidateTargetPlatform()
        {
            this.SetAvailability(
                this.IsSupported(this.GetSpyConnectionInfo().SelectedConnectionInfo?.TargetPlatformName));
        }

        private ConnectionInfo FindConnectionPlugin(string targetPlatformName)
        {
            return this.GetSpyConnectionInfo()
                .ConnectionInfos
                ?.Where(info => info.TargetPlatformName == targetPlatformName)
                .FirstOrDefault();
        }

        private bool IsSupported(string targetPlatformName)
        {
            // すべてのプラットフォームで有効
            if (this.SupportedPlatforms.Contains(SpyPanelPlugin.PlatformNames.All))
            {
                return true;
            }

            var plugin = this.FindConnectionPlugin(targetPlatformName);

            // すべての NintendoSDK 対応プラットフォームで有効
            if (plugin != null && plugin.IsForNintendoSDK)
            {
                return this.SupportedPlatforms.Contains(SpyPanelPlugin.PlatformNames.AllForNintendoSdk);
            }

            // 指定プラットフォームで有効
            if (this.SupportedPlatforms.Contains(targetPlatformName))
            {
                return true;
            }

            return false;
        }
    }
}
