﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Markup;
using System.Xaml;

namespace NintendoWare.Spy.Settings
{
    /// <summary>
    /// アイテムを格納するコンテナのXML要素クラスです。
    /// </summary>
    /// <typeparam name="TItem">アイテムの型を指定します。</typeparam>
    [ContentProperty(nameof(Items))]
    public sealed class SettingsRoot
    {
        private readonly List<SettingItem> _items = new List<SettingItem>();

        //-----------------------------------------------------------------

        /// <summary>
        /// アイテムを取得または設定します。
        /// </summary>
        public IList<SettingItem> Items
        {
            get { return _items; }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// ストリームの内容を読み込み SettingsRoot を返します。
        /// </summary>
        /// <param name="stream">読み込み元ストリームを指定します。</param>
        /// <returns>読み込んだ SettingsRoot を返します。</returns>
        public static SettingsRoot Load(Stream stream)
        {
            using (var textReader = new StreamReader(stream))
            {
                var settings = ConvertToCurrentSettings(textReader);
                return LoadSettings(settings);
            }
        }

        /// <summary>
        /// TextReader の内容を読み込み SettingsRoot を返します。
        /// </summary>
        /// <param name="reader">読み込み元 TextReader を指定します。</param>
        /// <returns>読み込んだ SettingsRoot を返します。</returns>
        public static SettingsRoot Load(TextReader reader)
        {
            var settings = ConvertToCurrentSettings(reader);
            return LoadSettings(settings);
        }

        /// <summary>
        /// テキストの内容を読み込み SettingsRoot を返します。
        /// </summary>
        /// <param name="stream">読み込むテキストを指定します。</param>
        /// <returns>読み込んだ SettingsRoot を返します。</returns>
        public static SettingsRoot Load(string text)
        {
            using (var reader = new StringReader(text))
            {
                var settings = ConvertToCurrentSettings(reader);
                return LoadSettings(settings);
            }
        }

        private static SettingsRoot LoadSettings(string settings)
        {
            using (var reader = new StringReader(settings))
            {
                return XamlServices.Load(reader) as SettingsRoot;
            }
        }

        /// <summary>
        /// SettingsRoot をストリームに書き込みます。
        /// </summary>
        /// <param name="stream">書き込み先ストリームを指定します。</param>
        /// <param name="container">書き込むコンテンツを指定します。</param>
        /// <returns>読み込んだ SettingsRoot を返します。</returns>
        public static void Save(Stream stream, SettingsRoot container)
        {
            XamlServices.Save(stream, container);
        }

        /// <summary>
        /// SettingsRoot をストリームに書き込みます。
        /// </summary>
        /// <param name="writer">書き込み先 TextWriter を指定します。</param>
        /// <param name="container">書き込むコンテンツを指定します。</param>
        /// <returns>読み込んだ SettingsRoot を返します。</returns>
        public static void Save(TextWriter writer, SettingsRoot container)
        {
            XamlServices.Save(writer, container);
        }

        /// <summary>
        /// 指定した型とキーに該当するアイテムの有無を調べます。
        /// </summary>
        /// <typeparam name="TItem">アイテムの型を指定します。</typeparam>
        /// <param name="key">
        /// アイテムのキーを指定します。
        /// null を指定すると、すべてのキーを対象にします。
        /// </param>
        /// <returns>
        /// 該当するアイテムが見つかった場合は true、見つからなかった場合は false を返します。
        /// </returns>
        public bool ContainsItem<TItem>(string key)
        {
            try
            {
                this.GetItem<TItem>(key);
                return true;
            }
            catch (InvalidOperationException)
            {
                return false;
            }
        }

        /// <summary>
        /// 指定した型とキーに該当するアイテムを取得します。
        /// </summary>
        /// <typeparam name="TItem">アイテムの型を指定します。</typeparam>
        /// <param name="key">
        /// アイテムのキーを指定します。
        /// null を指定すると、すべてのキーを対象にします。
        /// </param>
        /// <returns>指定した型とキーに該当するアイテムを返します。</returns>
        public TItem GetItem<TItem>(string key)
        {
            var targetItem = _items
                .Where(item => key == null || item.Key == key)
                .First();

            return targetItem == null ? default(TItem) : (TItem)targetItem.Value;
        }

        /// <summary>
        /// 指定した型とキーに該当するアイテムを取得します。
        /// </summary>
        /// <typeparam name="TItem">アイテムの型を指定します。</typeparam>
        /// <param name="key">
        /// アイテムのキーを指定します。
        /// null を指定すると、すべてのキーを対象にします。
        /// </param>
        /// <param name="item">
        /// 指定した型とキーに該当するアイテムを返します。
        /// </param>
        /// <returns>
        /// 該当するアイテムが見つかった場合は true、見つからなかった場合は false を返します。
        /// </returns>
        public bool TryGetItem<TItem>(string key, out TItem item)
        {
            try
            {
                item = this.GetItem<TItem>(key);
                return true;
            }
            catch (InvalidOperationException)
            {
                item = default(TItem);
                return false;
            }
        }

        /// <summary>
        /// 古い設定を新しい表現に変換します。
        /// </summary>
        /// <param name="textReader"></param>
        /// <returns></returns>
        private static string ConvertToCurrentSettings(TextReader textReader)
        {
            const string SpyNamespace = "http://schemas.nintendo.com/nintendoware/spy";
            const string SoundToolFoundationObsoleteNamespace = "clr-namespace:NintendoWare.Sound.Foundation.Settings;assembly=SoundToolFoundationObsolete";

            bool isDefaultNamespaceWritten = false;
            var stringBuilder = new StringBuilder();
            var schemaContext = new XamlSchemaContext();

            using (var xamlWriter = new XamlXmlWriter(new StringWriter(stringBuilder), schemaContext))
            {
                using (var xamlReader = new XamlXmlReader(textReader))
                {
                    while (xamlReader.Read())
                    {
                        switch (xamlReader.NodeType)
                        {
                            case XamlNodeType.None:
                                break;

                            case XamlNodeType.StartObject:
                                {
                                    var type = xamlReader.Type;
                                    if (type.Name == "SettingsRoot" && type.PreferredXamlNamespace == SoundToolFoundationObsoleteNamespace)
                                    {
                                        xamlWriter.WriteStartObject(new XamlType(SpyNamespace, nameof(SettingsRoot), null, schemaContext));
                                    }
                                    else if (type.Name == "SettingItem" && type.PreferredXamlNamespace == SoundToolFoundationObsoleteNamespace)
                                    {
                                        xamlWriter.WriteStartObject(new XamlType(SpyNamespace, nameof(SettingItem), null, schemaContext));
                                    }
                                    else
                                    {
                                        xamlWriter.WriteStartObject(xamlReader.Type);
                                    }
                                }
                                break;

                            case XamlNodeType.GetObject:
                                xamlWriter.WriteGetObject();
                                break;

                            case XamlNodeType.EndObject:
                                xamlWriter.WriteEndObject();
                                break;

                            case XamlNodeType.StartMember:
                                xamlWriter.WriteStartMember(xamlReader.Member);
                                break;

                            case XamlNodeType.EndMember:
                                xamlWriter.WriteEndMember();
                                break;

                            case XamlNodeType.Value:
                                xamlWriter.WriteValue(xamlReader.Value);
                                break;

                            case XamlNodeType.NamespaceDeclaration:
                                {
                                    if (!isDefaultNamespaceWritten)
                                    {
                                        // spy の名前空間をデフォルトにします。
                                        xamlWriter.WriteNamespace(new NamespaceDeclaration(SpyNamespace, string.Empty));
                                        isDefaultNamespaceWritten = true;
                                    }

                                    var ns = xamlReader.Namespace;
                                    if (ns.Namespace == SoundToolFoundationObsoleteNamespace)
                                    {
                                        // 削除します。
                                    }
                                    else if (ns.Namespace == SpyNamespace)
                                    {
                                        // デフォルトの名前空間として出力済みなので削除します。
                                    }
                                    else
                                    {
                                        xamlWriter.WriteNamespace(xamlReader.Namespace);
                                    }
                                }
                                break;
                        }
                    }
                }
            }

            return stringBuilder.ToString();
        }
    }
}
