﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;

namespace NintendoWare.Spy.Foundation.Commands
{
    /// <summary>
    /// コマンドハンドラの基本実装クラスです。
    /// </summary>
    public abstract class CommandHandler : ICommandHandler
    {
        private readonly Command _targetCommand;

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="targetCommand">
        /// 対象コマンドを取得します。
        /// </param>
        public CommandHandler(Command targetCommand)
        {
            Ensure.Argument.NotNull(targetCommand);
            _targetCommand = targetCommand;
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 指定コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <param name="command">実行するコマンドを指定します。</param>
        /// <param name="commandArgs">コマンド引数を指定します。</param>
        /// <returns>コマンドの状態を返します。</returns>
        public CommandStatus QueryStatus(Command command, CommandArgs commandArgs)
        {
            if (command != _targetCommand)
            {
                return CommandStatus.Unsupported;
            }

            return QueryStatusImpl(command, commandArgs);
        }

        /// <summary>
        /// 指定コマンドを実行します。
        /// </summary>
        /// <param name="command">実行するコマンドを指定します。</param>
        /// <param name="commandArgs">コマンド引数を指定します。</param>
        /// <returns>コマンドの処理結果を返します。</returns>
        public CommandResult Execute(Command command, CommandArgs commandArgs)
        {
            if (command != _targetCommand)
            {
                return new CommandResult(CommandResult.Code.CommandIDNotSupported);
            }

            return ExecuteImpl(command, commandArgs);
        }

        /// <summary>
        /// 指定コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <param name="command">実行するコマンドを指定します。</param>
        /// <param name="commandArgs">コマンド引数を指定します。</param>
        /// <returns>コマンドの状態を返します。</returns>
        protected abstract CommandStatus QueryStatusImpl(Command command, CommandArgs commandArgs);

        /// <summary>
        /// 指定コマンドを実行します。
        /// </summary>
        /// <param name="command">実行するコマンドを指定します。</param>
        /// <param name="commandArgs">コマンド引数を指定します。</param>
        /// <returns>コマンドの処理結果を返します。</returns>
        protected abstract CommandResult ExecuteImpl(Command command, CommandArgs commandArgs);
    }
}
