﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using System;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.Reflection;

namespace NintendoWare.Spy.Communication
{
    internal class DataPacket : Packet<DataPacket.Content>
    {
        public DataPacket() :
            base(SpySession._Signature, (uint)PacketID.Data)
        { }

        //-----------------------------------------------------------------

        [SuppressMessage("StyleCop.CSharp.ReadabilityRules", "SA1121:UseBuiltInTypeAlias", Justification = "バイナリのサイズを明示するため")]
        public class Content : PacketBody
        {
            private UInt64 _timestamp = 0;
            private UInt32 _dataID = 0;
            private UInt32 _payloadLength = 0;
            private UInt32 _paddingLength = 0;
            private byte[] _payload = new byte[0];

            //-----------------------------------------------------------------

            public override uint Size
            {
                get
                {
                    return (uint)(
                        sizeof(UInt64) +
                        sizeof(UInt32) +
                        sizeof(UInt32) +
                        sizeof(UInt32) +
                        _payloadLength +
                        _paddingLength);
                }
            }

            /// <summary>
            /// パケットの送信時間です。
            /// 単位は[usec]です。
            /// </summary>
            public UInt64 Timestamp
            {
                get { return _timestamp; }
                set { _timestamp = value; }
            }

            public UInt32 DataID
            {
                get { return _dataID; }
                set { _dataID = value; }
            }

            public UInt32 PayloadLength
            {
                get { return _payloadLength; }
                set { _payloadLength = value; }
            }

            public UInt32 PaddingLength
            {
                get { return _paddingLength; }
                set { _paddingLength = value; }
            }

            public byte[] Payload
            {
                get { return _payload; }
                set
                {
                    if (_payload == value)
                    {
                        return;
                    }

                    _payload = value;
                    this.OnSizeChanged();
                }
            }

            protected override IEnumerable<PropertyInfo> GetProperties()
            {
                // リフレクションによる読み込みは遅いためこのメソッドは使われません。
                throw new NotImplementedException();
            }
        }
    }

    /// <summary>
    /// データファイルの読み出しを開始したことをアプリに通知します。
    /// </summary>
    internal class NotifyDataReadPacket : Packet<NotifyDataReadPacket.Content>
    {
        public NotifyDataReadPacket() :
            base(SpySession._Signature, (uint)PacketID.NotifyDataRead)
        { }

        public NotifyDataReadPacket(uint fileNo) :
            base(SpySession._Signature, (uint)PacketID.NotifyDataRead)
        {
            this.Body.FileNo = fileNo;
        }

        //-----------------------------------------------------------------

        [SuppressMessage("StyleCop.CSharp.ReadabilityRules", "SA1121:UseBuiltInTypeAlias", Justification = "バイナリ出力のサイズ情報を明示するため")]
        public class Content : PacketBody
        {
            public override uint Size
            {
                get
                {
                    return (uint)(sizeof(UInt32));
                }
            }

            /// <summary>
            /// 読み出しを開始したデータファイルの番号です(0, 1)。
            /// </summary>
            public UInt32 FileNo { get; set; }

            protected override IEnumerable<PropertyInfo> GetProperties()
            {
                yield return this.GetProperty(target => target.FileNo);
            }
        }
    }

    internal class NotifyDataReadReplyPacket : Packet<NullPacketBody>
    {
        public NotifyDataReadReplyPacket() :
            base(SpySession._Signature, (uint)PacketID.NotifyDataReadReply)
        { }
    }

    /// <summary>
    /// データファイルの終端を表します。
    /// </summary>
    internal class DataEndPacket : Packet<NullPacketBody>
    {
        public DataEndPacket() :
            base(SpySession._Signature, (uint)PacketID.DataEnd)
        { }
    }
}
