﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
//#define DebugState
using System;
using System.Collections.Generic;
using System.IO;
using System.Threading;
using System.Diagnostics;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.ToolFoundation.IO;
using NintendoWare.Spy.Foundation.Binary;
using NintendoWare.Spy.Foundation.Communications;
using System.Net.Sockets;
using System.Text;

namespace NintendoWare.Spy.Communication
{
    internal class SpySyncSession
    {
        private const int ConnectionTimeout = 5000;
        private const int PingTimeout = 5000;
        private const int SendTimeout = 5000;
        private const int ReceiveTimeout = 5000;

        //-----------------------------------------------------------------

        private class ComAction
        {
            public ComAction(Action<object> action, object param)
            {
                this.Action = action;
                this.Parameter = param;
            }

            public Action<object> Action { get; private set; }

            public object Parameter { get; private set; }

            public void Execute()
            {
                Assertion.Operation.NotNull(this.Action);
                this.Action(this.Parameter);
            }
        }

        public enum SessionState
        {
            NotStarted,
            Starting,
            Stopping,
            Running,
        }

        public class DataInfoReceivedEventArgs : EventArgs
        {
            public uint DataID { get; private set; }
            public string DataName { get; private set; }
            public Version DataVersion { get; private set; }

            public DataInfoReceivedEventArgs(QueryDataInfoReplyPacket packet)
            {
                this.DataID = packet.Body.DataID;
                this.DataName = packet.Body.DataName;
                this.DataVersion = packet.Body.DataVersion;
            }
        }

        //-----------------------------------------------------------------

        private readonly Queue<ComAction> _comActions = new Queue<ComAction>();
        private readonly object _comActionsLock = new object();
        private readonly object _hostIOLock = new object();

        private SessionState _state = SessionState.NotStarted;
        private uint[] _selectedSpyDataIDFlags = new uint[0];

        private DateTime _lastPingTime = DateTime.MinValue;

        private SynchronizationContext _syncContext;
        private IComEndPoint _hostIO;
        private IComChannel _channel;

        private bool _isLittleEndian = true;
        private volatile bool _isAvailable = false;
        private Thread _thread;
        /// <summary>
        /// <see cref="_isAvailable"/> , <see cref="_comActions"/> の状態が変わったときに <see cref="_thread"/> をスリープ状態から目覚めさせます。
        /// </summary>
        private readonly AutoResetEvent _awakeEvent = new AutoResetEvent(false);

        private ObjectBinaryReader _reader;
        private ObjectBinaryWriter _writer;
        private Stream _stream;
        private object _port;
        private string _workDirPath;

        //-----------------------------------------------------------------

        public event EventHandler StateChanged;

        public event EventHandler<DataInfoReceivedEventArgs> DataInfoReceived;

        //-----------------------------------------------------------------

        public SessionState State
        {
            get { return _state; }

            private set
            {
                if (_state == value)
                {
                    return;
                }

#if DebugState
                Debug.WriteLine($"[SyncStateChanged] {_state} --> {value} ({DateTime.Now})");
#endif

                _state = value;

                if (this.StateChanged != null)
                {
                    this.StateChanged(this, EventArgs.Empty);
                }
            }
        }

        public bool IsLittleEndian
        {
            get { return _isLittleEndian; }
        }

        /// <summary>
        /// 有効な SpyDataID フラグを取得または設定します。
        /// </summary>
        public uint[] SelectedSpyDataIDFlags
        {
            get { return _selectedSpyDataIDFlags; }

            set
            {
                _selectedSpyDataIDFlags = value;
                this.OnSelectedSpyDataIDFlagsUpdated();
            }
        }

        private bool IsConnected
        {
            get
            {
                lock (_hostIOLock)
                {
                    return _hostIO != null && _hostIO.IsConnected;
                }
            }
        }

        public Version TargetProtocolVersion { get; private set; }

        //-----------------------------------------------------------------

        public void Start(IComEndPoint hostIO, object port, SynchronizationContext syncContext, string workDirPath)
        {
            Ensure.Argument.NotNull(hostIO);
            Ensure.Argument.NotNull(syncContext);

            Assertion.Operation.AreEqual(this.State, SessionState.NotStarted);

            _syncContext = syncContext;
            _hostIO = hostIO;
            _workDirPath = workDirPath;
            _isLittleEndian = hostIO.IsLittleEndian;
            _isAvailable = true;
            this.TargetProtocolVersion = null;

            try
            {
                lock (_comActionsLock)
                {
                    _comActions.Clear();
                }

                _thread = new Thread(this.ThreadMain);
                _thread.Name = $"{nameof(SpySyncSession)}.{nameof(ThreadMain)}";
                _thread.Start(port);
            }
            catch
            {
                this.StopAsync();

                if (_thread != null)
                {
                    _thread.Join();
                    _thread = null;
                }

                _syncContext = null;
                _hostIO = null;
                _isAvailable = false;
                throw;
            }
        }

        public void StopAsync()
        {
            _isAvailable = false;
            _awakeEvent.Set();
        }

        private bool Connect(object port)
        {
            Ensure.Operation.True(this.State == SessionState.Starting);

            try
            {
                _channel = this.ConnectHostIO(port, SendTimeout, ReceiveTimeout);
                if (_channel == null)
                {
                    return false;
                }
            }
            catch
            {
                this.DisconnectHostIO();
                throw;
            }

            // 通信パフォーマンスを向上させるために、BufferedStream を利用
            _stream = new BufferedStream(_channel.GetStream());
            Ensure.Operation.NotNull(_stream);

            _port = port;

            if (this.IsLittleEndian)
            {
                _reader = new ObjectBinaryReader(LittleEndianBinaryReader.Create(_stream, Encoding.ASCII));
                _writer = new ObjectBinaryWriter(LittleEndianBinaryWriter.Create(_stream, Encoding.ASCII));
            }
            else
            {
                _reader = new ObjectBinaryReader(BigEndianBinaryReader.Create(_stream, Encoding.ASCII));
                _writer = new ObjectBinaryWriter(BigEndianBinaryWriter.Create(_stream, Encoding.ASCII));
            }

            return true;
        }

        private void Disconnect()
        {
            _reader = null;
            _writer = null;

            if (_stream != null)
            {
                try
                {
                    _stream.Close();
                }
                catch (IOException)
                {
                    // BufferedStream.Close() は Flush() を呼び出します。
                    // 相手側が通信を切断していると IOException が発生しますが無視します。
                }

                _stream = null;
            }

            this.DisconnectHostIO();
            _port = null;
        }

        private IComChannel ConnectHostIO(object channelID, int sendTimeout, int receiveTimeout)
        {
            lock (_hostIOLock)
            {
                _hostIO.Connect(_syncContext);
                if (!this.IsConnected)
                {
                    return null;
                }
            }

            var stopwatch = Stopwatch.StartNew();
            while (true)
            {
                if (!_isAvailable || stopwatch.ElapsedMilliseconds > ConnectionTimeout)
                {
                    return null;
                }

                IComChannel result = null;

                lock (_hostIOLock)
                {
                    if (!this.IsConnected)
                    {
                        return null;
                    }

                    result = _hostIO.OpenChannel(channelID);
                }

                if (result != null)
                {
                    result.SendTimeout = sendTimeout;
                    result.ReceiveTimeout = receiveTimeout;

                    return result;
                }

                Thread.Sleep(10);
            }
        }

        private void DisconnectHostIO()
        {
            if (_channel != null)
            {
                _channel.Close();
                _channel = null;
            }

            lock (_hostIOLock)
            {
                if (_hostIO != null)
                {
                    _hostIO.Disconnect();
                    _hostIO = null;
                }
            }

            _syncContext = null;
        }

        /// <summary>
        /// 同期パケット通信スレッドです。
        /// 状態変更はすべてこのスレッドで行います。
        /// </summary>
        /// <param name="port">ポートを指定します。</param>
        private void ThreadMain(object port)
        {
            this.State = SessionState.Starting;

            try
            {
                while (true)
                {
                    if (!_isAvailable)
                    {
                        this.State = SessionState.Stopping;
                        break;
                    }

                    try
                    {
                        // 接続処理
                        if (this.State == SessionState.Running)
                        {
                            // 接続済みなので何もせず流します。
                        }
                        else if (this.State == SessionState.Starting)
                        {
                            if (!this.Connect(port))
                            {
                                return;
                            }

                            this.TransferInitializeTarget();

                            // 有効なDataIDが返される間、問合せを繰り返します。
                            while (this.TransferQeuryDataInfo())
                            {
                            }

                            // 次の一行をコメントアウトするとHostFileIOによる
                            // データ通信を無効にできる。
                            this.TransferOutputDir(_workDirPath);

                            this.State = SessionState.Running;
                        }
                        else if (this.State == SessionState.NotStarted)
                        {
                            Assertion.Fail("invalid state.");
                            break;
                        }

                        while (true)
                        {
                            ComAction comAction = null;

                            lock (_comActionsLock)
                            {
                                if (_comActions.Count == 0)
                                {
                                    break;
                                }

                                comAction = _comActions.Dequeue();
                            }

                            Assertion.Operation.NotNull(comAction);
                            comAction.Execute();
                        }

                        // HACK : 3秒おきにPing（仮実装）
                        if (_lastPingTime + TimeSpan.FromSeconds(3) < DateTime.Now)
                        {
                            this.TransferPing();
                        }
                        else
                        {
                            _awakeEvent.WaitOne(1000);
                        }
                    }
                    catch (Exception ex)
                    {
                        if ((ex.InnerException as SocketException)?.SocketErrorCode != SocketError.ConnectionReset)
                        {
                            Debug.Write(ex.ToString());
                        }

                        // 接続中に例外が発生した場合は、停止します。
                        this.StopAsync();
                        break;
                    }
                }

                try
                {
                    this.TransferFinalizeTarget();
                }
                catch
                {
                }
            }
            finally
            {
                this.Disconnect();
                _thread = null;

                this.State = SessionState.NotStarted;
            }
        }

        private void TransferInitializeTarget()
        {
            // リクエストパケット
            var initializePacket = new InitializePacket();

            _writer.Write(initializePacket);
            _writer.Flush();

            // リプライパケット
            var replyPacket = new InitializeReplyPacket();
            _reader.Read(replyPacket);

            Ensure.Operation.True(replyPacket.Header.IsResultSuccess);

            this.TargetProtocolVersion = replyPacket.Body.ProtocolVersion ?? SpySession.ProtocolVersion_0_9_0_0;
        }

        private void TransferFinalizeTarget()
        {
            if (this.State == SessionState.Stopping && _writer != null)
            {
                // リクエストパケット
                _writer.Write(new FinalizePacket());
                _writer.Flush();

                // リプライパケット
                var replyPacket = new FinalizeReplyPacket();
                _reader.Read(replyPacket);
            }
        }

        private void TransferSelectDataID(uint[] selectionFlags)
        {
            // リクエストパケット
            var packet = new SelectDataIDPacket();

            packet.Body.SelectionFlags = selectionFlags;

            _writer.Write(packet);
            _writer.Flush();

            // リプライパケット
            var replyPacket = new SelectDataIDReplyPacket();
            _reader.Read(replyPacket);

            Ensure.Operation.True(replyPacket.Header.IsResultSuccess);
        }

        private void TransferOutputDir(string path)
        {
            // リクエストパケット
            var packet = new SetOutputDirPacket();

            packet.Body.SetPath(path);

            _writer.Write(packet);
            _writer.Flush();

            // リプライパケット
            var replyPacket = new SetOutputDirReplyPacket();
            _reader.Read(replyPacket);

            Ensure.Operation.True(replyPacket.Header.IsResultSuccess);
        }

        private void TransferPing()
        {
            Ensure.Operation.True(this.State == SessionState.Running);

            try
            {
                if (_writer != null)
                {
                    // リクエストパケット
                    _writer.Write(new PingPacket());
                    _writer.Flush();

                    using (var timer = new Timer(
                        state =>
                        {
                            Debug.WriteLine($"[SpySyncSession] Ping timeout ({DateTime.Now})");
                            this.StopAsync();
                        },
                        null,
                        PingTimeout,
                        Timeout.Infinite))
                    {
                        // リプライパケット
                        var replyPacket = new PongPacket();
                        _reader.Read(replyPacket);

                        _lastPingTime = DateTime.Now;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.WriteLine(ex.ToString());
                this.StopAsync();
            }
        }

        private void OnSelectedSpyDataIDFlagsUpdated()
        {
            lock (_comActionsLock)
            {
                _comActions.Enqueue(
                    new ComAction(
                        param => this.TransferSelectDataID((uint[])param),
                        this.SelectedSpyDataIDFlags));
            }
            _awakeEvent.Set();
        }

        public void TransferNotifyDataRead(uint fileNo)
        {
            lock (_comActionsLock)
            {
                _comActions.Enqueue(
                    new ComAction(
                        param =>
                        {
                            Ensure.Operation.True(this.State == SessionState.Running);

                            // リクエストパケット
                            var packet = new NotifyDataReadPacket(fileNo);
                            _writer.Write(packet);
                            _writer.Flush();

#if DebugState
                            Debug.WriteLine("[SpySynSession] NotifyDataReadPacket sent");
#endif

                            // リプライパケット
                            var replyPacket = new NotifyDataReadReplyPacket();
                            _reader.Read(replyPacket);

                            Ensure.Operation.True(replyPacket.Header.IsResultSuccess);
                        },
                        null));
            }
            _awakeEvent.Set();
        }

        private bool TransferQeuryDataInfo()
        {
            Ensure.Operation.True(this.State == SessionState.Starting);

            // リクエストパケット
            var packet = new QueryDataInfoPacket();

            _writer.Write(packet);
            _writer.Flush();

            // リプライパケット
            var replyPacket = new QueryDataInfoReplyPacket();
            _reader.Read(replyPacket);

            if (this.DataInfoReceived != null && replyPacket.Body.DataID != (uint)(SpyDataID.Invalid))
            {
                this.DataInfoReceived(this, new DataInfoReceivedEventArgs(replyPacket));
            }

            Ensure.Operation.True(replyPacket.Header.Result >= 0);

            return replyPacket.Body.DataID != (uint)(SpyDataID.Invalid);
        }
    }
}
