﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.ToolFoundation.Windows.Input;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Windows.Data;
using System.Windows.Input;
using System.Windows.Markup;

namespace NintendoWare.Spy.Windows
{
    [ContentProperty("Markers")]
    public class MarkerPanelViewModel : ObservableObject
    {
        private readonly object _observerOwner = new object();

        private readonly SpyService _spyService;
        private readonly SpyPlaybackService _playbackService;

        private readonly ICommand _checkAllCommand;
        private readonly ICommand _uncheckAllCommand;
        private readonly ICommand _clearFilterCommand;

        private MarkerSpyModel _markerModel;

        private readonly ObservableList<MarkerViewModel> _markers = new ObservableList<MarkerViewModel>();
        private MarkerViewModel _selectedMarker;

        private string _markerNameFilterText;

        private SpyTimeUnit _timeUnit;

        /// <summary>
        /// モックアップ用のコンストラクタです。
        /// </summary>
        internal MarkerPanelViewModel()
        {
        }

        public MarkerPanelViewModel(SpyService spyService, SpyPlaybackService playbackService)
        {
            Ensure.Argument.NotNull(spyService);
            Ensure.Argument.NotNull(playbackService);

            _spyService = spyService;
            _playbackService = playbackService;

            _checkAllCommand = new DelegateCommand(
                param => _markers.ForEach(item => item.IsVisible = true),
                param => _markerModel != null);

            _uncheckAllCommand = new DelegateCommand(
                param => _markers.ForEach(item => item.IsVisible = false),
                param => _markerModel != null);

            _clearFilterCommand = new DelegateCommand(
                param => this.MarkerNameFilterText = string.Empty,
                param => !string.IsNullOrEmpty(this.MarkerNameFilterText));
        }

        protected override void DisposeManagedInstance()
        {
            CollectionChangedObservation.RemoveObservers(_observerOwner);

            _markers.ForEach(it => it.Dispose());

            base.DisposeManagedInstance();
        }

        public ObservableCollection<MarkerViewModel> Markers
        {
            get { return _markers; }
        }

        public ICollectionView MarkersView
        {
            get { return CollectionViewSource.GetDefaultView(_markers); }
        }

        public MarkerViewModel SelectedMarker
        {
            get { return _selectedMarker; }
            set
            {
                if (this.SetPropertyValue(ref _selectedMarker, value))
                {
                    if (value != null)
                    {
                        _playbackService.SetCurrentTime(value.Time.Timestamp);
                    }
                }
            }
        }

        public string MarkerNameFilterText
        {
            get { return _markerNameFilterText; }
            set
            {
                if (this.SetPropertyValue(ref _markerNameFilterText, value))
                {
                    if (string.IsNullOrEmpty(_markerNameFilterText))
                    {
                        this.MarkersView.Filter = null;
                    }
                    else
                    {
                        var lowerFilterText = _markerNameFilterText.ToLower();
                        this.MarkersView.Filter = marker =>
                            {
                                var markerVM = (MarkerViewModel)marker;
                                return markerVM.Description.ToLower().Contains(lowerFilterText);
                            };
                    }
                }
            }
        }

        public SpyTimeUnit TimeUnit
        {
            get { return _timeUnit; }
            set { this.SetPropertyValue(ref _timeUnit, value); }
        }

        public void SetMarkerModel(MarkerSpyModel model)
        {
            if (_markerModel == model)
            {
                return;
            }

            CollectionChangedObservation.RemoveObservers(_observerOwner);

            _markerModel = model;
            this.SyncMarkers();
        }

        private void SyncMarkers()
        {
            if (_markerModel == null)
            {
                _markers.ForEach(it => it.Dispose());
                _markers.Clear();
                return;
            }

            CollectionChangedObservation.GetObserver(_observerOwner, _markerModel.Markers).AddHandlerForAddItems(
                (sender, e) =>
                {
                    this.NewMarkers(
                        e.Items.Cast<KeyValuePair<uint, MarkerSpyModel.Marker>>().Select(item => item.Value));
                });

            this.NewMarkers(_markerModel.Markers.Values);
        }

        private void NewMarkers(IEnumerable<MarkerSpyModel.Marker> markers)
        {
            foreach (var marker in markers)
            {
                // MarkerSpyModel.Marker.Values に要素が追加されたら、_markers に追加
                CollectionChangedObservation.GetObserver(_observerOwner, marker.Values).AddHandlerForAddItems(
                    (sender, e) =>
                    {
                        e.Items.Cast<MarkerSpyModel.MarkerValue>()
                            .ForEach(value => this.NewMarker(value));
                    });

                // 新規 Marker を _markers に追加
                marker.Values.ForEach(value => this.NewMarker(value));
            }
        }

        private void NewMarker(MarkerSpyModel.MarkerValue value)
        {
            _markers.Add(new MarkerViewModel(value));
        }
    }
}
