﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using NintendoWare.Spy.Framework.Settings;
using NintendoWare.Spy.Framework.Windows;
using NintendoWare.Spy.Settings;
using System;
using System.Collections.Generic;
using System.Diagnostics;

namespace NintendoWare.Spy.Windows
{
    /// <summary>
    /// グローバルタイムラインのプレゼンターです。
    /// </summary>
    internal sealed class GlobalTimelinePresenter : PanelPresenter
    {
        private readonly object _observerOwner = new object();
        private readonly object _settingsObserverOwner = new object();
        private GlobalTimelinePanel _view;
        private GlobalTimelineViewModel _viewModel;

        //-----------------------------------------------------------------

        /// <summary>
        /// 依存するサービスの型列挙子を取得します。
        /// </summary>
        public override IEnumerable<Type> DependentServiceTypes
        {
            get
            {
                yield return typeof(SpyService);
                yield return typeof(SpyPlaybackService);
                yield return typeof(SettingsService);
            }
        }

        //-----------------------------------------------------------------

        protected override void OnInitialize()
        {
            base.OnInitialize();

            this.GetSettingsService().Applied += this.OnSettingsApplied;
            this.OnSettingsApplied();
        }

        protected override void OnUninitialize()
        {
            this.GetSettingsService().Applied -= this.OnSettingsApplied;
            PropertyChangedObservation.RemoveObservers(_observerOwner);
            PropertyChangedObservation.RemoveObservers(_settingsObserverOwner);

            Disposer.SafeDispose(ref _viewModel);
            _view = null;

            base.OnUninitialize();
        }

        protected override System.Windows.Controls.Control CreateContent()
        {
            _viewModel = new GlobalTimelineViewModel();
            _viewModel.SetModel(this.GetPlaybackService());

            _view = new GlobalTimelinePanel()
            {
                DataContext = _viewModel,
            };

            PropertyChangedObservation.GetObserver(_observerOwner, _viewModel)
                .AddHandler(
                    target => target.TimeUnit,
                    (target, args) =>
                    {
                        var applicationSettings = this.GetSettingsService().GetApplicationSettings();
                        applicationSettings.TimeUnit = target.TimeUnit;
                    });

            return _view;
        }

        private void OnSettingsApplied(object sender, EventArgs args)
        {
            this.OnSettingsApplied();
        }

        private void OnSettingsApplied()
        {
            PropertyChangedObservation.RemoveObservers(_settingsObserverOwner);

            var applicationSettings = this.GetSettingsService().GetApplicationSettings();
            _viewModel.TimeUnit = applicationSettings.TimeUnit;

            PropertyChangedObservation.GetObserver(_settingsObserverOwner, applicationSettings)
                .AddHandler(
                    target => target.TimeUnit,
                    (target, args) => _viewModel.TimeUnit = target.TimeUnit);
        }

        [DebuggerStepThrough]
        private SpyService GetSpyService()
        {
            return this.ServiceProvider.GetService<SpyService>();
        }

        [DebuggerStepThrough]
        private SpyPlaybackService GetPlaybackService()
        {
            return this.ServiceProvider.GetService<SpyPlaybackService>();
        }

        [DebuggerStepThrough]
        private SettingsService GetSettingsService()
        {
            return this.ServiceProvider.GetService<SettingsService>();
        }
    }
}
