﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using System;
using System.Globalization;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Controls.Primitives;
using System.Windows.Data;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Shapes;

namespace NintendoWare.Spy.Windows
{
    public class GlobalTimelineBar : Control
    {
        public static readonly DependencyProperty MinimumProperty = DependencyProperty.Register(
            nameof(Minimum),
            typeof(double),
            typeof(GlobalTimelineBar),
            new FrameworkPropertyMetadata(double.NaN, (d, e) => Self(d).HandleMinimumChanged(e)));

        public static readonly DependencyProperty MaximumProperty = DependencyProperty.Register(
            nameof(Maximum),
            typeof(double),
            typeof(GlobalTimelineBar),
            new FrameworkPropertyMetadata(
                double.NaN,
                (d, e) => Self(d).HandleMaximumChanged(e),
                (d, v) => Self(d).CoerceMaximum(v)));

        public static readonly DependencyProperty CurrentValueProperty = DependencyProperty.Register(
            nameof(CurrentValue),
            typeof(double),
            typeof(GlobalTimelineBar),
            new FrameworkPropertyMetadata(
                double.NaN,
                FrameworkPropertyMetadataOptions.BindsTwoWayByDefault,
                (d, e) => Self(d).HandleCurrentValueChanged(e),
                (d, v) => Self(d).CoerceCurrentValue(v)));

        public static readonly DependencyProperty IsDraggingProperty = DependencyProperty.Register(
            nameof(IsDragging),
            typeof(bool),
            typeof(GlobalTimelineBar),
            new PropertyMetadata(false));

        public static readonly DependencyProperty TimeUnitProperty = DependencyProperty.Register(
            nameof(TimeUnit),
            typeof(SpyTimeUnit),
            typeof(GlobalTimelineBar),
            new FrameworkPropertyMetadata(SpyTimeUnit.Timestamp, FrameworkPropertyMetadataOptions.AffectsRender));

        public static readonly DependencyProperty MinimumSpanMajorTickProperty = DependencyProperty.Register(
            nameof(MinimumSpanMajorTick),
            typeof(double),
            typeof(GlobalTimelineBar),
            new FrameworkPropertyMetadata(100.0, (d, e) => Self(d).HandleMinimumSpanMajorTickChanged(e)));

        public static readonly DependencyProperty LabelConverterProperty = DependencyProperty.Register(
            nameof(LabelConverter),
            typeof(System.Windows.Data.IValueConverter),
            typeof(GlobalTimelineBar),
            new FrameworkPropertyMetadata((d, e) => Self(d).HandleLabelConverterChanged(e)));

        public static readonly DependencyProperty TickBrushProperty = DependencyProperty.Register(
            nameof(TickBrush),
            typeof(Brush),
            typeof(GlobalTimelineBar),
            new FrameworkPropertyMetadata(null, FrameworkPropertyMetadataOptions.AffectsRender));

        public static readonly DependencyProperty TickThicknessProperty = DependencyProperty.Register(
            nameof(TickThickness),
            typeof(double),
            typeof(GlobalTimelineBar),
            new FrameworkPropertyMetadata(1.0, FrameworkPropertyMetadataOptions.AffectsRender));

        public static readonly DependencyProperty MajorTickLengthProperty = DependencyProperty.Register(
            nameof(MajorTickLength),
            typeof(double),
            typeof(GlobalTimelineBar),
            new FrameworkPropertyMetadata(8.0, FrameworkPropertyMetadataOptions.AffectsRender));

        public static readonly DependencyProperty MinorTickLengthProperty = DependencyProperty.Register(
            nameof(MinorTickLength),
            typeof(double),
            typeof(GlobalTimelineBar),
            new FrameworkPropertyMetadata(4.0, FrameworkPropertyMetadataOptions.AffectsRender));

        public static readonly DependencyProperty MajorTickIntervalProperty = DependencyProperty.Register(
            nameof(MajorTickInterval),
            typeof(double),
            typeof(GlobalTimelineBar));

        public static readonly DependencyProperty MinorTickDivideProperty = DependencyProperty.Register(
            nameof(MinorTickDivide),
            typeof(int),
            typeof(GlobalTimelineBar));

        //-----------------------------------------------------------------

        static GlobalTimelineBar()
        {
            DefaultStyleKeyProperty.OverrideMetadata(
                typeof(GlobalTimelineBar),
                new FrameworkPropertyMetadata(typeof(GlobalTimelineBar)));
        }

        public GlobalTimelineBar()
        {
        }

        private bool _updateTick = true;
        private TextBlock _partLabelMinimum;
        private TextBlock _partLabelMaximum;
        private Popup _partPopupCurrent;
        private TextBlock _partLabelCurrent;
        private Line _partCurrentLine;

        //-----------------------------------------------------------------

        /// <summary>
        /// 目盛範囲の最小値（値の単位）
        /// </summary>
        public double Minimum
        {
            get { return (double)GetValue(MinimumProperty); }
            set { SetValue(MinimumProperty, value); }
        }

        /// <summary>
        /// 目盛の範囲の最大値（値の単位）
        /// </summary>
        public double Maximum
        {
            get { return (double)GetValue(MaximumProperty); }
            set { SetValue(MaximumProperty, value); }
        }

        /// <summary>
        /// カレント値（値の単位）
        /// </summary>
        public double CurrentValue
        {
            get { return (double)GetValue(CurrentValueProperty); }
            set { SetValue(CurrentValueProperty, value); }
        }

        /// <summary>
        /// マウスドラッグ中かを表します
        /// </summary>
        public bool IsDragging
        {
            get { return (bool)this.GetValue(IsDraggingProperty); }
            set { throw new InvalidOperationException("This property is read only."); }
        }

        /// <summary>
        /// 目盛の単位
        /// </summary>
        public SpyTimeUnit TimeUnit
        {
            get { return (SpyTimeUnit)GetValue(TimeUnitProperty); }
            set { SetValue(TimeUnitProperty, value); }
        }

        /// <summary>
        /// 主目盛間隔の最小値（デバイス・ピクセル）
        /// </summary>
        public double MinimumSpanMajorTick
        {
            get { return (double)GetValue(MinimumSpanMajorTickProperty); }
            set { SetValue(MinimumSpanMajorTickProperty, value); }
        }

        /// <summary>
        /// ラベル生成
        /// </summary>
        public System.Windows.Data.IValueConverter LabelConverter
        {
            get { return (System.Windows.Data.IValueConverter)GetValue(LabelConverterProperty); }
            set { SetValue(LabelConverterProperty, value); }
        }

        /// <summary>
        /// 目盛線の描画ブラシ
        /// </summary>
        public Brush TickBrush
        {
            get { return (Brush)GetValue(TickBrushProperty); }
            set { SetValue(TickBrushProperty, value); }
        }

        /// <summary>
        /// 目盛線の太さ
        /// </summary>
        public double TickThickness
        {
            get { return (double)GetValue(TickThicknessProperty); }
            set { SetValue(TickThicknessProperty, value); }
        }

        /// <summary>
        /// 主目盛の長さ
        /// </summary>
        public double MajorTickLength
        {
            get { return (double)GetValue(MajorTickLengthProperty); }
            set { SetValue(MajorTickLengthProperty, value); }
        }

        /// <summary>
        /// 補助目盛の長さ
        /// </summary>
        public double MinorTickLength
        {
            get { return (double)GetValue(MinorTickLengthProperty); }
            set { SetValue(MinorTickLengthProperty, value); }
        }

        /// <summary>
        /// 主目盛間隔（値の単位）
        /// </summary>
        public double MajorTickInterval
        {
            get { return (double)GetValue(MajorTickIntervalProperty); }
            private set { SetValue(MajorTickIntervalProperty, value); }
        }

        /// <summary>
        /// 補助目盛の分割数
        /// </summary>
        public int MinorTickDivide
        {
            get { return (int)GetValue(MinorTickDivideProperty); }
            private set { SetValue(MinorTickDivideProperty, value); }
        }

        //-----------------------------------------------------------------

        private void HandleMinimumChanged(DependencyPropertyChangedEventArgs e)
        {
            this.CoerceValue(MaximumProperty);
            this.CoerceValue(CurrentValueProperty);
            this.UpdateLabelMinimum();
            this.InvalidateTick();
        }

        private void HandleMaximumChanged(DependencyPropertyChangedEventArgs e)
        {
            this.CoerceValue(CurrentValueProperty);
            this.UpdateLabelMaximum();
            this.InvalidateTick();
        }

        private object CoerceMaximum(object v)
        {
            if (v is double)
            {
                double d = (double)v;

                if (!double.IsNaN(this.Minimum))
                {
                    d = Math.Max(d, this.Minimum);
                }

                return d;
            }
            else
            {
                return double.NaN;
            }
        }

        private void HandleCurrentValueChanged(DependencyPropertyChangedEventArgs e)
        {
            this.UpdateLabelCurrent();
            this.UpdateCurrentPosition();
            this.InvalidateVisual();
        }

        private object CoerceCurrentValue(object v)
        {
            if (v is double)
            {
                double d = (double)v;

                if (!double.IsNaN(this.Minimum))
                {
                    d = Math.Max(d, this.Minimum);
                }

                if (!double.IsNaN(this.Maximum))
                {
                    d = Math.Min(d, this.Maximum);
                }

                return d;
            }
            else
            {
                return double.NaN;
            }
        }

        private void HandleMinimumSpanMajorTickChanged(DependencyPropertyChangedEventArgs e)
        {
            this.InvalidateTick();
        }

        private void HandleLabelConverterChanged(DependencyPropertyChangedEventArgs e)
        {
            this.UpdateLabelMinimum();
            this.UpdateLabelMaximum();
            this.UpdateLabelCurrent();
            this.InvalidateVisual();
        }

        public override void OnApplyTemplate()
        {
            base.OnApplyTemplate();

            _partLabelMinimum = this.Template.FindName("PART_LabelMinimum", this) as TextBlock;
            _partLabelMaximum = this.Template.FindName("PART_LabelMaximum", this) as TextBlock;
            _partPopupCurrent = this.Template.FindName("PART_PopupCurrent", this) as Popup;
            _partLabelCurrent = this.Template.FindName("PART_LabelCurrent", this) as TextBlock;
            _partCurrentLine = this.Template.FindName("PART_CurrentLine", this) as Line;

            if (_partPopupCurrent != null)
            {
                _partPopupCurrent.SetBinding(Popup.IsOpenProperty, new Binding("IsDragging") { RelativeSource = RelativeSource.TemplatedParent });
            }
        }

        protected override Size MeasureOverride(Size constraint)
        {
            var requirement = base.MeasureOverride(constraint);

            var textHeight = Math.Ceiling(this.FontFamily.LineSpacing * this.FontSize);

            requirement.Height = Math.Max(requirement.Height, textHeight + Math.Max(this.MajorTickLength, this.MinorTickLength) + 2);

            return requirement;
        }

        protected override Size ArrangeOverride(Size arrangeBounds)
        {
            var size = base.ArrangeOverride(arrangeBounds);

            this.UpdateCurrentPosition();
            this.InvalidateTick();

            return size;
        }

        protected override void OnMouseLeftButtonDown(MouseButtonEventArgs e)
        {
            base.OnMouseLeftButtonDown(e);

            SetValue(IsDraggingProperty, true);
            Mouse.Capture(this);

            var position = e.GetPosition(this);
            this.CurrentValue = MathUtility.Clamp((this.Maximum - this.Minimum) / this.ActualWidth * position.X + this.Minimum, this.Minimum, this.Maximum);
        }

        protected override void OnMouseLeftButtonUp(MouseButtonEventArgs e)
        {
            base.OnMouseLeftButtonUp(e);

            this.ReleaseMouseCapture();
            SetValue(IsDraggingProperty, false);
        }

        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);

            if (this.IsDragging == true)
            {
                var position = e.GetPosition(this);

                this.CurrentValue = MathUtility.Clamp((this.Maximum - this.Minimum) / this.ActualWidth * position.X + this.Minimum, this.Minimum, this.Maximum);
            }
        }

        protected override void OnRender(DrawingContext drawingContext)
        {
            base.OnRender(drawingContext);

            if (ActualWidth > 0)
            {
                if (_updateTick)
                {
                    _updateTick = false;

                    UpdateTick();
                }

                drawingContext.DrawRectangle(this.Background, null, new Rect(0, 0, ActualWidth, ActualHeight));

                this.DrawTimelineBar(drawingContext);
            }
        }

        /// <summary>
        /// グローバルタイムラインバーを描画します。
        /// </summary>
        /// <param name="drawingContext"></param>
        private void DrawTimelineBar(DrawingContext drawingContext)
        {
            var majorTickInterval = this.MajorTickInterval;
            var minorTickDivide = this.MinorTickDivide;
            var minorTickInterval = majorTickInterval / minorTickDivide;
            var majorTickLength = this.MajorTickLength;
            var minorTickLength = this.MinorTickLength;
            var valueToPixel = this.ActualWidth / (this.Maximum - this.Minimum);
            var labelLimitLeft = 0.0;
            var labelLimitRight = this.ActualWidth;

            // ラベルの描画範囲を求めます。
            {
                if (_partLabelMinimum != null)
                {
                    var topRight = this.PointFromScreen(_partLabelMinimum.PointToScreen(new Point(_partLabelMinimum.ActualWidth, 0)));
                    labelLimitLeft = topRight.X;
                }
            }

            {
                if (_partLabelMaximum != null)
                {
                    var topLeft = this.PointFromScreen(_partLabelMaximum.PointToScreen(new Point(0, 0)));
                    labelLimitRight = topLeft.X;
                }
            }

            var brush = this.TickBrush;
            var pen = new Pen(brush, this.TickThickness);
            pen.Freeze();

            var typeface = new Typeface(this.FontFamily, this.FontStyle, this.FontWeight, this.FontStretch);

            drawingContext.PushClip(new RectangleGeometry(new Rect(new Point(0.0, 0.0), new Point(this.ActualWidth, this.ActualHeight))));

            for (double value = this.Rounding(this.Minimum, majorTickInterval) - majorTickInterval;
                value < this.Maximum + majorTickInterval;
                value += majorTickInterval)
            {
                // 主目盛の描画
                {
                    double x = (value - this.Minimum) * valueToPixel;
                    double y = this.ActualHeight - majorTickLength;

                    DrawTick(drawingContext, pen, x, y);

                    // 目盛ラベルの描画
                    {
                        var label = ConvertLabel(value);
                        if (!string.IsNullOrEmpty(label))
                        {
                            var text = new FormattedText(label, CultureInfo.CurrentCulture, FlowDirection.LeftToRight, typeface, this.FontSize, this.Foreground, null, TextFormattingMode.Display);
                            text.MaxTextWidth = majorTickInterval * valueToPixel;

                            var tx = x - text.Width / 2;

                            if (labelLimitLeft < tx && tx + text.Width < labelLimitRight)
                            {
                                drawingContext.DrawText(text, new Point(tx, 0));
                            }
                        }
                    }
                }

                // 補助目盛の描画
                for (int i = 1; i < minorTickDivide; ++i)
                {
                    double x = (value + i * minorTickInterval - this.Minimum) * valueToPixel;
                    double y = this.ActualHeight - minorTickLength;

                    DrawTick(drawingContext, pen, x, y);
                }
            }

            drawingContext.Pop();
        }

        /// <summary>
        /// 目盛線を引きます。
        /// </summary>
        /// <param name="drawingContext"></param>
        /// <param name="pen"></param>
        /// <param name="x"></param>
        /// <param name="y"></param>
        private void DrawTick(DrawingContext drawingContext, Pen pen, double x, double y)
        {
            var guideline = new GuidelineSet();
            guideline.GuidelinesX.Add(x - 0.5);
            guideline.GuidelinesX.Add(x + 0.5);

            drawingContext.PushGuidelineSet(guideline);
            drawingContext.DrawGeometry(null, pen, new LineGeometry(new Point(x, y), new Point(x, this.ActualHeight)));
            drawingContext.Pop();
        }

        /// <summary>
        /// 主目盛の間隔と補助目盛の分割数を求めます。
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        private Tuple<double, int> GetTickAndDivideCount(double value)
        {
            var baseValue = Math.Pow(10.0, Math.Floor(Math.Log10(value)));
            var relativeValue = value / baseValue;

            if (relativeValue < 2.0)
            {
                return new Tuple<double, int>(baseValue * 1.0, 2);
            }
            else if (relativeValue < 5.0)
            {
                return new Tuple<double, int>(baseValue * 2.0, 2);
            }
            else
            {
                return new Tuple<double, int>(baseValue * 5.0, 5);
            }
        }

        private string ConvertLabel(object value)
        {
            if (this.LabelConverter != null)
            {
                return (string)this.LabelConverter.Convert(value, typeof(string), parameter: null, culture: null);
            }
            else
            {
                return value.ToString();
            }
        }

        private void InvalidateTick()
        {
            _updateTick = true;
            this.InvalidateVisual();
        }

        private void UpdateLabelMinimum()
        {
            if (_partLabelMinimum != null)
            {
                _partLabelMinimum.Text = this.ConvertLabel(this.Minimum);
            }
        }

        private void UpdateLabelMaximum()
        {
            if (_partLabelMaximum != null)
            {
                _partLabelMaximum.Text = this.ConvertLabel(this.Maximum);
            }
        }

        private void UpdateLabelCurrent()
        {
            if (_partLabelCurrent != null)
            {
                _partLabelCurrent.Text = this.ConvertLabel(this.CurrentValue);
            }
        }

        private void UpdateCurrentPosition()
        {
            if (this.ActualWidth > 0 && this.Minimum < this.Maximum)
            {
                var currentPosition = (this.CurrentValue - this.Minimum) / (this.Maximum - this.Minimum) * this.ActualWidth;

                if (_partCurrentLine != null)
                {
                    _partCurrentLine.X1 = currentPosition;
                    _partCurrentLine.Y1 = 0;
                    _partCurrentLine.X2 = currentPosition;
                    _partCurrentLine.Y2 = this.ActualHeight;
                }

                if (_partPopupCurrent != null && _partPopupCurrent.Child != null)
                {
                    _partPopupCurrent.HorizontalOffset = currentPosition - _partPopupCurrent.Child.DesiredSize.Width / 2;
                    _partPopupCurrent.VerticalOffset = -_partPopupCurrent.Child.DesiredSize.Height;
                }
            }
        }

        private void UpdateTick()
        {
            var value = this.GetTickAndDivideCount(this.MinimumSpanMajorTick * (this.Maximum - this.Minimum) / this.ActualWidth);
            var majorTick = value.Item1;
            var minorTickDivide = value.Item2;

            if (majorTick != this.MajorTickInterval)
            {
                this.MajorTickInterval = majorTick;
            }

            if (minorTickDivide != this.MinorTickDivide)
            {
                this.MinorTickDivide = minorTickDivide;
            }
        }

        private double Rounding(double target, double align)
        {
            return target - (target % align);
        }

        private static GlobalTimelineBar Self(DependencyObject o)
        {
            return (GlobalTimelineBar)o;
        }
    }
}
