﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Windows;

namespace NintendoWare.NwSoundSpyPlugin.Windows
{
    internal class TimelineUtil
    {
    }

    /// <summary>
    /// テキストボックスが空の時にデフォルトテキストを書ビヘイビア
    /// </summary>
    public static class PlaceHolderBehaivor
    {
        public static readonly DependencyProperty PlaceHolderTextProperty = DependencyProperty.RegisterAttached(
            "PlaceHolderText",
            typeof(string),
            typeof(PlaceHolderBehaivor),
            new PropertyMetadata(null, OnPlaceHolderChanged));

        public static void SetPlaceHolderText(DependencyObject obj, string placeHolder)
        {
            obj.SetValue(PlaceHolderTextProperty, placeHolder);
        }

        public static string GetPlaceHolderText(DependencyObject obj)
        {
            return (string)obj.GetValue(PlaceHolderTextProperty);
        }

        private static void OnPlaceHolderChanged(DependencyObject sender, DependencyPropertyChangedEventArgs e)
        {
            var textBox = sender as System.Windows.Controls.TextBox;
            if (textBox == null)
            {
                return;
            }

            var placeHolder = (string)e.NewValue;
            var handler = CreateEventHandler(placeHolder);

            if (string.IsNullOrEmpty(placeHolder))
            {
                textBox.TextChanged -= handler;
            }
            else
            {
                textBox.TextChanged += handler;
                if (string.IsNullOrEmpty(textBox.Text))
                {
                    textBox.Background = CreateVisualBrush(placeHolder);
                }
            }
        }

        private static System.Windows.Controls.TextChangedEventHandler CreateEventHandler(string placeHolder)
        {
            return (sender, e) =>
            {
                var textBox = (System.Windows.Controls.TextBox)sender;
                if (string.IsNullOrEmpty(textBox.Text))
                {
                    textBox.Background = CreateVisualBrush(placeHolder);
                }
                else
                {
                    textBox.Background = new System.Windows.Media.SolidColorBrush(System.Windows.Media.Colors.Transparent);
                }
            };
        }

        private static System.Windows.Media.VisualBrush CreateVisualBrush(string placeHolder)
        {
            var visual = new System.Windows.Controls.Label()
            {
                Content = placeHolder,
                Padding = new Thickness(5, 1, 1, 1),
                Foreground = new System.Windows.Media.SolidColorBrush(System.Windows.Media.Colors.LightGray),
                HorizontalAlignment = HorizontalAlignment.Left,
                VerticalAlignment = VerticalAlignment.Center,
            };

            var visualBrush = new System.Windows.Media.VisualBrush(visual)
            {
                Stretch = System.Windows.Media.Stretch.None,
                TileMode = System.Windows.Media.TileMode.None,
                AlignmentX = System.Windows.Media.AlignmentX.Left,
                AlignmentY = System.Windows.Media.AlignmentY.Center,
            };

            return visualBrush;
        }
    }

    /// <summary>
    /// 描画関連のユーティリティ
    /// </summary>
    public class TimelineViewUtil
    {
        #region プロパティとか

        /// <summary>
        /// カレントフレームが変わっても表示位置を維持する左限界を指定します。
        /// キャンバス幅を１としたときの位置です。
        /// </summary>
        public static readonly double ScrollMarginLeft = 0;

        /// <summary>
        /// カレントフレームが変わっても表示位置を維持する右限界を指定します。
        /// キャンバス幅を１としたときの位置です。
        /// </summary>
        public static readonly double ScrollMarginRight = 1;

        private static ZoomRatio Scale = new ZoomRatio(0);
        private static ZoomGrid GridParams = new ZoomGrid();

        public delegate void ViewChangedEventHandler();
        public static event ViewChangedEventHandler ZoomChanged = null;
        public static event ViewChangedEventHandler PanningChanged = null;

        /// <summary>
        /// キャンバスの左端のフレーム
        /// </summary>
        public static long ViewStartFrame { get; private set; }

        public static long ViewEndFrame { get; private set; }

        public static long TotalStartFrame { get; set; }

        public static long TotalEndFrame { get; set; }

        /// <summary>
        /// 現状のズーム比のグリッドのピクセル幅
        /// </summary>
        public static double GridWidth { get { return GridParams.Width; } }

        /// <summary>
        /// 現状のズーム比のグリッドのフレーム数
        /// </summary>
        public static long GridFrame { get { return GridParams.Frame; } }

        /// <summary>
        /// 縦方向のビューオフセット
        /// </summary>
        public static double ViewVerticalOffset { get; private set; }

        /// <summary>
        /// ビューの高さ
        /// </summary>
        public static double ViewHeight { get; set; }

        /// <summary>
        /// 表示内容の高さ
        /// </summary>
        public static double ContentHeight { get; set; }

        #endregion

        static TimelineViewUtil()
        {
            ContentHeight = double.MaxValue;
        }

        /// <summary>
        /// 指定したフレームの描画幅
        /// </summary>
        /// <param name="frame"></param>
        /// <returns></returns>
        public static double GetRenderWidth(long frame)
        {
            return Scale.CalcWidthFromPointCount(frame);
        }

        /// <summary>
        /// 指定したピクセル幅のフレーム数
        /// </summary>
        /// <param name="width"></param>
        /// <returns></returns>
        public static long GetFrame(double width)
        {
            return (long)(Scale.CalcPointCountFromWidth(width));
        }

        /// <summary>
        /// 指定したフレームのキャンバス上の描画位置
        /// </summary>
        /// <param name="frame"></param>
        /// <returns></returns>
        public static double GetRenderPos(long frame)
        {
            return GetRenderWidth(frame - ViewStartFrame);
        }

        /// <summary>
        /// キャンバスに引かれる目盛線の開始フレーム値
        /// </summary>
        /// <returns></returns>
        public static long GetViewStartGrid()
        {
            return ViewStartFrame - ViewStartFrame % GridFrame;
        }

        /// <summary>
        /// キャンバス上の座標位置におけるフレーム値
        /// </summary>
        /// <param name="canvasPos"></param>
        /// <returns></returns>
        public static long GetViewEndGrid(double canvasPos)
        {
            return ViewStartFrame + GetFrame(canvasPos);
        }

        /// <summary>
        /// キャンバスのサイズが変わった時に呼んでもらいたい
        /// </summary>
        /// <param name="canvasWidth"></param>
        public static void SetViewEndGrid(double canvasWidth)
        {
            ViewEndFrame = GetViewEndGrid(canvasWidth);
        }

        public static void SetViewStartFrameForAutoScroll(long frame, double canvasWidth)
        {
            // 境界上のカレントラインがはっきりと見えるようにするため。
            long MARGIN = GetFrame(1);

            var frameWidth = GetFrame(canvasWidth);
            var leftMarginFrame = (long)(frameWidth * ScrollMarginLeft);
            var rightMarginFrame = (long)(frameWidth * ScrollMarginRight);

            var oldStartFrame = ViewStartFrame;
            if (frame <= ViewStartFrame + leftMarginFrame + MARGIN)
            {
                ViewStartFrame = frame - leftMarginFrame - MARGIN;
            }
            else if (ViewStartFrame + rightMarginFrame < frame + MARGIN)
            {
                ViewStartFrame = frame - rightMarginFrame + MARGIN;
            }
            else
            {
                // マージン内なら表示位置はそのまま。
            }

            // 現在の表示位置は維持できるようにします。
            var leftLimit = Math.Min(oldStartFrame, TotalStartFrame - MARGIN);
            var rightLimit = Math.Max(oldStartFrame, TotalEndFrame - frameWidth + MARGIN);

            ViewStartFrame = MathUtil.Clamp(ViewStartFrame, leftLimit, rightLimit);

            var oldEndFrame = ViewEndFrame;
            SetViewEndGrid(canvasWidth);

            if ((oldStartFrame != ViewStartFrame || oldEndFrame != ViewEndFrame) && PanningChanged != null)
            {
                PanningChanged();
            }
        }

        /// <summary>
        /// 時間の表示単位を切り替えます。
        /// 引数には新しい時間の表示単位での各フレーム値を渡します。
        /// </summary>
        /// <param name="newTotalStartFrame"></param>
        /// <param name="newTotalEndFrame"></param>
        /// <param name="newViewStartFrame"></param>
        /// <param name="newViewEndFrame"></param>
        /// <param name="canvasWidth"></param>
        public static void ChangeTimeUnit(
            long newTotalStartFrame,
            long newTotalEndFrame,
            long newViewStartFrame,
            long newViewEndFrame,
            double canvasWidth)
        {
            // 新しいフレーム範囲に変更します。
            TotalStartFrame = newTotalStartFrame;
            TotalEndFrame = newTotalEndFrame;
            ViewStartFrame = newViewStartFrame;
            SetViewEndGrid(canvasWidth);

            // ViewEndFrame がなるべくぴったりと収まるようにズームレベルを調整します。
            if (ViewEndFrame < newViewEndFrame)
            {
                // newViewEndFrame がキャンバスの外にでているので、キャンバスに収まるまでズームアウトします。
                while (Scale.Index > ZoomScaleMin)
                {
                    TimelineViewUtil.ZoomOut(0, canvasWidth);
                    if (ViewEndFrame >= newViewEndFrame)
                    {
                        break;
                    }
                }
            }
            else if (ViewEndFrame > newViewEndFrame)
            {
                // なるべく newViewEndFrame がキャンバスの右端にくるようにズームインします。
                while (Scale.Index < ZoomScaleMax)
                {
                    TimelineViewUtil.ZoomIn(0, canvasWidth);
                    if (ViewEndFrame < newViewEndFrame)
                    {
                        TimelineViewUtil.ZoomOut(0, canvasWidth);
                        break;
                    }
                }
            }
        }

        #region Zoom

        private static readonly int ZoomScaleMax = 10;
        private static readonly int ZoomScaleMin = -40;

        public static void ZoomInitialize()
        {
            ZoomChange(-18, offsetPos: 0, canvasWidth: 0);
        }

        /// <summary>
        /// ズームイン
        /// </summary>
        /// <param name="offsetPos"></param>
        public static void ZoomIn(double offsetPos, double canvasWidth)
        {
            ZoomChange(1, offsetPos, canvasWidth);
        }

        /// <summary>
        /// ズームアウト
        /// </summary>
        /// <param name="offsetPos"></param>
        public static void ZoomOut(double offsetPos, double canvasWidth)
        {
            ZoomChange(-1, offsetPos, canvasWidth);
        }

        /// <summary>
        /// ズーム変更
        /// </summary>
        /// <param name="value"></param>
        /// <param name="offsetPos"></param>
        private static void ZoomChange(int value, double offsetPos, double canvasWidth)
        {
            var newScaleIndex = Scale.Index + value;
            if (ZoomScaleMin <= newScaleIndex && newScaleIndex <= ZoomScaleMax)
            {
                // 現在のズーム比のオフセット
                long offsetFrame = (long)(Scale.CalcPointCountFromWidth(offsetPos)) + ViewStartFrame;

                // グリッドの幅計算
                Scale.Index += value;
                GridParams.CalcGrid(Scale);

                // オフセット位置計算
                ViewStartFrame = offsetFrame - (long)(Scale.CalcPointCountFromWidth(offsetPos));
                if (ViewStartFrame < 0)
                {
                    ViewStartFrame = 0;
                }

                ViewEndFrame = GetViewEndGrid(canvasWidth);

                if (ZoomChanged != null)
                {
                    ZoomChanged();
                }
            }
        }

        /// <summary>
        /// ズーム比計算クラス
        /// </summary>
        private class ZoomRatio
        {
            private int _zoomIndex = 0;
            private int _zoomValue = 1;

            public ZoomRatio(int index)
            {
                Index = index;
            }

            public int Index
            {
                get { return _zoomIndex; }
                set
                {
                    _zoomIndex = value;
                    _zoomValue = CalcZoomValue(_zoomIndex);
                }
            }

            public double CalcPointCountFromWidth(double width)
            {
                if (_zoomValue > 0)
                {
                    return width / (double)_zoomValue;
                }
                else
                {
                    return width * (double)(-_zoomValue);
                }
            }

            public double CalcWidthFromPointCount(double pointCount)
            {
                if (_zoomValue > 0)
                {
                    return pointCount * (double)_zoomValue;
                }
                else
                {
                    return pointCount / (double)(-_zoomValue);
                }
            }

            // -4 => 1:6
            // -3 => 1:4
            // -2 => 1:3
            // -1 => 1:2
            //  0 => 1:1
            //  1 => 2:1
            //  2 => 3:1
            //  3 => 4:1
            //  4 => 6:1
            //  5 => 8:1
            //  6 => 12:1
            private int CalcZoomValue(int zoomIndex)
            {
                if (zoomIndex == 0)
                {
                    return 1;
                }

                int index = Math.Abs(zoomIndex);

                int value;
                int n;
                if ((index & 0x1) != 0)
                {
                    // 奇数の時は、2 * 2^n
                    n = (index - 1) / 2;
                    value = 2;
                }
                else
                {
                    // 偶数の時は、3 * 2^n
                    n = (index - 2) / 2;
                    value = 3;
                }
                value <<= n;

                return (zoomIndex >= 0) ? value : -value;
            }
        }

        /// <summary>
        /// ズーム比における目盛幅計算クラス
        /// </summary>
        private class ZoomGrid
        {
            private static readonly long FirstGridFrame = 5;
            private static readonly double GridWidthMin = 80;
            private static readonly double[] ScaleParams = { 2, 2, 2.5 }; // 1, 2, 5

            public double Width { get; private set; }
            public long Frame { get; private set; }

            public ZoomGrid()
            {
                Width = 100; // ZoomRatio = 0
                Frame = 100; // 5 -> 10 -> 20 -> 50 -> 100
            }

            public void CalcGrid(ZoomRatio zoomRatio)
            {
                double tmpFrame = FirstGridFrame;
                double tmpWidth = zoomRatio.CalcWidthFromPointCount(tmpFrame);
                int scaleParamPos = 0;

                while (tmpWidth < GridWidthMin)
                {
                    tmpFrame *= ScaleParams[scaleParamPos];
                    tmpWidth = zoomRatio.CalcWidthFromPointCount(tmpFrame);
                    scaleParamPos = (scaleParamPos + 1) % ScaleParams.Length;
                }

                Frame = (long)(tmpFrame);
                Width = tmpWidth;
            }
        }

        #endregion

        #region Panning

        public enum PanningDirection
        {
            Horizontal,
            Vertical,
            Free,
        }

        /// <summary>
        /// 描画オフセットの計算
        /// </summary>
        /// <param name="target"></param>
        /// <param name="source"></param>
        /// <param name="dir"></param>
        public static void PanningView(Point target, Point source, PanningDirection dir)
        {
            PanningView(new Point(ViewStartFrame, ViewVerticalOffset), target, source, dir);
        }

        /// <summary>
        /// 描画オフセットの計算
        /// </summary>
        /// <param name="viewStartFrame"></param>
        /// <param name="target"></param>
        /// <param name="source"></param>
        /// <param name="dir"></param>
        public static void PanningView(Point startView, Point target, Point source, PanningDirection dir)
        {
            // ドラッグ操作をしたときにいきなり位置が変わることを防ぐために、
            // 現在位置はそのまま維持できる計算式にする必要があります。

            bool bPanningChanged = false;
            if (dir == PanningDirection.Horizontal || dir == PanningDirection.Free)
            {
                // 境界上のカレントラインがはっきりと見えるようにするため。
                var MARGIN = GetFrame(1);

                double diffX = target.X - source.X;

                var movedFrame = GetFrame(-diffX);

                long leftLimit, rightLimit;
                var viewWidth = ViewEndFrame - ViewStartFrame;
                var contentWidth = TotalEndFrame - TotalStartFrame;
                if (viewWidth < contentWidth)
                {
                    leftLimit = TotalStartFrame;
                    rightLimit = TotalEndFrame - viewWidth;
                }
                else
                {
                    leftLimit = TotalEndFrame - viewWidth;
                    rightLimit = TotalStartFrame;
                }

                // 現在の表示位置は維持できるようにします。
                leftLimit = Math.Min(ViewStartFrame, leftLimit - MARGIN);
                rightLimit = Math.Max(ViewStartFrame, rightLimit + MARGIN);

                var newStartFrame = MathUtil.Clamp((long)(startView.X + movedFrame), leftLimit, rightLimit);
                if (ViewStartFrame != newStartFrame)
                {
                    ViewStartFrame = newStartFrame;
                    ViewEndFrame = newStartFrame + viewWidth;
                    bPanningChanged = true;
                }
            }

            if (dir == PanningDirection.Vertical || dir == PanningDirection.Free)
            {
                // ビューに対してコンテンツを上に動かすのでオフセットは負の値になります。
                double diffY = target.Y - source.Y;

                double lowerLimit, upperLimit;
                if (ContentHeight <= ViewHeight)
                {
                    lowerLimit = Math.Min(ViewVerticalOffset, 0);
                    upperLimit = Math.Max(ViewVerticalOffset, 0);
                }
                else
                {
                    lowerLimit = Math.Min(ViewVerticalOffset, ViewHeight - ContentHeight);
                    upperLimit = Math.Max(ViewVerticalOffset, 0);
                }

                var newViewVerticalOffset = MathUtil.Clamp(startView.Y + diffY, lowerLimit, upperLimit);

                if (newViewVerticalOffset != ViewVerticalOffset)
                {
                    ViewVerticalOffset = newViewVerticalOffset;
                    bPanningChanged = true;
                }
            }

            if (bPanningChanged && PanningChanged != null)
            {
                PanningChanged();
            }
        }

        #endregion
    }
}
