﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using NintendoWare.Spy;
using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Shapes;
using System.Windows.Threading;

namespace NintendoWare.NwSoundSpyPlugin.Windows
{
    /// <summary>
    /// TimelineControl.xaml の相互作用ロジック
    /// </summary>
    public partial class TimelinePanel : UserControl
    {
        public static readonly DependencyProperty StartTimeProperty = DependencyProperty.Register(
            nameof(StartTime),
            typeof(SpyTime),
            typeof(TimelinePanel),
            new PropertyMetadata(SpyTime.Zero, (d, e) => ((TimelinePanel)d).HandleStartTimeChanged()));

        public SpyTime StartTime
        {
            get { return (SpyTime)this.GetValue(StartTimeProperty); }
            set { this.SetValue(StartTimeProperty, value); }
        }

        public static readonly DependencyProperty EndTimeProperty = DependencyProperty.Register(
            nameof(EndTime),
            typeof(SpyTime),
            typeof(TimelinePanel),
            new PropertyMetadata(SpyTime.Zero, (d, e) => ((TimelinePanel)d).HandleEndTimeChanged()));

        public SpyTime EndTime
        {
            get { return (SpyTime)this.GetValue(EndTimeProperty); }
            set { this.SetValue(EndTimeProperty, value); }
        }

        public event EventHandler<DragMoveEventArgs> DragCurrentFrame;

        private enum ScrollDirType
        {
            None = 0,
            Vertical,
            Horizontal,
            Free
        }

        private enum DragMode
        {
            None = 0,
            Panning,
            Scrolling,
        }

        private DragMode _dragMode = DragMode.None;
        private ScrollDirType _scrollDir = ScrollDirType.None;
        private Point _dragStartView;
        private Point _dragStartPos;
        private Point _dragUpdatePos;
        private double _scrollPos = 0;
        private readonly DispatcherTimer _scrollTimer = new DispatcherTimer();

        private readonly List<TimelineGrid> _grids = new List<TimelineGrid>();
        private readonly RectangleGeometry _clipRect = new RectangleGeometry(new Rect());

        private TimelinePanelPresenter _presenter = null;
        private IList<TimelineItemViewModel> _vmItems = null;
        private TimelineItem _selectedItemView = null;

        private readonly List<TimelineGroup> _groups = new List<TimelineGroup>();

        private TimelineCurrentLine _currentLine = null;
        private long _currentFrame = 0;
        private long _startFrame = 0;
        private long _endFrame = 0;
        private SpyTimeUnit _timeUnit = SpyTimeUnit.Timestamp;

        /// <summary>
        /// カレントフレーム
        /// </summary>
        public long CurrentFrame
        {
            get { return _currentFrame; }
            private set
            {
                if (_currentFrame != value)
                {
                    _currentFrame = value;

                    _currentLine.UpdateCurrentFrame(value);
                }
            }
        }

        /// <summary>
        /// 開始フレーム
        /// </summary>
        public long StartFrame
        {
            get { return _startFrame; }
            private set
            {
                if (_startFrame != value)
                {
                    _startFrame = value;

                    TimelineViewUtil.TotalStartFrame = this.StartFrame;
                }
            }
        }

        /// <summary>
        /// 終了フレーム
        /// </summary>
        public long EndFrame
        {
            get { return _endFrame; }
            private set
            {
                if (_endFrame != value)
                {
                    _endFrame = value;

                    TimelineViewUtil.TotalEndFrame = this.EndFrame;
                }
            }
        }

        /// <summary>
        /// 時間表示単位
        /// </summary>
        public SpyTimeUnit TimeUnit
        {
            get { return _timeUnit; }
            private set
            {
                if (_timeUnit != value)
                {
                    _timeUnit = value;
                }
            }
        }

        public void UpdateCurrentPos(long frame, bool isAutoScrollEnabled)
        {
            IsAutoScroll = isAutoScrollEnabled;
            if (this.CurrentFrame != frame)
            {
                this.CurrentFrame = frame;
                if (_dragMode == DragMode.None)
                {
                    TimelineViewUtil.SetViewStartFrameForAutoScroll(frame, this.TimelineControlCanvas.ActualWidth);
                }
            }
        }

        /// <summary>
        /// 時間表示単位を変更します。
        /// </summary>
        /// <param name="timeUnit"></param>
        /// <param name="frameSync"></param>
        public void ChangeTimeUnit(SpyTimeUnit timeUnit, FrameSyncSpyModel frameSync)
        {
            if (frameSync != null)
            {
                // 現在のフレームに対する時刻を求めます。
                SpyTime viewStartTime;
                if (!TryFindTimeOfFrame(TimelineViewUtil.ViewStartFrame, this.TimeUnit, frameSync, out viewStartTime))
                {
                    viewStartTime = this.StartTime;
                }

                SpyTime viewEndTime;
                if (!TryFindTimeOfFrame(TimelineViewUtil.ViewEndFrame, this.TimeUnit, frameSync, out viewEndTime))
                {
                    viewEndTime = this.EndTime;
                }

                SpyTime currentTime;
                if (!TryFindTimeOfFrame(this.CurrentFrame, this.TimeUnit, frameSync, out currentTime))
                {
                    currentTime = (this.StartTime.Timestamp > viewStartTime.Timestamp) ? this.StartTime : viewStartTime;
                }

                // 新しいフレームを求めます。
                var newTotalStartFrame = this.StartTime.SelectFrameValue(timeUnit);
                var newTotalEndFrame = this.EndTime.SelectFrameValue(timeUnit);
                var newViewStartFrame = viewStartTime.SelectFrameValue(timeUnit);
                var newViewEndFrame = viewEndTime.SelectFrameValue(timeUnit);

                // TimelineViewUtil に個別に反映されないようにするためフィールドを変更します。
                _startFrame = newTotalStartFrame;
                _endFrame = newTotalEndFrame;

                // TimelieViewUtil に新しいフレームをまとめて反映します。
                TimelineViewUtil.ChangeTimeUnit(
                    newTotalStartFrame,
                    newTotalEndFrame,
                    newViewStartFrame,
                    newViewEndFrame,
                    TimelineEventCanvas.ActualWidth);

                // カレントフレームが以前と同じ時刻を指すように設定し直します。
                this.CurrentFrame = currentTime.SelectFrameValue(timeUnit);
            }

            this.TimeUnit = timeUnit;

            foreach (var grid in _grids)
            {
                grid.TimeUnit = this.TimeUnit;
            }

            foreach (var group in _groups)
            {
                group.TimeUnit = this.TimeUnit;
            }

            UpdateGrids();
            foreach (var group in _groups)
            {
                group.UpdateView();
            }
        }

        private bool TryFindTimeOfFrame(long frame, SpyTimeUnit timeUnit, FrameSyncSpyModel frameSync, out SpyTime time)
        {
            switch (timeUnit)
            {
                case SpyTimeUnit.AppFrame:
                    if (frameSync.AppFrames.TryFindValueOf(new Spy.Frame(frame), out time))
                    {
                        return true;
                    }
                    break;

                case SpyTimeUnit.AudioFrame:
                    if (frameSync.AudioFrames.TryFindValueOf(new Spy.Frame(frame), out time))
                    {
                        return true;
                    }
                    break;

                case SpyTimeUnit.Timestamp:
                case SpyTimeUnit.TimestampUsec:
                    if (frameSync.AllFrames.TryFindValueOf(SpyGlobalTime.FromMicroSeconds(frame), out time))
                    {
                        return true;
                    }
                    break;
            }

            time = null;
            return false;
        }

        private void HandleStartTimeChanged()
        {
            this.StartFrame = this.StartTime.SelectFrameValue(this.TimeUnit);
        }

        private void HandleEndTimeChanged()
        {
            this.EndFrame = this.EndTime.SelectFrameValue(this.TimeUnit);
        }

        /// <summary>
        /// 自動スクロールするか
        /// </summary>
        public bool IsAutoScroll { get; set; }

        public TimelinePanel()
        {
            this.AddHandler(TimelineItem.IsSelectedChangedEvent, new RoutedEventHandler(OnTimelineItemIsSelectedChanged));

            this.SetBinding(StartTimeProperty, "StartTime");
            this.SetBinding(EndTimeProperty, "EndTime");

            InitializeComponent();

            IsAutoScroll = false;

            _currentLine = new TimelineCurrentLine(TimelineControlCanvas);

            TimelineControlCanvas.Clip = _clipRect;
            Canvas.SetZIndex(TimelineEventCanvas, 1);

            TimelineViewUtil.ZoomInitialize();

            SizeChanged += (sender, e) =>
            {
                System.Windows.Rect rect = _clipRect.Rect;
                rect.Size = TimelineControlCanvas.RenderSize;
                _clipRect.Rect = rect;
                TimelineViewUtil.SetViewEndGrid(rect.Width);
                UpdateGrids();
                TimelineViewUtil.ViewHeight = rect.Height;
            };
            TimelineViewUtil.ZoomChanged += () => UpdateGrids();
            TimelineViewUtil.PanningChanged += () =>
            {
                UpdateGrids();
                UpdateGroupPos(this, null);
            };

            Focusable = true;

            _scrollTimer.Interval = new TimeSpan(0, 0, 0, 0, 20); // 20msec
            _scrollTimer.Tick += ScrollOnTimer;
        }

        public void SetPresenter(TimelinePanelPresenter presenter)
        {
            _presenter = presenter;
        }

        public void SetViewModel(IList<TimelineItemViewModel> viewModelItems)
        {
            if (_vmItems != viewModelItems)
            {
                this.SelectItem(null);

                if (_vmItems != null)
                {
                    ((INotifyCollectionChanged)_vmItems).CollectionChanged -= UpdateItems;
                }

                _vmItems = viewModelItems;

                ((INotifyCollectionChanged)_vmItems).CollectionChanged += UpdateItems;
            }
        }

        private void SelectItem(TimelineItem itemView)
        {
            if (_selectedItemView != itemView)
            {
                if (_selectedItemView != null)
                {
                    _selectedItemView.IsSelected = false;
                }

                if (itemView != null)
                {
                    itemView.IsSelected = true;
                }

                _selectedItemView = itemView;
            }
        }

        #region スクロールイベント

        protected override void OnMouseWheel(MouseWheelEventArgs e)
        {
            if (Keyboard.Modifiers == ModifierKeys.Control)
            {
                if (e.Delta > 0)
                {
                    TimelineViewUtil.ZoomIn(e.GetPosition(TimelineControlCanvas).X, TimelineEventCanvas.ActualWidth);
                }
                else if (e.Delta < 0)
                {
                    TimelineViewUtil.ZoomOut(e.GetPosition(TimelineControlCanvas).X, TimelineEventCanvas.ActualWidth);
                }
            }
            else if (Keyboard.Modifiers == ModifierKeys.Shift)
            {
                var sign = e.Delta > 0 ? -1 : 1;
                var scrollAmount = sign * TimelineViewUtil.GridWidth;
                TimelineViewUtil.PanningView(new Point(0, 0), new Point(scrollAmount, 0), TimelineViewUtil.PanningDirection.Horizontal);
            }
            else if (Keyboard.Modifiers == ModifierKeys.None)
            {
                var sign = e.Delta > 0 ? -1 : 1;
                var scrollAmount = sign * TimelineItem.ItemHeight;
                TimelineViewUtil.PanningView(new Point(0, 0), new Point(0, scrollAmount), TimelineViewUtil.PanningDirection.Vertical);
            }
            //timelineEventPanel.InvalidateVisual();

            base.OnMouseWheel(e);
        }

        protected override void OnMouseDown(MouseButtonEventArgs e)
        {
            switch (e.ChangedButton)
            {
                case MouseButton.Right:
                    if (_dragMode == DragMode.None)
                    {
                        _dragMode = DragMode.Panning;
                        PanStart(e);
                        e.Handled = true;
                    }
                    break;

                case MouseButton.Left:
                    if (_dragMode == DragMode.None)
                    {
                        this.SelectItem(null);

                        _dragMode = DragMode.Scrolling;
                        ScrollStart(e);
                        e.Handled = true;
                    }
                    break;
            }

            base.OnMouseDown(e);
        }

        protected override void OnMouseUp(MouseButtonEventArgs e)
        {
            switch (e.ChangedButton)
            {
                case MouseButton.Right:
                    if (_dragMode == DragMode.Panning)
                    {
                        PanEnd(e);
                        _dragMode = DragMode.None;
                        e.Handled = true;
                    }
                    break;

                case MouseButton.Left:
                    if (_dragMode == DragMode.Scrolling)
                    {
                        ScrollEnd(e);
                        _dragMode = DragMode.None;
                        e.Handled = true;
                    }
                    break;
            }

            base.OnMouseUp(e);
        }

        protected override void OnMouseMove(MouseEventArgs e)
        {
            switch (_dragMode)
            {
                case DragMode.Panning:
                    PanOnMouseMove(e);
                    e.Handled = true;
                    break;

                case DragMode.Scrolling:
                    ScrollOnMouseMove(e);
                    e.Handled = true;
                    break;
            }

            base.OnMouseMove(e);
        }

        private void PanStart(MouseButtonEventArgs e)
        {
            _scrollDir = ScrollDirType.None;
            _dragStartView = new Point(TimelineViewUtil.ViewStartFrame, TimelineViewUtil.ViewVerticalOffset);
            _dragStartPos = e.GetPosition(this.TimelineControlCanvas);
            _dragUpdatePos = e.GetPosition(this.TimelineControlCanvas);

            Keyboard.Focus(this);

            // CaptureMouse() から OnMouseMove() が呼び出されるので、
            // 変数の初期化はこの前におこなっておくこと。
            if (IsMouseOver)
            {
                CaptureMouse();
            }
        }

        private void PanEnd(MouseButtonEventArgs e)
        {
            // ReleaseMouseCapture() から OnMouseMove() が呼び出されるので、
            // 後始末はこの後に行うこと。
            if (IsMouseCaptured)
            {
                ReleaseMouseCapture();
            }

            Mouse.OverrideCursor = null;
        }

        private void PanOnMouseMove(MouseEventArgs e)
        {
            if (_scrollDir == ScrollDirType.None)
            {
                _dragUpdatePos = e.GetPosition(this.TimelineControlCanvas);
                // 動かし始めの方向でスクロール方向を固定する
                Vector vec = Point.Subtract(_dragUpdatePos, _dragStartPos);
                if (vec.Length > 8)
                {
                    double angle = Math.Abs(Math.Atan2(vec.Y, vec.X));
                    const double AngleRange = Math.PI * 20 / 180;

                    if (angle > Math.PI - AngleRange || angle < AngleRange)
                    {
                        _scrollDir = ScrollDirType.Horizontal;
                        Mouse.OverrideCursor = Cursors.ScrollWE;
                    }
                    else if (angle > Math.PI / 2 - AngleRange && angle < Math.PI / 2 + AngleRange)
                    {
                        _scrollDir = ScrollDirType.Vertical;
                        Mouse.OverrideCursor = Cursors.ScrollNS;
                    }
                    else
                    {
                        _scrollDir = ScrollDirType.Free;
                        Mouse.OverrideCursor = Cursors.ScrollAll;
                    }
                }
            }
            else
            {
                _dragUpdatePos = e.GetPosition(this.TimelineControlCanvas);
                switch (_scrollDir)
                {
                    case ScrollDirType.Horizontal:
                        TimelineViewUtil.PanningView(_dragStartView, _dragUpdatePos, _dragStartPos, TimelineViewUtil.PanningDirection.Horizontal);
                        break;
                    case ScrollDirType.Vertical:
                        TimelineViewUtil.PanningView(_dragStartView, _dragUpdatePos, _dragStartPos, TimelineViewUtil.PanningDirection.Vertical);
                        break;
                    case ScrollDirType.Free:
                        TimelineViewUtil.PanningView(_dragStartView, _dragUpdatePos, _dragStartPos, TimelineViewUtil.PanningDirection.Free);
                        break;
                }
                //timelineEventPanel.InvalidateVisual();
            }
        }

        private void ScrollStart(MouseButtonEventArgs e)
        {
            if (this.DragCurrentFrame != null)
            {
                this.DragCurrentFrame(this, DragMoveEventArgs.Start);
            }

            this.ScrollToCursorPos(e.GetPosition(this.TimelineControlCanvas).X);

            Keyboard.Focus(this);

            Mouse.OverrideCursor = Cursors.Hand;

            // CaptureMouse() から OnMouseMove() が呼び出されるので
            // 変数の初期化はこの前におこなっておくこと。
            if (IsMouseOver)
            {
                CaptureMouse();
            }
        }

        private void ScrollEnd(MouseButtonEventArgs e)
        {
            // ReleaseMouseCapture() から OnMouseMove() が呼び出されるので、
            // 後始末はこの後に行うこと。
            if (IsMouseCaptured)
            {
                ReleaseMouseCapture();
            }

            Mouse.OverrideCursor = null;

            _scrollTimer.Stop();

            if (this.DragCurrentFrame != null)
            {
                this.DragCurrentFrame(this, DragMoveEventArgs.End);
            }
        }

        private void ScrollOnMouseMove(MouseEventArgs e)
        {
            var currentPos = e.GetPosition(this.TimelineControlCanvas).X;
            var marginLeft = TimelineViewUtil.ScrollMarginLeft * this.TimelineControlCanvas.ActualWidth;
            var marginRight = TimelineViewUtil.ScrollMarginRight * this.TimelineControlCanvas.ActualWidth;

            if (marginLeft <= currentPos && currentPos <= marginRight)
            {
                // マウスカーソルが表示範囲内なら直ちにカレントフレームに反映します。
                _scrollTimer.Stop();
                this.ScrollToCursorPos(currentPos);
            }
            else
            {
                // マウスカーソルが表示範囲外の時は定期的にスクロールします。
                _scrollPos = currentPos;
                _scrollTimer.Start();
            }
        }

        private void ScrollOnTimer(object sender, EventArgs arg)
        {
            this.ScrollToCursorPos(_scrollPos);
        }

        /// <summary>
        /// マウスカーソルの位置をカレントフレームに設定し、
        /// 画面に入るようにスクロールします。
        /// </summary>
        /// <param name="cursorPos"></param>
        private void ScrollToCursorPos(double cursorPos)
        {
            long currentFrame = MathUtil.Clamp(
                TimelineViewUtil.ViewStartFrame + TimelineViewUtil.GetFrame(cursorPos),
                TimelineViewUtil.TotalStartFrame,
                TimelineViewUtil.TotalEndFrame);

            if (this.CurrentFrame != currentFrame)
            {
                this.CurrentFrame = currentFrame;
                TimelineViewUtil.SetViewStartFrameForAutoScroll(currentFrame, this.TimelineControlCanvas.ActualWidth);

                if (this.DragCurrentFrame != null)
                {
                    this.DragCurrentFrame(this, DragMoveEventArgs.Update);
                }
            }
        }

        private void OnTimelineItemIsSelectedChanged(object sender, RoutedEventArgs e)
        {
            var item = e.OriginalSource as TimelineItem;
            if (item.IsSelected)
            {
                this.SelectItem(item);
            }
            else
            {
                if (_selectedItemView == item)
                {
                    this.SelectItem(null);
                }
            }
        }

        private void UpdateViewOffset()
        {
            UpdateGrids();
        }

        #endregion

        /// <summary>
        /// 目盛線をキャンバスのサイズとズーム比によって引き直す
        /// </summary>
        private void UpdateGrids()
        {
            long frame = TimelineViewUtil.GetViewStartGrid();
            long endFrame = TimelineViewUtil.GetViewEndGrid(TimelineControlCanvas.ActualWidth);

            foreach (TimelineGrid grid in _grids)
            {
                if (frame <= endFrame)
                {
                    grid.DataContext = frame;
                    grid.Visibility = System.Windows.Visibility.Visible;
                }
                else
                {
                    grid.Visibility = System.Windows.Visibility.Collapsed;
                }

                frame += TimelineViewUtil.GridFrame;
            }

            while (frame <= endFrame)
            {
                TimelineGrid grid = new TimelineGrid(TimelineControlCanvas, this.TimeUnit);
                _grids.Add(grid);
                grid.DataContext = frame;

                TimelineControlCanvas.Children.Add(grid);

                frame += TimelineViewUtil.GridFrame;
            }
        }

        /// <summary>
        /// ViewModelのコレクションが変化したときに呼ばれる
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void UpdateItems(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    {
                        foreach (TimelineItemViewModel newItemVM in e.NewItems)
                        {
                            bool isAdd = false;
                            foreach (var group in _groups)
                            {
                                if (group.PlayerId == newItemVM.PlayerId)
                                {
                                    group.AddTimelineItemVM(newItemVM, this.TimeUnit);
                                    isAdd = true;
                                    break;
                                }
                            }
                            if (!isAdd)
                            {
                                var newGroup = new TimelineGroup(TimelineEventCanvas)
                                {
                                    PlayerId = newItemVM.PlayerId,
                                };
                                newGroup.SizeChanged += UpdateGroupPos;
                                newGroup.AddTimelineItemVM(newItemVM, this.TimeUnit);

                                TimelineEventCanvas.Children.Add(newGroup);
                                _groups.Add(newGroup);

                                _groups.Sort((x, y) => { return x.PlayerId.CompareTo(y.PlayerId); });
                            }
                        }
                    }
                    break;
                case NotifyCollectionChangedAction.Reset:
                    if (((System.Collections.IList)sender).Count == 0)
                    {
                        foreach (var group in _groups)
                        {
                            TimelineEventCanvas.Children.Remove(group);
                        }
                        _groups.Clear();
                    }
                    break;
            }
        }

        private void UpdateGroupPos(object sender, RoutedEventArgs e)
        {
            double contentHeight = 0;
            for (int i = 0; i < _groups.Count; i++)
            {
                contentHeight += _groups[i].ActualHeight;
                if (i == 0)
                {
                    Canvas.SetTop(_groups[i], TimelineViewUtil.ViewVerticalOffset + 0);
                }
                else
                {
                    Canvas.SetTop(_groups[i], Canvas.GetTop(_groups[i - 1]) + _groups[i - 1].ActualHeight);
                }
            }

            TimelineViewUtil.ContentHeight = contentHeight;
        }

        private void FilterTextBoxChagned(object sender, TextChangedEventArgs e)
        {
            var text = (string)((TextBox)sender).Text;

            TimelineItem.FilterText = text;

            foreach (var group in _groups)
            {
                group.OnFilterTextChanged();
            }

            UpdateGroupPos(this, null);
        }

        private void OrderChanged(object sender, SelectionChangedEventArgs e)
        {
            var comboBox = (ComboBox)sender;

            TimelineGroup.OrderType orderType = TimelineGroup.OrderType.Seq;
            if (comboBox.SelectedItem == OrderSeq)
            {
                orderType = TimelineGroup.OrderType.Seq;
            }
            else if (comboBox.SelectedItem == OrderID)
            {
                orderType = TimelineGroup.OrderType.ID;
            }

            foreach (var group in _groups)
            {
                group.OnOrderChanged(orderType);
            }

            UpdateGroupPos(this, null);
        }
    }

    /// <summary>
    /// グループで閉じたりする
    /// </summary>
    public class TimelineGroup : Expander
    {
        private readonly Canvas _eventCanvas = new Canvas();

        public uint PlayerId { get; set; }

        public SpyTimeUnit TimeUnit
        {
            set
            {
                foreach (var item in _items)
                {
                    item.TimeUnit = value;
                }
            }
        }

        private List<TimelineItem> _items = new List<TimelineItem>();

        private List<ItemSlot> _vmSlots = new List<ItemSlot>();

        private class ItemSlot : List<TimelineItem>
        {
            public TimelineItem Peek()
            {
                return this[this.Count - 1];
            }
        }

        public TimelineGroup(UIElement parent)
        {
            this.AddChild(_eventCanvas);

            SetBinding(WidthProperty, new Binding("ActualWidth") { Source = parent });
            _eventCanvas.SetBinding(Canvas.WidthProperty, new Binding("ActualWidth") { Source = parent });

            SizeChanged += (sender, e) => UpdateHorizontalGrouping();
            TimelineViewUtil.ZoomChanged += () => UpdateHorizontalGrouping();
            TimelineViewUtil.PanningChanged += () => UpdateHorizontalGrouping();

            IsExpanded = true;
        }

        /// <summary>
        /// アイテムを追加
        /// </summary>
        /// <param name="item"></param>
        /// <param name="timeUnit"></param>
        public void AddTimelineItemVM(TimelineItemViewModel item, SpyTimeUnit timeUnit)
        {
            // プレイヤー名
            if (this.Header == null)
            {
                this.Header = item.PlayerName;
            }

            // 何列目に入れるか
            ItemSlot setSlot = null;
            int slotNo = 0;
            foreach (var slotItem in _vmSlots)
            {
                if (!slotItem.Peek().IsActive && slotItem.Peek().EndTime.Timestamp < item.StartTime.Timestamp)
                {
                    setSlot = slotItem;
                    break;
                }
                slotNo++;
            }
            if (setSlot == null)
            {
                setSlot = new ItemSlot();
                _vmSlots.Add(setSlot);
                _eventCanvas.Height = (slotNo + 1) * TimelineItem.ItemHeight;
            }

            // test
            {
                var newItem = new TimelineItem()
                {
                    DataContext = item,
                    VerticalPos = slotNo,
                };
                setSlot.Add(newItem);

                item.States_.CollectionChanged += (sender_, e_) =>
                {
                    switch (e_.Action)
                    {
                        case NotifyCollectionChangedAction.Add:
                            foreach (TimelineItemStateViewModel stateVM in e_.NewItems)
                            {
                                var newState = new TimelineItemState(timeUnit)
                                {
                                    DataContext = stateVM,
                                };
                                newItem.AddState(newState);
                            }
                            break;
                    }
                };

                foreach (var stateVM in item.States)
                {
                    var newState = new TimelineItemState(timeUnit)
                    {
                        DataContext = stateVM,
                    };
                    newItem.AddState(newState);
                }

                _eventCanvas.Children.Add(newItem);

                // 全アイテムリスト
                _items.Add(newItem);
            }
        }

        /// <summary>
        /// フィルタテキストが変わった時
        /// </summary>
        public void OnFilterTextChanged()
        {
            // NOTE:
            // 時間の表示単位によらないようにするため、
            // アイテムの位置関係は実時間ベースで決定します。

            // スロットを開放
            foreach (var slot in _vmSlots)
            {
                slot.Clear();
            }
            _vmSlots.Clear();

            // キャンバスをクリア
            _eventCanvas.Children.Clear();
            _eventCanvas.Height = 0;

            foreach (var item in _items)
            {
                // 何列目に入れるか
                if (item.IsFiltering())
                {
                    ItemSlot setSlot = null;
                    int slotNo = 0;
                    foreach (var slotItem in _vmSlots)
                    {
                        if (!slotItem.Peek().IsActive && slotItem.Peek().EndTime.Timestamp < item.StartTime.Timestamp)
                        {
                            setSlot = slotItem;
                            break;
                        }
                        slotNo++;
                    }
                    if (setSlot == null)
                    {
                        setSlot = new ItemSlot();
                        _vmSlots.Add(setSlot);
                        _eventCanvas.Height = (slotNo + 1) * TimelineItem.ItemHeight;
                    }

                    item.VerticalPos = slotNo;

                    setSlot.Add(item);

                    _eventCanvas.Children.Add(item);
                }
            }

            if (_eventCanvas.Height == 0)
            {
                this.Visibility = System.Windows.Visibility.Collapsed;
            }
            else
            {
                this.Visibility = System.Windows.Visibility.Visible;
            }
        }

        public enum OrderType
        {
            Seq,
            ID,
        }

        /// <summary>
        /// 並び順を変更
        /// </summary>
        /// <param name="type"></param>
        public void OnOrderChanged(OrderType type)
        {
            // NOTE:
            // 時間の表示単位によらないようにするため、
            // アイテムの位置関係は実時間ベースで決定します。

            // スロットを開放
            foreach (var slot in _vmSlots)
            {
                slot.Clear();
            }
            _vmSlots.Clear();

            // キャンバスをクリア
            _eventCanvas.Children.Clear();
            _eventCanvas.Height = 0;

            // アイテムをソート
            switch (type)
            {
                case OrderType.Seq:
                    _items.Sort((x, y) => { return x.StartTime.Timestamp.CompareTo(y.StartTime.Timestamp); });
                    break;
                case OrderType.ID:
                    {
                        _items.Sort((x, y) => { return x.SoundId.CompareTo(y.SoundId); });

                        var newItems = new List<TimelineItem>();
                        uint tmpId = 0;
                        foreach (var item in _items)
                        {
                            if (tmpId != item.SoundId)
                            {
                                tmpId = item.SoundId;
                                var group = _items.FindAll((e) => { return e.SoundId == tmpId; });
                                group.Sort((x, y) => { return x.StartTime.Timestamp.CompareTo(y.StartTime.Timestamp); });
                                newItems.AddRange(group);
                            }
                        }

                        _items = newItems;
                    }
                    break;
            }

            foreach (var item in _items)
            {
                // 何列目に入れるか
                if (item.IsFiltering())
                {
                    ItemSlot setSlot = null;
                    int slotNo = 0;

                    switch (type)
                    {
                        case OrderType.Seq:
                            {
                                foreach (var slotItem in _vmSlots)
                                {
                                    if (!slotItem.Peek().IsActive && slotItem.Peek().EndTime.Timestamp < item.StartTime.Timestamp)
                                    {
                                        setSlot = slotItem;
                                        break;
                                    }
                                    slotNo++;
                                }
                            }
                            break;
                        case OrderType.ID:
                            {
                                foreach (var slotItem in _vmSlots)
                                {
                                    if (slotItem.Peek().SoundId == item.SoundId)
                                    {
                                        if (!slotItem.Peek().IsActive && slotItem.Peek().EndTime.Timestamp < item.StartTime.Timestamp)
                                        {
                                            setSlot = slotItem;
                                            break;
                                        }
                                    }
                                    slotNo++;
                                }
                            }
                            break;
                    }

                    if (setSlot == null)
                    {
                        setSlot = new ItemSlot();
                        _vmSlots.Add(setSlot);
                        _eventCanvas.Height = (slotNo + 1) * TimelineItem.ItemHeight;
                    }

                    item.VerticalPos = slotNo;

                    setSlot.Add(item);

                    _eventCanvas.Children.Add(item);
                }
            }
        }

        /// <summary>
        /// 横方向のまとめ
        /// </summary>
        public void UpdateHorizontalGrouping()
        {
            foreach (var slot in _vmSlots)
            {
                double groupStartPos = -1;
                foreach (var item in slot)
                {
                    item.UpdateView();

                    // 横軸の描画範囲内かどうか
                    if (item.IsItemVisible())
                    {
                        if (item.ActualWidth < 2)
                        {
                            if (groupStartPos == -1)
                            {
                                groupStartPos = Canvas.GetLeft(item);
                                item.Visibility = System.Windows.Visibility.Visible;
                            }
                            else
                            {
                                var pos = Canvas.GetLeft(item);
                                if (Math.Abs(pos - groupStartPos) < 6)
                                {
                                    item.Visibility = System.Windows.Visibility.Collapsed;
                                }
                                else
                                {
                                    groupStartPos = Canvas.GetLeft(item);
                                    item.Visibility = System.Windows.Visibility.Visible;
                                }
                            }
                        }
                        else
                        {
                            item.Visibility = System.Windows.Visibility.Visible;
                        }
                    }
                    else
                    {
                        item.Visibility = System.Windows.Visibility.Collapsed;
                    }
                }
            }
        }

        public void UpdateView()
        {
            foreach (var item in _items)
            {
                item.UpdateView();
            }
        }
    }

    /// <summary>
    /// カレントポジションを示すライン
    /// </summary>
    public class TimelineCurrentLine
    {
        private readonly Line _currentLine = new Line();
        private Canvas _parentCanvas = null;
        public long CurrentFrame { get; private set; }
        public double CurrentPos { get; private set; }

        /// <summary>
        /// 初期化
        /// </summary>
        /// <param name="parentCanvas"></param>
        public TimelineCurrentLine(Canvas parentCanvas)
        {
            _parentCanvas = parentCanvas;
            _parentCanvas.Children.Add(_currentLine);
            Canvas.SetZIndex(_currentLine, 127);

            _currentLine.Stroke = new SolidColorBrush(Colors.Red);
            //CurrentLine.StrokeThickness = 1;
            //CurrentLine.SnapsToDevicePixels = true;

            CurrentFrame = 0;
            CurrentPos = 0;

            _parentCanvas.SizeChanged += (sender, e) => { _currentLine.Y2 = _parentCanvas.ActualHeight; };
            TimelineViewUtil.ZoomChanged += () => UpdateCurrentFrame(CurrentFrame);
            TimelineViewUtil.PanningChanged += () => UpdateCurrentFrame(CurrentFrame);
        }

        /// <summary>
        /// カレントフレームが変わったら呼ばれる
        /// </summary>
        /// <param name="frame"></param>
        public void UpdateCurrentFrame(long frame)
        {
            CurrentFrame = frame;

            CurrentPos = TimelineViewUtil.GetRenderPos(CurrentFrame);
            if (CurrentPos < 0 || CurrentPos > _parentCanvas.ActualWidth)
            {
                _currentLine.Visibility = Visibility.Collapsed;
            }
            else
            {
                _currentLine.X1 = _currentLine.X2 = CurrentPos;
                _currentLine.Visibility = Visibility.Visible;
            }
        }
    }
}
