﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using NintendoWare.Spy;
using System;
using System.Collections.Concurrent;
using System.Drawing;
using System.Threading;
using System.Windows.Forms;

namespace NintendoWare.NwSoundSpyPlugin.Windows
{
    public partial class FinalOutControl : UserControl
    {
        private class PaintTaskResult
        {
            public bool IsCanceled { get; set; }
            public SampleRangeCollection IndexRanges { get; set; }
            public CancelableLineGraphDrawer.DataStore DataStore { get; set; }

            public PaintTaskResult()
            {
                this.IndexRanges = new SampleRangeCollection();
            }
        }

        private const int SamplePerFrame = 144;

        private LineGraphControlHandler _controlHandler = null;
        private LineGraphDrawer.BitmapStock _bitmapStock = new LineGraphDrawer.BitmapStock();
        private readonly SampleRangeCollection _bitmapStockRanges = new SampleRangeCollection();
        private readonly TaskController<PaintTaskResult> _paintTaskController = new TaskController<PaintTaskResult>();
        private readonly ConcurrentQueue<PaintTaskResult> _paintTaskResults = new ConcurrentQueue<PaintTaskResult>();
        private Frame _audioFrameBegin = Frame.InvalidValue;

        private const int MaxDrawChannelCountTv = 6;
        private const int MaxDrawChannelCountDrc = 2;
        private static readonly string[] ChannelNameArray = new string[MaxDrawChannelCountTv + MaxDrawChannelCountDrc]
        {
            "F-L", "F-R", "R-L", "R-R", "F-C", "LFE", "DRC-L", "DRC-R"
        };

        public FinalOutSpyModel Model { get; set; } = null;

        public int DrawChannelsTv { get; set; } = 4;

        public int DrawChannelsDrc { get; set; } = 2;

        public bool PauseFlag
        {
            get { return _controlHandler.PauseFlag; }
            set { _controlHandler.PauseFlag = value; }
        }

        public event EventHandler<DragMoveEventArgs> CurrentDraggedEvent
        {
            add { _controlHandler.CurrentDraggedEvent += value; }
            remove { _controlHandler.CurrentDraggedEvent -= value; }
        }

        public delegate FinalOutSpyModel.FinalOut GetFinalOutHandler(Frame audioFrame);
        public event GetFinalOutHandler GetFinalOut;

        public FinalOutControl()
        {
            // this.InitializeComponent();
            DoubleBuffered = true;
            BackColor = Color.Black;

            _controlHandler = new LineGraphControlHandler(this, -16);
        }

        public void Reset()
        {
            this.Model = null;
            _bitmapStock = new LineGraphDrawer.BitmapStock();
            _bitmapStockRanges.Clear();
            Invalidate();
        }

        /// <summary>
        /// プロットするオーディオフレームの範囲を指定します。
        /// </summary>
        /// <param name="audioFrameBegin"></param>
        /// <param name="audioFrameEnd"></param>
        public void SetAudioFrameRange(Frame audioFrameBegin, Frame audioFrameEnd)
        {
            if (audioFrameBegin != Frame.InvalidValue && audioFrameBegin <= audioFrameEnd)
            {
                _audioFrameBegin = audioFrameBegin;
                _controlHandler.LastSampleIndex = (audioFrameEnd - audioFrameBegin + 1) * SamplePerFrame - 1;
            }
            else if (_audioFrameBegin != Frame.InvalidValue)
            {
                _audioFrameBegin = Frame.InvalidValue;
                _controlHandler.LastSampleIndex = -1;
            }
        }

        public Frame GetCurrentFrame()
        {
            if (_audioFrameBegin == Frame.InvalidValue)
            {
                return Frame.InvalidValue;
            }
            return _audioFrameBegin + _controlHandler.CurrentSampleIndex / SamplePerFrame;
        }

        public void ScrollToAudioFramePos(Frame audioFrame)
        {
            if (this.Model == null || _controlHandler.CaptureFlag)
            {
                // キャプチャー時、外部からの変更は受け付けない
                return;
            }

            if (_audioFrameBegin == Frame.InvalidValue)
            {
                return;
            }

            if (audioFrame == GetCurrentFrame())
            {
                // 現在のオーディオフレームを同じなら、currentSampleIndexを更新しない
                return;
            }

            this.UpdateSamples();
            _controlHandler.SetCurrentSampleIndex((audioFrame - _audioFrameBegin) * SamplePerFrame);
            Invalidate();
        }

        public void UpdateSamples()
        {
            if (this.Model == null)
            {
                return;
            }
        }

        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);

            if (e.Button == MouseButtons.Left)
            {
                if (!PauseFlag && this.Model != null)
                {
                    this.UpdateSamples();
                }
            }
            _controlHandler.OnMouseDown(e);
        }

        protected override void OnMouseUp(MouseEventArgs e)
        {
            base.OnMouseUp(e);

            _controlHandler.OnMouseUp(e);
        }

        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);

            _controlHandler.OnMouseMove(e);
        }

        protected override void OnMouseWheel(MouseEventArgs e)
        {
            base.OnMouseWheel(e);

            _controlHandler.OnMouseWheel(e);
        }

        protected override void OnPreviewKeyDown(PreviewKeyDownEventArgs e)
        {
            base.OnPreviewKeyDown(e);

            _controlHandler.OnPreviewKeyDown(e);
        }

        private float[] GetSampleValue(CancellationToken ct, long sampleIndex)
        {
            int finalOutIndex = (int)(sampleIndex / SamplePerFrame);
            FinalOutSpyModel.FinalOut finalOut = null;
            if (GetFinalOut != null)
            {
                finalOut = GetFinalOut(_audioFrameBegin + finalOutIndex);
            }

            if (finalOut == null)
            {
                return new float[0];
            }

            long sampleOffset = sampleIndex - finalOutIndex * SamplePerFrame;

            var nbChannels = finalOut.SampleData.Samples.Length;

            var samples = new float[nbChannels];

            for (int i = 0; i < nbChannels; ++i)
            {
                samples[i] = finalOut.SampleData.Samples[i][sampleOffset];
            }

            return samples;
        }

        private const int ScrollBarHeight = 10;
        private const int MarginTop = 10;
        private const int MarginMiddle = 10; // グラフ間のマージン

        protected override void OnPaint(PaintEventArgs e)
        {
            const int NoPaintSize = 30;

            _paintTaskController.Polling();

            if (ClientSize.Width <= NoPaintSize || ClientSize.Height <= NoPaintSize)
            {
                ClearPaintTaskResults();
                return;
            }

            Graphics g = e.Graphics;

            int totalDrawChannelCount = DrawChannelsTv + DrawChannelsDrc;
            if (totalDrawChannelCount == 0)
            {
                return;
            }

            int GraphHeight = ClientSize.Height - MarginTop - ScrollBarHeight;
            int RangeHeight = (GraphHeight / totalDrawChannelCount - MarginMiddle) / 2;

            float fontSize = this.Font.Size * 0.7f;
            Font newFont = new Font(this.Font.FontFamily, fontSize, this.Font.Style);

            // 罫線
            for (int graphIndex = 0; graphIndex < totalDrawChannelCount; graphIndex++)
            {
                int topLineY = GraphHeight * graphIndex / totalDrawChannelCount + MarginTop;
                int centerLineY = topLineY + RangeHeight;
                int bottomLineY = topLineY + RangeHeight * 2;

                g.DrawLine(Pens.Gray, new Point(0, centerLineY), new Point(ClientSize.Width - 1, centerLineY));
                g.DrawLine(Pens.DimGray, new Point(0, topLineY), new Point(ClientSize.Width - 1, topLineY));
                g.DrawLine(Pens.DimGray, new Point(0, bottomLineY), new Point(ClientSize.Width - 1, bottomLineY));

                int channelIndex = graphIndex;
                if (graphIndex >= DrawChannelsTv)
                {
                    channelIndex = MaxDrawChannelCountTv + graphIndex - DrawChannelsTv;
                }
                g.DrawString(ChannelNameArray[channelIndex], newFont, Brushes.White, 0, topLineY);
            }

            if (this.Model != null)
            {
                long leftSampleIndex = _controlHandler.LeftSampleIndex;
                long rightSampleIndex = _controlHandler.RightSampleIndex;

                // 時間遷移グラフ表示
                if (rightSampleIndex >= leftSampleIndex)
                {
                    int x = 0;
                    int y = MarginTop;
                    int width = ClientSize.Width;
                    int height = ClientSize.Height - MarginTop - MarginMiddle - ScrollBarHeight;
                    int centerY = RangeHeight;
                    float yScale = RangeHeight / 32768.0f;

                    if (height <= 0)
                    {
                        ClearPaintTaskResults();
                        return;
                    }

                    Bitmap bmp = new Bitmap(width, height, g);
                    Graphics gBmp = Graphics.FromImage(bmp);

                    long drawLeft = leftSampleIndex;
                    long drawRight = rightSampleIndex;

                    var bitmapStockApplied = _bitmapStock.Apply(
                        gBmp,
                        height,
                        ref drawLeft, ref drawRight,
                        leftSampleIndex, rightSampleIndex,
                        _controlHandler.BeginSamplePosX,
                        _controlHandler.ZoomRatio.Index, yScale);

                    if (!bitmapStockApplied)
                    {
                        _bitmapStockRanges.Clear();
                    }

                    var graphSampleRange = new SampleRange(leftSampleIndex, rightSampleIndex);
                    while (!_paintTaskResults.IsEmpty)
                    {
                        PaintTaskResult result;
                        if (_paintTaskResults.TryDequeue(out result))
                        {
                            foreach (var range in result.IndexRanges)
                            {
                                // 折れ線グラフ描画
                                for (int graphIndex = 0; graphIndex < totalDrawChannelCount; graphIndex++)
                                {
                                    int posY = GraphHeight * graphIndex / totalDrawChannelCount;
                                    int channelIndex = graphIndex;
                                    if (graphIndex >= DrawChannelsTv)
                                    {
                                        channelIndex = MaxDrawChannelCountTv + graphIndex - DrawChannelsTv;
                                    }

                                    CancelableLineGraphDrawer.Draw(
                                        gBmp,
                                        Pens.SkyBlue,
                                        0,
                                        posY,
                                        width,
                                        height,
                                        RangeHeight,
                                        yScale,
                                        result.DataStore,
                                        _controlHandler.BeginSamplePosX,
                                        range.Left,
                                        range.Right,
                                        channelIndex);
                                }

                                _bitmapStockRanges.AddRange(range);
                            }
                        }
                    }

                    _bitmapStockRanges.LimitRange(graphSampleRange);

                    // バックグラウンドタスクが無制限に繰り返し発行されないようにするため、
                    // 新しい領域が画面中に現れたときにだけタスクの発行をします。
                    if (!bitmapStockApplied ||
                        leftSampleIndex != _bitmapStock.LeftSampleIndex ||
                        rightSampleIndex != _bitmapStock.RightSampleIndex)
                    {
                        var indexRanges = _bitmapStockRanges.InvertRange(graphSampleRange);

                        // 描画していない領域が残っていたら、バックグラウンドでリクエストします。
                        if (indexRanges.Count > 0)
                        {
                            var beginSamplePosX = _controlHandler.BeginSamplePosX;
                            var zoomIndex = _controlHandler.ZoomRatio.Index;

                            _paintTaskController.Request((ct, lastResult) =>
                                {
                                    var result = new PaintTaskResult();

                                    if (lastResult != null &&
                                        lastResult.DataStore != null &&
                                        lastResult.DataStore.ZoomRatioIndex == zoomIndex)
                                    {
                                        //System.Diagnostics.Trace.WriteLine(string.Format("prev {0}", lastResult.IsCanceled ? "cancel" : "finish"));

                                        // 前のタスクがキャンセルされたときは、結果を再利用します。
                                        if (lastResult.IsCanceled)
                                        {
                                            result.IndexRanges = lastResult.IndexRanges;
                                            result.DataStore = lastResult.DataStore;
                                        }

                                        // 前のタスクで取得済みのデータは再取得しません。
                                        foreach (var item in lastResult.IndexRanges)
                                        {
                                            indexRanges.RemoveRange(item);
                                        }
                                    }

                                    if (result.DataStore == null)
                                    {
                                        result.DataStore = new CancelableLineGraphDrawer.DataStore(zoomIndex);
                                    }

                                    foreach (var range in indexRanges)
                                    {
                                        if (ct.IsCancellationRequested)
                                        {
                                            result.IsCanceled = true;
                                            break;
                                        }

                                        var left = range.Left;
                                        var right = range.Right;
                                        CancelableLineGraphDrawer.FetchData(
                                            ct,
                                            result.DataStore,
                                            ref left,
                                            ref right,
                                            this.GetSampleValue);

                                        if (left <= right)
                                        {
                                            result.IndexRanges.AddRange(left, right);
                                        }
                                    }

                                    // キャンセルされずに完了したら、画面を再描画します。
                                    if (!result.IsCanceled)
                                    {
                                        _paintTaskResults.Enqueue(result);
                                        this.Invoke((Action)(() => this.Invalidate()));
                                    }

                                    return result;
                                });
                        }
                    }

                    g.DrawImage(bmp, new Point(x, y));

                    gBmp.Dispose();
                    gBmp = null;

                    _bitmapStock.Update(
                        bmp,
                        leftSampleIndex, rightSampleIndex,
                        _controlHandler.BeginSamplePosX,
                        _controlHandler.ZoomRatio.Index, yScale);
                }

                // カレント位置描画
                if ((_controlHandler.CurrentSampleIndex >= 0 && PauseFlag) || _controlHandler.CaptureFlag)
                {
                    int currentSamplePosX = _controlHandler.CurrentSamplePosX;
                    g.DrawLine(Pens.Red, currentSamplePosX, 0, currentSamplePosX, ClientSize.Height);
                }

                // スクロールバー描画
                {
                    LineGraphDrawer.DrawScrollBar(g,
                        0, ClientSize.Height - ScrollBarHeight,
                        ClientSize.Width, ScrollBarHeight,
                        leftSampleIndex, rightSampleIndex, _controlHandler.LastSampleIndex);
                }
            }
        }

        private void ClearPaintTaskResults()
        {
            while (!_paintTaskResults.IsEmpty)
            {
                PaintTaskResult request;
                _paintTaskResults.TryDequeue(out request);
            }
        }
    }
}
