﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using NintendoWare.Spy;
using System;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;

namespace NintendoWare.NwSoundSpyPlugin.Models
{
    /// <summary>
    /// NwSnd ボイスインフォ Spy モデルです。
    /// </summary>
    public sealed class VoiceInfoSpyModel : SpyModel
    {
        /// <summary>
        /// バージョン 0.1.0.0
        /// </summary>
        /// <remarks>
        /// パケットフォーマット：
        /// <code>
        /// struct VoiceInfoPacketData {
        ///     struct VoiceInfo
        ///     {
        ///         u8 index;
        ///         u8 status;
        ///         u16 padding1[1];
        ///         f32 volume;
        ///         f32 pitch;
        ///         f32 mainMixLeft;
        ///         f32 mainMixRight;
        ///         f32 drcMixLeft;
        ///         f32 drcMixRight;
        ///     };
        ///
        ///     u32 audioFrame;
        ///     u32 arraySize;
        ///     VoiceInfo voiceInfo[arraySize];
        /// };
        /// </code>
        /// </remarks>
        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        private static readonly Version Version_0_1_0_0 = new Version(0, 1, 0, 0);

        /// <summary>
        /// 非サポートバージョン。
        /// 最新のサポートバージョンよりマイナーバージョンを１つ大きくします。
        /// </summary>
        private static readonly Version VersionUnexpected = new Version(0, 2, 0, 0);

        public enum VoiceStatus
        {
            Play = 0,
            Stop = 1,
            Pause = 2
        }

        public class VoiceInfo
        {
            public VoiceInfo()
            {
                this.Status = VoiceStatus.Stop;
                this.Volume = 0.0f;
                this.Pitch = 0.0f;
                this.MainMixLeft = 0.0f;
                this.MainMixRight = 0.0f;
                this.DrcMixLeft = 0.0f;
                this.DrcMixRight = 0.0f;
            }

            public VoiceStatus Status { get; set; }
            public float Volume { get; set; }
            public float Pitch { get; set; }
            public float MainMixLeft { get; set; }
            public float MainMixRight { get; set; }
            public float DrcMixLeft { get; set; }
            public float DrcMixRight { get; set; }
        }

        public class VoiceArray
        {
            public const int VoiceCountMax = 96;
            public readonly VoiceInfo[] Item = new VoiceInfo[VoiceCountMax];
            public SpyGlobalTime Timestamp { get; set; }
            public SpyTime BelongingFrame { get; set; }

            public VoiceArray Clone()
            {
                VoiceArray voiceArray = new VoiceArray();
                voiceArray.Timestamp = this.Timestamp;
                voiceArray.BelongingFrame = this.BelongingFrame;
                for (int index = 0; index < VoiceCountMax; index++)
                {
                    voiceArray.Item[index] = this.Item[index];
                }

                return voiceArray;
            }
        }

        private int _audioFrameBegin = -1;

        public List<VoiceArray> VoiceArrayList { get; } = new List<VoiceArray>();

        public delegate void UpdateEventHandler(object sender, EventArgs e);
        public event UpdateEventHandler UpdateEvent;

        protected override void OnPushData(SpyDataBlock dataBlock)
        {
            if (this.DataVersion >= VersionUnexpected)
            {
                return;
            }

            var reader = CreateDataReader(dataBlock);

            uint audioFrame = reader.ReadUInt32();
            uint arraySize = reader.ReadUInt32();

            if (_audioFrameBegin < 0)
            {
                _audioFrameBegin = (int)audioFrame;
            }

            int audioFrameIndex = (int)(audioFrame - _audioFrameBegin);
            while (VoiceArrayList.Count < audioFrameIndex + 1)
            {
                VoiceArray va;
                if (VoiceArrayList.Count > 0)
                {
                    va = VoiceArrayList[VoiceArrayList.Count - 1].Clone();
                }
                else
                {
                    va = new VoiceArray();
                }
                VoiceArrayList.Add(va);
            }

            VoiceArray voiceArray = VoiceArrayList[audioFrameIndex];
            voiceArray.Timestamp = dataBlock.Timestamp;
            voiceArray.BelongingFrame = GetBelongingFrame(dataBlock.Timestamp);
            for (uint index = 0; index < arraySize; index++)
            {
                byte channelId = reader.ReadByte();
                byte status = reader.ReadByte();
                reader.ReadByte();
                reader.ReadByte();
                float volume = reader.ReadSingle();
                float pitch = reader.ReadSingle();
                float mainMixLeft = reader.ReadSingle();
                float mainMixRight = reader.ReadSingle();
                float drcMixLeft = reader.ReadSingle();
                float drcMixRight = reader.ReadSingle();

                if (channelId < VoiceArray.VoiceCountMax)
                {
                    VoiceInfo voiceInfo = new VoiceInfo();
                    voiceArray.Item[channelId] = voiceInfo;

                    voiceInfo.Status = (VoiceStatus)status;
                    voiceInfo.Volume = volume;
                    voiceInfo.Pitch = pitch;
                    voiceInfo.MainMixLeft = mainMixLeft;
                    voiceInfo.MainMixRight = mainMixRight;
                    voiceInfo.DrcMixLeft = drcMixLeft;
                    voiceInfo.DrcMixRight = drcMixRight;
                }
            }

            if (UpdateEvent != null)
            {
                UpdateEvent(this, EventArgs.Empty);
            }
        }

        public int FindIndexFromTime(long frame)
        {
            if (_audioFrameBegin < 0)
            {
                return -1;
            }
            int index = (int)(frame - _audioFrameBegin);
            if (index < 0)
            {
                index = 0;
            }
            else if (index > this.VoiceArrayList.Count - 1)
            {
                index = this.VoiceArrayList.Count - 1;
            }
            return index;
        }
    }
}
