﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using NintendoWare.Spy;
using System;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;

namespace NintendoWare.NwSoundSpyPlugin.Models
{
    /// <summary>
    /// NwSnd プロファイル Spy モデルです。
    /// </summary>
    public sealed class ProfileSpyModel : SpyModel
    {
        /// <summary>
        /// バージョン 0.1.0.0
        /// </summary>
        /// <remarks>
        /// パケットフォーマット：
        /// <code>
        /// struct ProfilePacket {
        ///     u32 audioFrame;
        ///     u32 voiceCount;
        ///     s64 nwFrameProcessBegin;
        ///     s64 nwFrameProcessEnd;
        ///     s64 nwVoiceParamUpdateBegin;
        ///     s64 nwVoiceParamUpdateEnd;
        ///     s64 ppcVoiceRenderingBegin;
        ///     s64 ppcVoiceRenderingEnd;
        ///     s64 auxProcessBegin;
        ///     s64 auxProcessEnd;
        ///     s64 dspFrameProcessBegin;
        ///     s64 dspFrameProcessEnd;
        ///     s64 syncVoiceParamBegin;
        ///     s64 syncVoiceParamEnd;
        ///     s64 outputFormatProcessBegin;
        ///     s64 outputFormatProcessEnd;
        ///     s64 mainMixProcessBegin;
        ///     s64 mainMixProcessEnd;
        ///     s64 finalMixProcessBegin;
        ///     s64 finalMixProcessEnd;
        ///     s64 axIntrBegin;
        ///     s64 axIntrEnd;
        /// }
        /// </code>
        /// </remarks>
        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        private static readonly Version Version_0_1_0_0 = new Version(0, 1, 0, 0);

        /// <summary>
        /// 非サポートバージョン。
        /// 最新のサポートバージョンよりマイナーバージョンを１つ大きくします。
        /// </summary>
        private static readonly Version VersionUnexpected = new Version(0, 2, 0, 0);

        public List<Profile> ProfileList { get; } = new List<Profile>();
        public int AudioFrameBegin { get; private set; } = -1;

        public delegate void UpdateEventHandler(object sender, EventArgs e);
        public event UpdateEventHandler UpdateEvent;

        public struct TimeSpan
        {
            public long Begin { get; }
            public long End { get; }
            public long Span { get { return End - Begin; } }

            public TimeSpan(long begin, long end)
            {
                this.Begin = begin;
                this.End = end;
            }
        }

        public class Profile
        {
            public SpyGlobalTime Timestamp { get; set; }
            public SpyTime BelongingFrame { get; set; }
            public uint VoiceCount { get; set; }
            public TimeSpan NwFrameProcess { get; set; }
            public TimeSpan NwVoiceParamUpdate { get; set; }
            public TimeSpan PpcVoiceRendering { get; set; }
            public TimeSpan AuxProcess { get; set; }
            public TimeSpan DspFrameProcess { get; set; }
            public TimeSpan SyncVoiceParam { get; set; }
            public TimeSpan OutputFormatProcess { get; set; }
            public TimeSpan MainMixProcess { get; set; }
            public TimeSpan FinalMixProcess { get; set; }
            public TimeSpan AxFrameProcess { get; set; }
        }

        public int FindProfileIndexFromTime(long frame)
        {
            if (this.AudioFrameBegin < 0)
            {
                return -1;
            }
            int profileIndex = (int)(frame - this.AudioFrameBegin);
            if (profileIndex < 0)
            {
                profileIndex = 0;
            }
            else if (profileIndex > this.ProfileList.Count - 1)
            {
                profileIndex = this.ProfileList.Count - 1;
            }
            return profileIndex;
        }

        protected override void OnPushData(SpyDataBlock dataBlock)
        {
            if (this.DataVersion >= VersionUnexpected)
            {
                return;
            }

            var reader = CreateDataReader(dataBlock);

            Profile profile = new Profile();

            uint audioFrame = reader.ReadUInt32();
            if (this.AudioFrameBegin < 0)
            {
                this.AudioFrameBegin = (int)audioFrame;
            }
            int audioFrameIndex = (int)(audioFrame - this.AudioFrameBegin);
            while (ProfileList.Count < audioFrameIndex + 1)
            {
                ProfileList.Add(null);
            }

            profile.Timestamp = dataBlock.Timestamp;
            profile.BelongingFrame = GetBelongingFrame(dataBlock.Timestamp);
            profile.VoiceCount = reader.ReadUInt32();
            profile.NwFrameProcess = new TimeSpan(reader.ReadInt64(), reader.ReadInt64());
            profile.NwVoiceParamUpdate = new TimeSpan(reader.ReadInt64(), reader.ReadInt64());
            profile.PpcVoiceRendering = new TimeSpan(reader.ReadInt64(), reader.ReadInt64());
            profile.AuxProcess = new TimeSpan(reader.ReadInt64(), reader.ReadInt64());
            profile.DspFrameProcess = new TimeSpan(reader.ReadInt64(), reader.ReadInt64());
            profile.SyncVoiceParam = new TimeSpan(reader.ReadInt64(), reader.ReadInt64());
            profile.OutputFormatProcess = new TimeSpan(reader.ReadInt64(), reader.ReadInt64());
            profile.MainMixProcess = new TimeSpan(reader.ReadInt64(), reader.ReadInt64());
            profile.FinalMixProcess = new TimeSpan(reader.ReadInt64(), reader.ReadInt64());
            profile.AxFrameProcess = new TimeSpan(reader.ReadInt64(), reader.ReadInt64());

            ProfileList[audioFrameIndex] = profile;

            if (UpdateEvent != null)
            {
                UpdateEvent(this, EventArgs.Empty);
            }
        }
    }
}
