﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Windows.Input;
using NintendoWare.NnAtkSpyPlugin.Plugins;
using NintendoWare.Spy;
using NintendoWare.Spy.Framework.Settings;
using NintendoWare.Spy.Settings;
using NintendoWare.Spy.Windows;
using System.ComponentModel;
using System.Linq;
using System.Windows.Controls;

namespace NintendoWare.NnAtkSpyPlugin.Windows
{
    internal class StartSoundHistoryPanelPresenter : SpyPanelPresenter
    {
        private readonly object _observerOwner = new object();
        private readonly object _settingsObserverOwner = new object();
        private StartSoundHistoryPanel _view;
        private StartSoundHistoryPanelViewModel _viewModel;
        private SpyTimeUnit _timeUnit;

        protected override void OnInitialize()
        {
            base.OnInitialize();

            PropertyChangedObservation.GetObserver(_observerOwner, this.GetPlaybackService())
                .AddHandler(
                    target => target.Current,
                    (target, args) => this.UpdateGlobalTimelineCurrent());

            this.UpdateGlobalTimelineCurrent();
        }

        protected override void OnUninitialize()
        {
            PropertyChangedObservation.RemoveObservers(_observerOwner);
            PropertyChangedObservation.RemoveObservers(_settingsObserverOwner);

            Disposer.SafeDispose(ref _viewModel);

            base.OnUninitialize();
        }

        protected override Control CreateContent()
        {
            _view = new StartSoundHistoryPanel();
            _viewModel = new StartSoundHistoryPanelViewModel();
            _viewModel.ListSortedCommand = new DelegateCommand(this.ExecuteSortedCommand, (p => true));
            _view.DataContext = _viewModel;

            this.ApplySettings(null);

            PropertyChangedObservation.GetObserver(_observerOwner, _viewModel)
                .AddHandler(
                    target => target.SelectedItem,
                    (target, args) =>
                    {
                        if (target.SelectedItem != null)
                        {
                            if (!this.GetPlaybackService().ShouldAutoScroll)
                            {
                                this.GetPlaybackService().SetCurrentTime(target.SelectedItem.Time.Timestamp);
                            }
                        }
                    });

            return _view;
        }

        private void UpdateGlobalTimelineCurrent()
        {
            var current = GetPlaybackService().Current;
            UpdateDrawBorderTypes(current);
            UpdateScrollIntoView();
        }

        private void UpdateDrawBorderTypes(SpyTime currentTime)
        {
            if (_viewModel == null || _viewModel.Items == null || _viewModel.Items.Count == 0)
            {
                return;
            }

            // 先頭のアイテムよりもカレントタイムが小さい場合の描画タイプです。
            DrawBorderType startType = DrawBorderType.None;

            // アイテムよりもカレントタイムが大きい場合の描画タイプです。
            DrawBorderType endType = DrawBorderType.None;

            // カラム "Time" の時に endType, startType は有効になります。
            if (_viewModel.ViewItems.SortDescriptions[0].PropertyName == "Time.Timestamp")
            {
                if (_viewModel.ViewItems.SortDescriptions[0].Direction == ListSortDirection.Descending)
                {
                    startType = DrawBorderType.Bottom;
                    endType = DrawBorderType.Top;
                }
                else
                {
                    startType = DrawBorderType.Top;
                    endType = DrawBorderType.Bottom;
                }
            }

            var comparer = new SpyTimeTimeUnitComparer(_timeUnit);

            // 先頭のアイテムよりもカレントタイムが小さい場合
            if (comparer.Compare(currentTime, _viewModel.Items[0].Time) < 0)
            {
                _viewModel.Items[0].DrawBorder = startType;

                // 先頭のアイテム以外にはカレントタイムを描画しない設定をします。
                _viewModel.Items.Skip(1).ForEach(vm => vm.DrawBorder = DrawBorderType.None);

                return;
            }

            // 末尾のアイテムよりもカレントタイムが大きい場合
            if (comparer.Compare(_viewModel.Items[_viewModel.Items.Count - 1].Time, currentTime) < 0)
            {
                _viewModel.Items[_viewModel.Items.Count - 1].DrawBorder = endType;

                // 最後のアイテム以外にはカレントタイムを描画しない設定をします。
                _viewModel.Items.Take(_viewModel.Items.Count - 1).ForEach(vm => vm.DrawBorder = DrawBorderType.None);

                return;
            }

            for (int i = 0; i < _viewModel.Items.Count; i++)
            {
                var item = _viewModel.Items[i];
                var result = comparer.Compare(item.Time, currentTime);

                // アイテムがカレントタイムと同じ場合
                if (result == 0)
                {
                    item.DrawBorder = DrawBorderType.Full;

                    // 以後のアイテムにはカレントタイムを描画しない設定をします。
                    _viewModel.Items.Skip(i + 1).ForEach(vm => vm.DrawBorder = DrawBorderType.None);

                    return;
                }

                // アイテム間にカレントタイムがある場合
                if (i + 1 < _viewModel.Items.Count &&
                    (result < 0 && comparer.Compare(currentTime, _viewModel.Items[i + 1].Time) < 0))
                {
                    item.DrawBorder = endType;

                    // 以後のアイテムにはカレントタイムを描画しない設定をします。
                    _viewModel.Items.Skip(i + 1).ForEach(vm => vm.DrawBorder = DrawBorderType.None);

                    return;
                }

                // 条件外のアイテムにはカレントタイムを描画しない設定をします。
                item.DrawBorder = DrawBorderType.None;
            }
        }

        private void UpdateScrollIntoView()
        {
            // ハイライト状態の最初の要素が表示されるようにスクロールさせます。
            // ビューはソートされているので最初の要素はViewItemsで判断します。
            if (_view != null && _viewModel.ViewItems != null)
            {
                foreach (StartSoundHistoryItemViewModel item in _viewModel.ViewItems)
                {
                    if (item.DrawBorder != DrawBorderType.None)
                    {
                        _view.List.ScrollIntoView(item);
                        _view.List.ScrollIntoView(item); // 暫定対処：２回呼ばないとスクロール位置が反映されない。
                        break;
                    }
                }
            }
        }

        protected override void UpdateSpyModel(string dataName, SpyModel model)
        {
            switch (dataName)
            {
                case AtkSpyModelPlugin.SpyDataName:
                    _viewModel.AttachModel(model as AtkSpyModel);
                    this.UpdateGlobalTimelineCurrent();
                    break;
            }
        }

        private void ApplySettings(ApplicationSettings settings)
        {
            if (settings != null)
            {
                _timeUnit = settings.TimeUnit;
            }

            if (_viewModel != null)
            {
                _viewModel.TimeUnit = _timeUnit;
            }
        }

        protected override void OnSettingsApplied(SettingsService service)
        {
            this.ApplySettings(service.GetApplicationSettings());

            PropertyChangedObservation.RemoveObservers(_settingsObserverOwner);

            PropertyChangedObservation.GetObserver(_settingsObserverOwner, service.GetApplicationSettings())
                .AddHandlerForAnyProperties(
                    (target, args) => ApplySettings(target));

            base.OnSettingsApplied(service);
        }

        private void ExecuteSortedCommand(object parameter)
        {
            this.UpdateDrawBorderTypes(this.GetPlaybackService().Current);
            this.UpdateScrollIntoView();
        }
    }
}
