﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Windows.Controls;
using Nintendo.ToolFoundation.Windows.Primitives;
using NintendoWare.Spy;
using System;
using System.Collections.Generic;
using System.Windows.Media;

namespace NintendoWare.NnAtkSpyPlugin.Windows
{
    /// <summary>
    /// 線状のチャート・アイテムを表わします。
    /// </summary>
    internal class LoudnessMeterChartItemLinearViewModel : ObservableObject, ILoudnessMeterChartItemViewModel, ILineChartItemData
    {
        private double _minimumX = 0.0;
        private double _maximumX = 0.0;
        private bool _visible = true;
        private Color _color = Colors.WhiteSmoke;

        public event EventHandler SamplesChanged;

        public LoudnessMeterPanelViewModel Owner { get; set; }

        public string Name { get; set; } = string.Empty;

        public bool IsVisible
        {
            get { return _visible; }
            set
            {
                if (this.SetPropertyValue(ref _visible, value))
                {
                    this.IsVisibleChanged?.Invoke(this, EventArgs.Empty);
                }
            }
        }

        LineChartDrawType ILineChartItemData.DrawType => LineChartDrawType.Line;

        public Color Color
        {
            get { return _color; }
            set { this.SetPropertyValue(ref _color, value); }
        }

        LineChartInterpolation ILineChartItemData.Interpolation => LineChartInterpolation.Linear;

        public double MaximumX
        {
            get { return _maximumX; }
            set
            {
                if (this.SetPropertyValue(ref _maximumX, value))
                {
                    this.NotifySamplesChanged();
                }
            }
        }

        public double MinimumX
        {
            get { return _minimumX; }
            set
            {
                if (this.SetPropertyValue(ref _minimumX, value))
                {
                    this.NotifySamplesChanged();
                }
            }
        }

        private double CurrentY { get; set; } = double.NaN;

        public double MinimumY => LoudnessMeterPanelViewModel.DisplayRangeMinimumY;

        public double MaximumY => LoudnessMeterPanelViewModel.DisplayRangeMaximumY;

        public double CenterY => LoudnessMeterPanelViewModel.DisplayRangeCenterY;

        public LoudnessMeterPanelViewModel.ValueSelectFunc ValueSelectFunc { get; set; }

        public bool IsFillEnabled { get; set; }

        public IList<LoudnessMeterChartItemFillViewModel> FillChartItems { get; set; }

        public event EventHandler IsVisibleChanged;

        public LoudnessMeterChartItemLinearViewModel()
        {
        }

        public void UpdateCurrent(WaveformSpyModel.LoudnessInfo info)
        {
            this.CurrentY = this.ValueSelectFunc(info);
        }

        public void InvalidateCurrent()
        {
            this.CurrentY = double.NegativeInfinity;
        }

        public void NotifySamplesChanged()
        {
            this.SamplesChanged?.Invoke(this, EventArgs.Empty);
            // LineChart が SampleChanged イベントを使っていないので、その代わり。
            this.NotifyPropertyChanged(nameof(MaximumX));
        }

        IEnumerable<LineChartSample> ILineChartItemData.EnumerateSamples(double beginX, double endX, double scale)
        {
            return this.EnumerateLineChartSamples(beginX, endX, scale);
        }

        double ILineChartItemData.Evaluate(double x)
        {
            return this.CurrentY;
        }

        private IEnumerable<LineChartSample> EnumerateLineChartSamples(
            double start,
            double end,
            double scale)
        {
            var loudnessProvider = this.Owner.LoudnessProvider;
            if (loudnessProvider == null)
            {
                yield break;
            }

            if (!loudnessProvider.IsWaveformMetadataReady)
            {
                yield break;
            }

            if (loudnessProvider.LoudnessCount == 0)
            {
                yield break;
            }

            double secStart = LoudnessMeterPanelViewModel.SecondsPerFrame * start;
            double secEnd = LoudnessMeterPanelViewModel.SecondsPerFrame * end;
            double secScale = LoudnessMeterPanelViewModel.SecondsPerFrame * scale;
            double secBase = loudnessProvider.BeginTime.Timestamp.Seconds;
            double rcpSampleRate = 1.0 / loudnessProvider.SampleRate;
            var loudnessInfos = this.Owner.LoudnessInfos;

            // SampleIndex を秒に変換します。
            Func<long, double> toSeconds =
                (sampleIndex) => secBase + sampleIndex * rcpSampleRate;

            // 無限大を実数にクランプします。
            double minValue = this.CenterY + (this.MinimumY - this.CenterY) * 100;
            double maxValue = this.CenterY + (this.MaximumY - this.CenterY) * 100;

            // LoudnessInfo から値に変換します。
            var valueSelector = this.ValueSelectFunc;
            Func<WaveformSpyModel.LoudnessInfo, double> getValue =
                (info) => Spy.MathUtility.Clamp(valueSelector(info), minValue, maxValue);

            // LoudnessInfo １つあたりのサンプル数が１ピクセルあたりのサンプル数より多ければ
            // LoudnessInfo をそのままプロットし、そうでなければ MinMax 値をプロットします。
            if (loudnessProvider.LoudnessStepSampleCount > secScale * loudnessProvider.SampleRate)
            {
                var index = BinarySearchUtility.BinarySearch(loudnessInfos, (int)((secStart - secBase) * loudnessProvider.SampleRate), it => it.SampleIndex);
                if (index < 0)
                {
                    index = Math.Max(0, ~index - 1);
                }

                for (; index < loudnessInfos.Count; ++index)
                {
                    var info = loudnessInfos[index];
                    var x = LoudnessMeterPanelViewModel.FramesPerSecond * toSeconds(info.SampleIndex);
                    var y = getValue(info);
                    yield return new LineChartSample(new Range1(x, x), new Range1(y, y));

                    if (x >= end + scale)
                    {
                        break;
                    }
                }
            }
            else
            {
                var index = BinarySearchUtility.BinarySearch(loudnessInfos, (int)((secStart - secBase) * loudnessProvider.SampleRate), it => it.SampleIndex);
                if (index < 0)
                {
                    index = Math.Max(0, ~index - 1);
                }

                // start ~ end までを scale の間隔で表示データを作ると LineChart の表示で隙間が入る問題があります。
                // scale の間隔を微妙に減らして、表示データの個数をピクセル数より多く詰め込むようにすると回避できます。
                secScale *= 0.9999999;

                var info = loudnessInfos[index];

                secStart = GetMultipleOf(Math.Max(secStart, toSeconds(info.SampleIndex)), secScale);
                var y = getValue(info);
                var minY = y;
                var maxY = y;
                var rangeY = new Range1(y, y);

                for (var t = secStart; t <= secEnd; t += secScale)
                {
                    if (index >= loudnessInfos.Count)
                    {
                        break;
                    }

                    info = loudnessInfos[index];
                    var secInfo = toSeconds(info.SampleIndex);
                    if (secInfo < t + secScale)
                    {
                        y = getValue(info);
                        minY = Math.Min(minY, y);
                        maxY = Math.Max(maxY, y);
                        index++;

                        while (index < loudnessInfos.Count)
                        {
                            info = loudnessInfos[index];
                            secInfo = toSeconds(info.SampleIndex);
                            if (secInfo >= t + secScale)
                            {
                                break;
                            }

                            y = getValue(info);
                            minY = Math.Min(minY, y);
                            maxY = Math.Max(maxY, y);
                            index++;
                        }

                        rangeY = new Range1(minY, maxY);

                        minY = y;
                        maxY = y;
                    }

                    // rangeX.Length >= 1 にすることで LineChart を MinMax モードで描画させます。
                    // rangeX = new Range1(x, x + 1) だと、計算誤差により rangeX.Length < 1 になってしまう場合があります。
                    var x = LoudnessMeterPanelViewModel.FramesPerSecond * t;
                    var rangeX = new Range1(x, x + 2);
                    yield return new LineChartSample(rangeX, rangeY);
                }
            }
        }

        /// <summary>
        /// baseValue の倍数になるように value を切り捨てます。
        /// </summary>
        /// <param name="value"></param>
        /// <param name="baseValue"></param>
        /// <returns></returns>
        private static double GetMultipleOf(double value, double baseValue)
        {
            return Math.Floor(value / baseValue) * baseValue;
        }
    }
}
