﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Globalization;
using System.Windows;
using System.Windows.Data;

namespace NintendoWare.NnAtkSpyPlugin.Windows
{
    /// <summary>
    /// double を読みやすい文字列に変換します。
    /// </summary>
    [ValueConversion(typeof(double), typeof(string))]
    public sealed class DoubleToLucidStringConverter : IValueConverter
    {
        private NumberFormatInfo _numberFormatInfo;

        public string Format { get; set; } = string.Empty;

        public string NaN
        {
            get { return _numberFormatInfo.NaNSymbol; }
            set { _numberFormatInfo.NaNSymbol = value; }
        }

        public string PositiveInfinity
        {
            get { return _numberFormatInfo.PositiveInfinitySymbol; }
            set { _numberFormatInfo.PositiveInfinitySymbol = value; }
        }

        public string NegativeInfinity
        {
            get { return _numberFormatInfo.NegativeInfinitySymbol; }
            set { _numberFormatInfo.NegativeInfinitySymbol = value; }
        }

        /// <summary>
        /// 切り上げを行う場合の刻み（間隔）を指定または取得します。
        /// <para>
        /// デフォルト値は 0 で、切り上げは行いません。
        /// </para>
        /// </summary>
        public double Ceiling { get; set; }

        public DoubleToLucidStringConverter()
        {
            _numberFormatInfo = (NumberFormatInfo)NumberFormatInfo.CurrentInfo.Clone();
            this.NaN = "NaN";
            this.PositiveInfinity = "∞";
            this.NegativeInfinity = "-∞";
        }

        public object Convert(object value, Type targetType, object parameter, CultureInfo culture)
        {
            if (value is double)
            {
                if (this.Ceiling != 0)
                {
                    value = Math.Ceiling((double)value / this.Ceiling) * this.Ceiling;
                }

                return string.Format(_numberFormatInfo, this.Format, value);
            }

            return DependencyProperty.UnsetValue;
        }

        public object ConvertBack(object value, Type targetType, object parameter, CultureInfo culture)
        {
            throw new NotImplementedException();
        }
    }
}
