﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;

namespace NintendoWare.NnAtkSpyPlugin.Windows
{
    public class AtkPerformancePanelViewModel : ObservableObject
    {
        private const double MaxScaleX = 20; // これより上がると横スクロールできなくなる。

        private readonly object _plotObserverOwner = new object();
        private readonly object _markerObserverOwner = new object();

        private readonly SpyService _spyService;
        private readonly SpyPlaybackService _playbackService;

        private PlotSpyModel _plotModel;
        private MarkerSpyModel _markerModel;
        private FrameSyncSpyModel _frameSyncModel;

        private ObservableKeyedList<string, AtkPerformanceStateViewModel> _plotNodes = new ObservableKeyedList<string, AtkPerformanceStateViewModel>(node => node.Name);
        private ObservableList<MarkerViewModel> _markers = new ObservableList<MarkerViewModel>();

        private SpyTimeUnit _timeUnit = SpyTimeUnit.TimestampUsec;
        private double _currentFrame;
        private double _minimumFrame = double.NaN;
        private double _maximumFrame = double.NaN;
        private double _scaleX = 1d / 64;
        private double _viewportLeft;
        private string _error = null;

        public AtkPerformancePanelViewModel(SpyService spyService, SpyPlaybackService playbackService)
        {
            Ensure.Argument.NotNull(spyService);
            Ensure.Argument.NotNull(playbackService);

            _spyService = spyService;
            _playbackService = playbackService;
        }

        protected override void DisposeManagedInstance()
        {
            CollectionChangedObservation.RemoveObservers(_plotObserverOwner);
            PropertyChangedObservation.RemoveObservers(_plotObserverOwner);
            CollectionChangedObservation.RemoveObservers(_markerObserverOwner);
            PropertyChangedObservation.RemoveObservers(_markerObserverOwner);

            _plotNodes.ForEach(it => it.Dispose());
            _markers.ForEach(it => it.Dispose());

            base.DisposeManagedInstance();
        }

        public FrameSyncSpyModel FrameSyncModel
        {
            get { return _frameSyncModel; }
            set { this.SetPropertyValue(ref _frameSyncModel, value); }
        }

        public ObservableCollection<AtkPerformanceStateViewModel> Nodes
        {
            get { return _plotNodes; }
        }

        public ObservableCollection<MarkerViewModel> Markers
        {
            get { return _markers; }
        }

        public SpyTimeUnit TimeUnit
        {
            get { return _timeUnit; }
            set { this.SetPropertyValue(ref _timeUnit, value); }
        }

        public double CurrentFrame
        {
            get { return _currentFrame; }
            set { this.SetCurrentFrameImpl(value, from: this); }
        }

        public double ScaleX
        {
            get { return _scaleX; }
            set
            {
                value = Math.Min(value, MaxScaleX);
                this.SetPropertyValue(ref _scaleX, value);
            }
        }

        public double MinimumFrame
        {
            get { return _minimumFrame; }
            set { this.SetPropertyValue(ref _minimumFrame, value); }
        }

        public double MaximumFrame
        {
            get { return _maximumFrame; }
            set { this.SetPropertyValue(ref _maximumFrame, value); }
        }

        public double ViewportLeft
        {
            get { return _viewportLeft; }
            set { this.SetPropertyValue(ref _viewportLeft, value); }
        }

        public string Error
        {
            get { return _error; }
            set { this.SetPropertyValue(ref _error, value); }
        }

        public void SetPlotModel(PlotSpyModel model)
        {
            if (_plotModel == model)
            {
                return;
            }

            CollectionChangedObservation.RemoveObservers(_plotObserverOwner);
            PropertyChangedObservation.RemoveObservers(_plotObserverOwner);

            _plotModel = model;
            this.SyncPlotItems();

            this.ResetErrorIfNecessary();
        }

        public void SetMarkerModel(MarkerSpyModel model)
        {
            if (_markerModel == model)
            {
                return;
            }

            CollectionChangedObservation.RemoveObservers(_markerObserverOwner);
            PropertyChangedObservation.RemoveObservers(_markerObserverOwner);

            _markerModel = model;
            this.SyncMarkers();

            this.ResetErrorIfNecessary();
        }

        public void SetFrameSyncModel(FrameSyncSpyModel model)
        {
            this.FrameSyncModel = model;
        }

        public void SetCurrentFrameFromPresenter(double value)
        {
            this.SetCurrentFrameImpl(value, from: null);
        }

        private void SetCurrentFrameImpl(double value, object from)
        {
            if (from == this)
            {
                value = MathUtility.Clamp(
                    Math.Floor(value), this.MinimumFrame, this.MaximumFrame);
            }

            if (this.SetPropertyValue(ref _currentFrame, value, nameof(CurrentFrame)))
            {
                foreach (var node in _plotNodes)
                {
                    node.CurrentFrame = value;
                }

                if (from == this)
                {
                    if (!double.IsNaN(value))
                    {
                        _playbackService?.SetCurrentTime(SpyGlobalTime.FromMicroSeconds((long)value));
                    }
                }
            }
        }

        private void SyncPlotItems()
        {
            _plotNodes.ForEach(it => it.Dispose());
            _plotNodes.Clear();

            if (_plotModel == null)
            {
                return;
            }

            CollectionChangedObservation.GetObserver(_plotObserverOwner, _plotModel.States).AddHandlerForAddItems(
                (sender, e) =>
                {
                    this.NewPlotStates(e.Items.Cast<PlotSpyModel.PlotState>());
                });

            this.NewPlotStates(_plotModel.States);

            PropertyChangedObservation.GetObserver(_plotObserverOwner, _plotModel).AddHandler(
                target => target.ErrorUnexpectedDataVersion,
                (sender, e) =>
                {
                    this.CheckPlotModelErrorUnexpectedDataVersion();
                });

            this.CheckPlotModelError();
        }

        private void SyncMarkers()
        {
            _markers.ForEach(it => it.Dispose());
            _markers.Clear();

            if (_markerModel == null)
            {
                return;
            }

            CollectionChangedObservation.GetObserver(_markerObserverOwner, _markerModel.Markers).AddHandlerForAddItems(
                (sender, e) =>
                {
                    this.NewMarkers(
                        e.Items.Cast<KeyValuePair<uint, MarkerSpyModel.Marker>>().Select(item => item.Value));
                });

            this.NewMarkers(_markerModel.Markers.Values);

            PropertyChangedObservation.GetObserver(_markerObserverOwner, _markerModel).AddHandler(
                target => target.ErrorUnexpectedDataVersion,
                (sender, e) =>
                {
                    this.CheckMarkerModelErrorUnexpectedDataVersion();
                });

            this.CheckMarkerModelError();
        }

        private void NewMarkers(IEnumerable<MarkerSpyModel.Marker> markers)
        {
            foreach (var marker in markers)
            {
                // MarkerSpyModel.Marker.Values に要素が追加されたら、_markers に追加
                CollectionChangedObservation.GetObserver(_markerObserverOwner, marker.Values).AddHandlerForAddItems(
                    (sender, e) =>
                    {
                        e.Items.Cast<MarkerSpyModel.MarkerValue>()
                            .ForEach(value => this.NewMarker(value));
                    });

                // 新規 Marker を _markers に追加
                marker.Values.ForEach(value => this.NewMarker(value));
            }
        }

        private void NewPlotStates(IEnumerable<PlotSpyModel.PlotState> states)
        {
            states.ForEach(it => this.NewPlotState(it));
        }

        private void NewPlotState(PlotSpyModel.PlotState state)
        {
            if (state.Name.StartsWith("@nn/atk/"))
            {
                _plotNodes.Add(
                    new AtkPerformanceStateViewModel(state, state.Name.Substring("@nn/atk/".Length)));
            }
        }

        private void NewMarker(MarkerSpyModel.MarkerValue value)
        {
            _markers.Add(new MarkerViewModel(value));
        }

        private void AddErrorMessage(string errorMessage)
        {
            if (this.Error == null)
            {
                this.Error = errorMessage;
            }
            else
            {
                this.Error = errorMessage + "\n" + this.Error;
            }
        }

        private void CheckPlotModelErrorUnexpectedDataVersion()
        {
            if (_plotModel != null && _plotModel.ErrorUnexpectedDataVersion)
            {
                // TODO : 実装する
                // this.AddErrorMessage(Resources.Messages.UnexpectedPlotDataVersion);
            }
        }

        private void CheckPlotModelError()
        {
            this.CheckPlotModelErrorUnexpectedDataVersion();
        }

        private void CheckMarkerModelErrorUnexpectedDataVersion()
        {
            if (_markerModel != null && _markerModel.ErrorUnexpectedDataVersion)
            {
                // TODO : 実装する
                // this.AddErrorMessage(Resources.Messages.UnexpectedMarkerDataVersion);
            }
        }

        private void CheckMarkerModelError()
        {
            this.CheckMarkerModelErrorUnexpectedDataVersion();
        }

        private void ResetErrorIfNecessary()
        {
            if (_plotModel == null && _markerModel == null)
            {
                this.Error = null;
            }
        }
    }
}
