﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;
using System.Collections.Specialized;
using System.Reflection;

namespace Nintendo.AudioToolkit.Views
{
    using Nintendo.Foundation.ComponentModel;
    using Nintendo.Foundation.Windows.Input;
    using Nintendo.AudioToolkit.DataModels;
    using Nintendo.AudioToolkit.Presenters;
    using Nintendo.AudioToolkit.Windows.Controls;
    using Nintendo.AudioToolkit.Extensions;

    public class WaveSoundClipResource : ObservableObject
    {
        private readonly WaveSoundClipData data = null;

        public WaveSoundClipResource(WaveSoundClipData data)
        {
            this.data = data;
        }

        public WaveSoundClipData Data
        {
            get { return this.data; }
        }

        public bool IsEnabled
        {
            get { return this.data.IsEnabled; }
            set
            {
                if (this.data.IsEnabled != value)
                {
                    this.OperationExecutor.SetParameter<bool>(this.data.IsEnabled, value,
                        (v) =>
                        {
                            this.data.IsEnabled = v;
                            this.NotifyPropertyChanged<bool>(() => IsEnabled);
                        });
                }
            }
        }

        public int Position
        {
            get { return this.data.Position; }
            set
            {
                if (this.data.Position != value)
                {
                    this.OperationExecutor.SetParameter<int>(this.data.Position, value,
                        (v) =>
                        {
                            this.data.Position = v;
                            this.NotifyPropertyChanged<int>(() => Position);
                        });
                }
            }
        }

        public int StartOffset
        {
            get { return this.data.StartOffset; }
            set
            {
                if (this.data.StartOffset != value)
                {
                    this.OperationExecutor.SetParameter<int>(this.data.StartOffset, value,
                        (v) =>
                        {
                            this.data.StartOffset = v;
                            this.NotifyPropertyChanged<int>(() => StartOffset);
                        });
                }
            }
        }

        public int Duration
        {
            get { return this.data.Duration; }
            set
            {
                if (this.data.Duration != value)
                {
                    this.OperationExecutor.SetParameter<int>(this.data.Duration, value,
                        (v) =>
                        {
                            this.data.Duration = v;
                            this.NotifyPropertyChanged<int>(() => Duration);
                        });
                }
            }
        }

        public float Volume
        {
            get { return this.data.Volume; }
            set
            {
                if (this.data.Volume != value)
                {
                    this.OperationExecutor.SetParameter<float>(this.data.Volume, value,
                        (v) =>
                        {
                            this.data.Volume = v;
                            this.NotifyPropertyChanged<float>(() => Volume);
                        });
                }
            }
        }

        public float Pitch
        {
            get { return this.data.Pitch; }
            set
            {
                if (this.data.Pitch != value)
                {
                    this.OperationExecutor.SetParameter<float>(this.data.Pitch, value,
                        (v) =>
                        {
                            this.data.Pitch = v;
                            this.NotifyPropertyChanged<float>(() => Pitch);
                        });
                }
            }
        }

        public float Pan
        {
            get { return this.data.Pan; }
            set
            {
                if (this.data.Pan != value)
                {
                    this.OperationExecutor.SetParameter<float>(this.data.Pan, value,
                        (v) =>
                        {
                            this.data.Pan = v;
                            this.NotifyPropertyChanged<float>(() => Pan);
                        });
                }
            }
        }

        public string Path
        {
            get { return this.data.Path; }
            set
            {
                if (this.data.Path != value)
                {
                    this.OperationExecutor.SetParameter<string>(this.data.Path, value,
                        (v) =>
                        {
                            this.data.Path = v;
                            this.NotifyPropertyChanged<string>(() => Path);
                        });
                }
            }
        }

        public void SetStartOffsetWithPositionAndDuration(int value)
        {
            if (this.StartOffset != value)
            {
                try
                {
                    this.OperationExecutor.BeginTransaction();

                    var diff = value - this.StartOffset;
                    this.StartOffset = value;
                    this.Position += diff;
                    this.Duration -= diff;
                }
                catch
                {
                    this.OperationExecutor.CancelTransaction();
                }
                finally
                {
                    this.OperationExecutor.EndTransaction();
                }
            }
        }

        internal WaveSoundTrackResource Owner
        {
            get;
            set;
        }

        internal IOperationExecutor OperationExecutor
        {
            get;
            set;
        }
    }

    public class WaveSoundTrackResource : ObservableObject
    {
        private readonly WaveSoundTrackData data = null;
        private readonly ObservableCollection<WaveSoundClipResource> clips = new ObservableCollection<WaveSoundClipResource>();
        private IOperationExecutor operationExecutor = null;

        public WaveSoundTrackResource(WaveSoundTrackData data)
        {
            this.data = data;
            this.BuildDataModel(this.data);

            this.clips.CollectionChanged += OnCollectionChangedClips;
        }

        public IList<WaveSoundClipResource> Clips
        {
            get { return this.clips; }
        }

        public bool IsEnabled
        {
            get { return this.data.IsEnabled; }
            set
            {
                this.data.IsEnabled = value;
                this.NotifyPropertyChanged();
            }
        }

        internal WaveSoundTrackData Data
        {
            get { return this.data; }
        }

        private void BuildDataModel(WaveSoundTrackData track)
        {
            foreach (var clip in track.Clips)
            {
                var clipDM = new WaveSoundClipResource(clip);
                clipDM.Owner = this;
                this.Clips.Add(clipDM);
            }
        }

        public void AddClip(WaveSoundClipResource clipDM)
        {
            if (this.Contains(clipDM) == true)
            {
                return;
            }

            this.OperationExecutor.AddItem(this.Clips, clipDM);
        }

        public void RemoveClip(WaveSoundClipResource clipDM)
        {

            if (this.Contains(clipDM) == false)
            {
                return;
            }

            this.OperationExecutor.RemoveItem(this.Clips, clipDM);
        }

        private bool Contains(WaveSoundClipResource clipDM)
        {
            return this.Clips.Where(c => c == clipDM).Count() > 0 ? true : false;
        }

        private void OnCollectionChangedClips(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    foreach (var newItem in e.NewItems)
                    {
                        var clipDM = newItem as WaveSoundClipResource;
                        clipDM.Owner = this;
                        clipDM.OperationExecutor = this.OperationExecutor;

                        this.data.Clips.Add(clipDM.Data);
                    }
                    break;

                case NotifyCollectionChangedAction.Remove:
                    foreach (var oldItem in e.OldItems)
                    {
                        var clipDM = oldItem as WaveSoundClipResource;
                        clipDM.Owner = null;
                        clipDM.OperationExecutor = null;

                        this.data.Clips.Remove(clipDM.Data);
                    }
                    break;
            }
        }

        internal WaveSoundResource Owner
        {
            get;
            set;
        }

        internal IOperationExecutor OperationExecutor
        {
            get { return this.operationExecutor; }
            set
            {
                this.operationExecutor = value;
                this.Clips.ToList().ForEach(c => c.OperationExecutor = value);
            }
        }
    }

    public class WaveSoundResource : ObservableObject
    {
        private readonly WaveSoundData data = null;
        private readonly ObservableCollection<WaveSoundTrackResource> tracks = new ObservableCollection<WaveSoundTrackResource>();
        private IOperationExecutor operationExecutor = null;

        public WaveSoundResource(WaveSoundData data, IOperationExecutor executor)
        {
            this.data = data;
            this.BuildDataModel(this.data);

            this.OperationExecutor = executor;

            this.tracks.CollectionChanged += OnCollectionChangedTracks;
        }

        public IList<WaveSoundTrackResource> Tracks
        {
            get { return this.tracks; }
        }

        public void Add(WaveSoundTrackResource trackDM, WaveSoundClipResource clipDM)
        {
            this.OperationExecutor.AddItem(trackDM.Clips, clipDM);
        }

        public void Remove(IEnumerable<WaveSoundClipResource> clips)
        {
            foreach (var clip in clips.ToList())
            {
                this.OperationExecutor.RemoveItem(clip.Owner.Clips, clip);
            }
        }

        public void Add(WaveSoundTrackResource trackDM)
        {
            this.OperationExecutor.AddItem(this.Tracks, trackDM);
        }

        public void Remove(IEnumerable<WaveSoundTrackResource> tracks)
        {
            foreach (var track in tracks.ToList())
            {
                this.OperationExecutor.RemoveItem(track.Owner.Tracks, track);
            }
        }

        private void BuildDataModel(WaveSoundData waveSound)
        {
            foreach (var track in waveSound.Tracks)
            {
                var trackDM = new WaveSoundTrackResource(track);
                trackDM.Owner = this;
                this.Tracks.Add(trackDM);
            }
        }

        private void OnCollectionChangedTracks(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    foreach (var newItem in e.NewItems)
                    {
                        var trackDM = newItem as WaveSoundTrackResource;
                        trackDM.Owner = this;
                        trackDM.OperationExecutor = this.OperationExecutor;

                        this.data.Tracks.Add(trackDM.Data);
                    }
                    break;

                case NotifyCollectionChangedAction.Remove:
                    foreach (var oldItem in e.OldItems)
                    {
                        var trackDM = oldItem as WaveSoundTrackResource;
                        trackDM.Owner = null;
                        trackDM.OperationExecutor = null;

                        this.data.Tracks.Remove(trackDM.Data);
                    }
                    break;
            }
        }

        internal IOperationExecutor OperationExecutor
        {
            get { return this.operationExecutor; }
            set
            {
                this.operationExecutor = value;
                this.Tracks.ToList().ForEach(c => c.OperationExecutor = value);
            }
        }
    }

    internal static class IOperationExecutorExtensions
    {
        public static void SetParameter<T>(this IOperationExecutor executor, T oldValue, T newValue, Action<T> action)
        {
            executor.ExecuteOperation(
                delegate()
                {
                    action(newValue);
                },
                delegate()
                {
                    action(oldValue);
                });
        }

        public static void AddItem<TItem>(this IOperationExecutor executor, IList<TItem> collection, TItem item) where TItem : class
        {
            executor.ExecuteOperation(
                delegate()
                {
                     collection.Add(item);
                },
                delegate()
                {
                    collection.Remove(item);
                });
        }

        public static void RemoveItem<TItem>(this IOperationExecutor executor, IList<TItem> collection, TItem item) where TItem : class
        {
            var index = collection.IndexOf(item);

            executor.ExecuteOperation(
                delegate()
                {
                    collection.RemoveAt(index);
                },
                delegate()
                {
                    collection.Insert(index, item);
                });
        }
    }

    public interface IOperationExecutor
    {
        void ExecuteOperation(Action executeAction, Action rollbackAction);
        void BeginTransaction();
        void EndTransaction();
        void CancelTransaction();
    }

}
